(ns test.tree.sauce
  (:require [sauce-api.jobs :as job]
            [clj-webdriver.taxi :as browser]
            [clj-webdriver.remote.server :as rs]))

(defn wrap-sauce
  "Run the tests in saucelabs. sauce-user and sauce-key are your
  credentials.  sauce-server is a remote webdriver instance that
  points to saucelabs (already including your credentials in the URL).
  'capabilities-chooser-fn' is a function that takes the test as an
  argument and returns browser capabilities associated with that test.
  'finder-fn' is a finder function to bind to taxi's *finder-fn*.  If
  test has key :headless true, saucelabs won't be used at all."
  [runner {:keys [sauce-user sauce-key sauce-server capabilities-chooser-fn finder-fn]}]
  (fn [{:keys [test] :as req}]
    (if (:headless test)
      (runner req) 
      (binding [browser/*driver* (rs/new-remote-driver sauce-server
                                                       {:capabilities (capabilities-chooser-fn test)})
                browser/*finder-fn* finder-fn]
        (let [job-id (second (re-find #"\(([^\)]+)\)" (str (:webdriver browser/*driver*))))]
          ;; give the job a name before starting
          (job/update-id sauce-user sauce-key job-id {:name (:name test)})

          (let [res (assoc (runner req) :sauce (job/get-id sauce-user sauce-key job-id))
                passed? (-> res :result (= :pass))]
            (try (browser/quit)
                 (job/update-id sauce-user sauce-key job-id
                                {:passed passed?
                                 :custom-data (if-not passed?
                                                {"throwable" (try (pr-str (:error res)) (catch Exception _ "could not print"))
                                                 "stacktrace" (-> res :error :stack-trace java.util.Arrays/toString)}
                                                {})})
                 ;; return result
                 res
                 (catch Exception e
                   (assoc res :sauce-error e)))))))))
