(ns the.trampoline
  (:require [clojure.string :as str]))

(defrecord InputState [input pos])
(defrecord SourcePos [line column])

(defrecord Ok [item])
(defrecord Err [errmsg])

(defprotocol ShowableError
  (show-error [this]))

(defrecord ParseError [pos msgs]
  ShowableError
  (show-error [_] (str (str/join ", " msgs)
                       " at"
                       " line: " (:line pos)
                       " column: " (:column pos))))

(defmacro bounce [f & args]
  `(fn [] (~f ~@args)))

(defn always [x]
  (fn [state cok cerr eok eerr]
    (eok x state)))

(defn bind [p f]
  (fn [state cok cerr eok eerr]
    (letfn [(pcok [item state]
              (bounce (f item) state cok cerr cok cerr))
            (peok [item state]
              (bounce (f item) state cok cerr eok eerr))]
      (p state pcok cerr peok eerr))))

(defn nxt [p q]
  (bind p (fn [_] q)))

(defn run-parser [p state]
  (trampoline p state
              (fn cok [item _]
                (Ok. item))
              (fn cerr [err]
                (Err. (show-error err)))
              (fn eok [item _]
                (Ok. item))
              (fn eerr [err]
                (Err. (show-error err)))))

(defn run [p input]
  (let [result (run-parser p (InputState. input (SourcePos. 1 1)))]
    (condp = (class result)
      Ok (:item result)
      Err (throw (RuntimeException. (:errmsg result))))))