(ns leiningen.test-loop
  (:use clojure.pprint)
  (:import (java.io File)
           (java.util Date Stack)
           (java.io PrintStream
                    ByteArrayOutputStream
                    InputStreamReader
                    BufferedReader
                    DataOutputStream))
  (:require [clojure.test]
            [clojure.java.io :as io]
            [leiningen.classpath :as lcp]
            [leiningen.core.eval :as eval]
            [clojure.tools.namespace.dir :as ns-dir]
            [clojure.set :as cset]
            [clojure.repl :as repl]
            [clojure.string :as string]))


(def runtime (Runtime/getRuntime))

(def state (atom {}))
(def processes (Stack.))

(defn modified-files []
  (let [tracker @state
        lastmod (tracker :clojure.tools.namespace.dir/time 0) 
        files (tracker :clojure.tools.namespace.dir/files)
        modded (filter #(< lastmod (.lastModified %)) files)] 

    (when (not (empty? modded))
      (swap! state assoc :clojure.tools.namespace.dir/time (System/currentTimeMillis)))

    modded
    ))

(defn get-dependents-of-ns [ns-sym]
  (get-in @state [:clojure.tools.namespace.track/deps :dependents ns-sym]))

(defn get-ns-of-file [file]
  (get-in @state [:clojure.tools.namespace.file/filemap file]))

(defn is-test-ns? [mod-ns]
  (re-find #"-test$" (name mod-ns)))

(defn find-tests-using [mod-ns]
  (let [dependents (get-dependents-of-ns mod-ns)]
    (filter is-test-ns? dependents)))

(defn expand-tests-to-run [file]
  (let [mod-ns (get-ns-of-file file)]
    (if (is-test-ns? mod-ns)
      (list mod-ns)
      (find-tests-using mod-ns)
      )))

(defn execute-in-child [tests-to-run]
  (let [proc (.pop processes)
        pout (PrintStream. (.getOutputStream proc))
        pin (.getInputStream proc)
        sb (ByteArrayOutputStream.)]

    (doseq [ns-sym tests-to-run]
      (.println pout (str ns-sym)))

    (.println pout "one speed, one gear: go!")
    (.flush pout)

    (.waitFor proc)
    ))

(defn run-applicable-tests [modified-files]
  (let [tests-to-run (mapcat expand-tests-to-run modified-files)]
    (if (empty? tests-to-run)
      (do
        (println "================================================================================")
        (println "WARNING: There are no tests for:" modified-files)
        (println "================================================================================"))
      (do
        (println "================================================================================")
        (execute-in-child tests-to-run)
        (println "================================================================================"))
    )))

(defn external-namespaces [state]
  (let [dependents
        (into #{} (keys (get-in state [:clojure.tools.namespace.track/deps :dependents])))
        project-namespaces
        (into #{} (vals (get state :clojure.tools.namespace.file/filemap)))]
    (cset/difference dependents project-namespaces)))

(defn scan-files [state project]
  (let [dirs (concat
               (project :source-paths [])
               (project :test-paths []))]
    (apply ns-dir/scan (assoc state ::dirs dirs) dirs)))

(defn build-jvm-cmd [state project]
  (assoc state
         ::jvm-cmd
         (into-array (eval/shell-command project (state ::client-template)))))

(def client-body
  '((let [namespaces (loop [ns-list []]
                       (let [line (read-line)]
                         (if (= "one speed, one gear: go!" line)
                           ns-list
                           (recur (conj ns-list (symbol line))))))
          start (System/currentTimeMillis)]

        (doseq [ns-sym namespaces]
          (require ns-sym))

        (apply clojure.test/run-tests namespaces)
        (println (str "Runtime: " (- (System/currentTimeMillis) start) "ms"))
        (.flush System/out)
      )))

(defn build-client-template [state]
  (let [externals (external-namespaces state)
        tmpl (concat
               (list 'do)
               (map (fn [sym] (list
                                'require
                                (list 'quote sym))) externals)
               client-body)]
    (assoc state ::client-template tmpl)))

(defn initialize-state [project]
  (-> {}
    (scan-files project)
    (build-client-template)
    (build-jvm-cmd project)))

(defn read-until-dead [prefix proc reader]
  (loop [reader reader]
    (when-let [line (.readLine reader)]
      (println (str prefix line))
      (recur reader))))

(defn launch-process []
  (let [cmd (get @state ::jvm-cmd)
        proc (.exec runtime cmd)]
    (-> (Thread. (fn [] (read-until-dead "" proc (io/reader (.getInputStream proc)))))
      (.start))
    (-> (Thread. (fn [] (read-until-dead "ERR: " proc (io/reader (.getErrorStream proc)))))
      (.start))
    (.push processes proc)))

(defn test-loop [project & args]
  (eval/prep project)

  (reset! state (initialize-state project))

  (println "Entering Test Loop ... (CONTROL-C to interrupt)")

  (try
    (while true
      (when (< (.size processes) 1)
        (launch-process))
      (let [modded (modified-files)]
        (if (empty? modded)
          (Thread/sleep 500)
          (run-applicable-tests modded)))
      )
    (catch Throwable t (repl/pst t)))

  true)
