(ns shadow.build.targets.external-index
  (:require
    [clojure.java.io :as io]
    [clojure.set :as set]
    [clojure.string :as str]
    [shadow.build.data :as data]
    [shadow.build.js-support :as js-support]
    [shadow.cljs.util :as util])
  (:import [java.io BufferedOutputStream FileOutputStream OutputStreamWriter]))

(defn generate-cjs [state require]
  (update state :body conj (str "ALL[\"" require "\"] = require(\"" require "\");")))

(defn join-lines [all]
  (str/join "\n" all))

(defn wrap-boilerplate [{:keys [imports body]}]
  (-> ["// WARNING: DO NOT EDIT!"
       "// THIS FILE WAS GENERATED BY SHADOW-CLJS AND WILL BE OVERWRITTEN!"
       ""]
      (into imports)
      (conj "var ALL = {};")
      (into body)
      (conj (slurp (io/resource "shadow/build/targets/external_boilerplate.js")))
      (conj "")
      (join-lines)))

(defn generate [build-state js-requires]

  ;; FIXME: need to figure out how to do proper compat first
  ;; (:require ["react" :as X]) will be invalid since it is commonjs
  ;; and future webpack would only allow
  ;;   import X from "react";
  ;; which maps to
  ;;   (:require ["react" :default X])
  ;; but that would break everything using :as ... which is everything ...
  ;; (reduce-kv generate-esm {:imports [] :body []})

  (-> (reduce generate-cjs {:build-state build-state :imports [] :body []} js-requires)
      (wrap-boilerplate)))

(comment
  (generate {} #{"react" "something"}))

(defn flush-esm [{:keys [build-sources] :as state}]
  (let [js-shim-namespaces
        (->> build-sources
             (map #(data/get-source-by-id state %))
             (filter ::js-support/require-shim)
             (reduce
               (fn [acc {:keys [js-require ns]}]
                 (assoc acc (name ns) js-require))
               {}))

        used-vars
        (if (= :dev (:shadow.build/mode state))
          ;; dev just import * everything
          (reduce-kv (fn [acc shim-ns _] (assoc acc shim-ns ::STAR)) {} js-shim-namespaces)
          ;; release mode only imports actual vars
          ;; need to look for matching namespaces, longest first
          ;; otherwise it'll put react-dom into the react basket
          (let [match-prefixes
                (->> (keys js-shim-namespaces)
                     (sort-by count)
                     (reverse))]

            (->> build-sources
                 (mapcat #(get-in state [:output % :used-vars]))
                 (filter #(= "js" (namespace %)))
                 (map name)
                 (reduce
                   (fn [acc js-name]
                     ;;   (:require ["pkg" :as x]) and x used directly
                     ;; meaning we need
                     ;;   import * as x from "pkg";
                     ;; no need to destructure props then

                     ;; (:require ["pkg" :as x]) using x somewhere
                     (if (contains? js-shim-namespaces js-name)
                       (assoc acc js-name ::STAR)

                       ;; record individual property access
                       ;; (:require ["pkg" :as x]) using x/foo
                       (let [match-ns (first (filter #(str/starts-with? js-name %) match-prefixes))]
                         (cond
                           (not match-ns) ;; only want to record shim access
                           acc

                           ;; don't override * if we encounter another prop
                           (= ::STAR (get acc match-ns))
                           acc

                           :else
                           (let [prop (subs js-name (-> match-ns count inc))]
                             (update acc match-ns util/set-conj prop))))))
                   {}))))

        aliases
        (reduce-kv
          (fn [acc shim-ns _]
            (assoc acc shim-ns (str "i" (count acc))))
          {}
          used-vars)

        output-to
        (io/file (get-in state [:js-options :external-index] "target/external.js"))]

    (io/make-parents output-to)

    (binding [*out* (-> output-to
                        (FileOutputStream.)
                        (BufferedOutputStream.)
                        (OutputStreamWriter.))]

      ;; first emits all imports
      (doseq [[shim-ns js-require] js-shim-namespaces
              :when (contains? used-vars shim-ns)]
        (println (str "import * as " (get aliases shim-ns) " from \"" js-require "\";")))

      ;; then boilerplate
      (println)
      (println "const ALL = {};")
      (println)
      (println "globalThis.shadow$bridge = function(name) {")
      (println "  const ret = ALL[name];")
      (println "  if (ret == undefined) {")
      (println "    throw new Error(\"Dependency: \" + name + \" not provided by external JS!\");")
      (println "  } else {")
      (println "    return ret;")
      (println "  }")
      (println "};")

      ;; then do reassignments of all used exports
      (doseq [[shim-ns used] used-vars]
        (println)
        (let [js-require (get js-shim-namespaces shim-ns)
              alias (get aliases shim-ns)]
          (if (= ::STAR used)
            (println (str "ALL[\"" js-require "\"] = " alias ";"))
            (do (println (str "ALL[\"" js-require "\"] = {"))
                (println
                  (->> used
                       (map (fn [prop]
                              (str "  " prop ": " alias "." prop)))
                       (str/join ",\n")))
                (println (str "};"))
                ))))))

  state)

(defn flush-common-js [{:keys [build-sources] :as state}]
  (let [js-requires
        (->> build-sources
             (map #(data/get-source-by-id state %))
             (filter ::js-support/require-shim)
             (map :js-require)
             (into #{}))]

    (if (= js-requires (::ext-info state))
      state
      (let [output-to (io/file (get-in state [:js-options :external-index] "target/external.js"))]
        (io/make-parents output-to)
        (spit output-to (generate state js-requires))
        (assoc state ::ext-info js-requires)))))

(defn flush-js [state]
  (if (= :esm (get-in state [:js-options :external-index-format]))
    (flush-esm state)
    (flush-common-js state)))
