(ns thi.ng.color.core
  #?(:cljs
  (:require-macros
   [thi.ng.math.macros :as mm]))
  (:require
   [thi.ng.math.core :as m :refer [PI TWO_PI]]
   [thi.ng.strf.core :as f]
   #?(:clj [thi.ng.math.macros :as mm])))

(def INV8BIT (/ 1.0 255))

(defn with-alpha
  [col a] (if a (conj col a) col))

(defn rgb->hsv
  ([rgb]
   (apply rgb->hsv rgb))
  ([r g b & [a]]
   (let [v (max r g b)
         d (- v (min r g b))
         s (if (zero? v) 0.0 (/ d v))
         h (if (zero? s)
             0.0
             (condp == v
               r (mm/subdiv g b d)
               g (+ 2.0 (mm/subdiv b r d))
               (+ 4.0 (mm/subdiv r g d))))
         h (/ h 6.0)]
     (with-alpha
       [(if (neg? h) (inc h) h) s v] a))))

(defn hsv->rgb
  ([hsv]
   (apply hsv->rgb hsv))
  ([h s v & [a]]
   (if (m/delta= 0.0 s)
     [v v v]
     (let [h (rem (* h 6.0) 6.0)
           i (int h)
           f (- h i)
           p (* v (- 1.0 s))
           q (* v (- 1.0 (* s f)))
           t (* v (- 1.0 (mm/subm 1.0 f s)))]
       (with-alpha
         (case i
           0 [v t p]
           1 [q v p]
           2 [p v t]
           3 [p q v]
           4 [t p v]
           [v p q])
         a)))))

(defn cmyk->rgb
  ([cmyk]
   (apply cmyk->rgb cmyk))
  ([c m y k & [a]]
   (with-alpha
     [(- 1.0 (min 1.0 (+ c k)))
      (- 1.0 (min 1.0 (+ m k)))
      (- 1.0 (min 1.0 (+ y k)))]
     a)))

(defn rgb->cmyk
  ([rgb]
   (apply rgb->cmyk rgb))
  ([r g b & [a]]
   (let [c (- 1.0 r)
         m (- 1.0 g)
         y (- 1.0 b)
         k (min (min c m) y)]
     (with-alpha
       [(max (- c k) 0.0)
        (max (- m k) 0.0)
        (max (- y k) 0.0)
        (max k 0.0)]
       a))))

(defn- cie1931-gamma-correct
  [x]
  (if (< x 0.0031308)
    (* 12.92 x)
    (mm/msub 1.055 (Math/pow x (/ 2.4)) 0.055)))

(defn cie1931->rgb
  ([xyz]
   (apply cie1931->rgb xyz))
  ([x y z & [a]]
   (let [r (mm/madd 3.2406 x -1.5372 y -0.4986 z)
         g (mm/madd -0.9689 x 1.8758 y 0.0415 z)
         b (mm/madd 0.0557 x -0.2040 y 1.0570 z)]
     (with-alpha (mapv cie1931-gamma-correct [r g b]) a))))

(defn rgba->int
  ([rgba]
   (apply rgba->int rgba))
  ([r g b & [a]]
   (let [rgb (bit-or
              (bit-or
               (-> r (* 255) int (bit-shift-left 16))
               (-> g (* 255) int (bit-shift-left 8)))
              (-> b (* 255) int))]
     (if a
       (bit-or rgb (-> a (* 255) int (bit-shift-left 24)))
       rgb))))

(defn int->rgba
  [int32]
  [(* INV8BIT (bit-and (bit-shift-right int32 16) 255))
   (* INV8BIT (bit-and (bit-shift-right int32 8) 255))
   (* INV8BIT (bit-and int32 255))
   (* INV8BIT (bit-and (unsigned-bit-shift-right int32 24) 255))])

(def ^:private hex6 ["#" (f/hex 6)])

(defn int->hex
  [i] (f/format hex6 (bit-and i 0xffffff)))

(defn rgba->css
  ([rgba] (apply rgba->css rgba))
  ([r g b & [a]]
   (if (and a (< a 1.0))
     (let [r (* 255 r) g (* 255 g) b (* 255 b)]
       (str "rgba(" (int r) "," (int g) "," (int b) "," (max 0.0 a) ")"))
     (f/format hex6 (rgba->int r g b)))))

(defn hsva->css
  ([hsva]
   (apply hsva->css hsva))
  ([h s v & [a]]
   (if a
     (apply rgba->css (conj (hsv->rgb h s v) a))
     (apply rgba->css (hsv->rgb h s v)))))

(defn hex->rgba
  [hex]
  (let [hex (if (= \# (first hex)) (subs hex 1) hex)
        rgba (if (== 3 (count hex))
               (let [[r g b] hex]
                 (int->rgba (f/parse-int (str r r g g b b) 16 0)))
               (int->rgba (f/parse-int hex 16 0)))]
    (if (and rgba (< (count hex) 7))
      (assoc rgba 3 1.0)
      rgba)))

(defn- parse-channel-val
  [^String c]
  (if (pos? (.indexOf c "%"))
    (* 0.01 (f/parse-float (subs c 0 (dec (count c)))))
    (* INV8BIT (f/parse-int c 10 0))))

(defn css->rgba
  [css]
  (if (= \# (first css))
    (hex->rgba css)
    (let [[[_ mode a b c d :as col]] (re-seq #"(rgb|hsl)a?\((\d+%?),(\d+%?),(\d+%?),?([0-9\.]+)?\)" css)]
      (if mode
        (if (#{"rgb" "rgba"} mode)
          (conj (mapv parse-channel-val [a b c]) (f/parse-float d 1.0))
          (let [h (/ (f/parse-float a) 360.0)
                s (parse-channel-val b)
                v (parse-channel-val c)]
            (conj (hsv->rgb h s v) (f/parse-float d 1.0))))))))

(def red first)

(def green #(nth % 1))

(def blue #(nth % 2))

(def hue first)

(def saturation #(nth % 1))

(def brightness #(nth % 2))

(def cyan first)

(def magenta #(nth % 1))

(def yellow #(nth % 2))

(def black #(nth % 3))

(def alpha #(nth % (if (== 4 (count %)) 3 4) 1))

(def hues
  (zipmap
   [:red :orange :yellow :lime :green :teal :cyan :azure :blue :purple :magenta]
   (map #(/ % 360.0) (range 0 360 30))))

(def primary-hues
  (select-keys hues [:red :yellow :green :cyan :blue :magenta]))

(defn closest-hue
  ([h] (closest-hue h hues))
  ([h hues]
     (first
      (reduce
       (fn [[h' d'] [k v]]
         (let [d (min (m/abs-diff h v) (m/abs-diff (dec h) v))]
           (if (< d d') [k d] [h' d'])))
       [nil 1e6] hues))))

(defn hue-rgb
  [rgb] (first (apply rgb->hsv rgb)))

(defn saturation-rgb
  [[r g b]]
  (let [v (max r g b)
        d (- v (min r g b))]
    (if (zero? v) 0.0 (/ d v))))

(defn brightness-rgb
  [[r g b]] (max r g b))

(defn luminance-rgb
  [[r g b]] (mm/madd 0.299 r 0.587 g 0.114 b))

(defn rotate-hue-hsv
  "Returns new HSV color with its hue rotated by theta (in radians)."
  [[h s v] theta]
  (let [h (+ h (/ (rem theta TWO_PI) TWO_PI))]
    [(cond (neg? h) (inc h) (>= h 1.0) (dec h) :default h) s v]))

(defn adjust-saturation-hsv
  [hsv x]
  (update-in hsv [1] #(m/clamp (+ x %) 0.0 1.0)))

(defn adjust-brightness-hsv
  [hsv x]
  (update-in hsv [2] #(m/clamp (+ x %) 0.0 1.0)))

(defn rotate-hue-rgb
  [rgb theta] (apply hsv->rgb (rotate-hue-hsv (apply rgb->hsv rgb) theta)))

(defn adjust-saturation-rgb
  [rgb x]
  (let [[h s v] (apply rgb->hsv rgb)]
    (hsv->rgb h (m/clamp (+ x s) 0.0 1.0) v)))

(defn adjust-brightness-rgb
  [rgb x]
  (let [[h s v] (apply rgb->hsv rgb)]
    (hsv->rgb h s (m/clamp (+ x v) 0.0 1.0))))

(defn adjust-alpha
  [col x]
  (update-in col [3] #(m/clamp (+ x (or % 1)) 0.0 1.0)))

(defn gamma-rgba
  [[r g b & [a]] e]
  (let [rgb' (mapv #(Math/pow % e) [r g b])]
    (if a (conj rgb' a) rgb')))

(defn complementary-hsv
  "Returns new HSV color with its hue rotated by 180 degrees."
  [hsv] (rotate-hue-hsv hsv PI))

(defn complementary-rgb
  [rgb] (rotate-hue-rgb rgb PI))

(defn invert-rgb
  [[r g b]] [(- 1.0 r) (- 1.0 g) (- 1.0 b)])

(defn invert-hsv
  [hsv] (->> hsv (apply hsv->rgb) (invert-rgb) (apply rgb->hsv)))

(defn dist-rgb
  [rgb1 rgb2]
  (let [dr (- (first rgb1) (first rgb2))
        dg (- (nth rgb1 1) (nth rgb2 1))
        db (- (nth rgb1 2) (nth rgb2 2))]
    (Math/sqrt (mm/madd dr dr dg dg db db))))

(defn dist-hsv
  [[ha sa va] [hb sb vb]]
  (let [ha (* TWO_PI ha)
        hb (* TWO_PI hb)
        dh (- (* sa (Math/cos ha)) (* sb (Math/cos hb)))
        ds (- (* sa (Math/sin ha)) (* sb (Math/sin hb)))
        dv (- va vb)]
    (Math/sqrt (mm/madd dh dh ds ds dv dv))))
(defn blend-rgb
  [rgb1 rgb2 t]
  [(mm/mix (first rgb1) (first rgb2) t)
   (mm/mix (nth rgb1 1) (nth rgb2 1) t)
   (mm/mix (nth rgb1 2) (nth rgb2 2) t)])

(defn blend-hsv
  [hsv1 hsv2 t]
  [(let [h1 (first hsv1)
         h2 (first hsv2)
         hd (m/abs-diff h1 h2)]
     (if (> hd 0.5)
       (if (> h2 h1)
         (rem (mm/mix (inc h1) h2 t) 1.0)
         (rem (mm/mix h1 (inc h2) t) 1.0))
       (mm/mix h1 h2 t)))
   (mm/mix (nth hsv1 1) (nth hsv2 1) t)
   (mm/mix (nth hsv1 2) (nth hsv2 2) t)])

(defn blend-rgba
  [[dr dg db da] [sr sg sb sa]]
  (let [sa (or sa 1.0)
        da (or da 1.0)
        da' (mm/subm 1.0 sa da)
        a'  (+ sa da')
        ia' (/ 1.0 a')]
    [(* ia' (mm/madd sr sa dr da'))
     (* ia' (mm/madd sg sa dg da'))
     (* ia' (mm/madd sb sa db da'))
     a']))
