(ns thi.ng.common.stringformat
  (:refer-clojure :exclude [float format]))

#+clj
(defn parse-int
  [^String x radix & [default]]
  (try (Integer/parseInt x radix) (catch Exception e default)))

#+clj
(defn parse-long
  [^String x radix & [default]]
  (try (Long/parseLong x radix) (catch Exception e default)))

#+clj
(defn parse-float
  [^String x & [default]]
  (try (Float/parseFloat x) (catch Exception e default)))

#+clj
(defn parse-double
  [^String x & [default]]
  (try (Double/parseDouble x) (catch Exception e default)))

#+clj
(defn parse-boolean
  [^String x] (Boolean/parseBoolean x))

#+cljs
(defn parse-int
  ([x] (parse-int x 10))
  ([x radix & [default]]
     (let [x' (js/parseInt x radix)]
       (if (js/isNaN x') default x'))))

#+cljs
(defn parse-float
  [x & [default]]
  (let [x' (js/parseFloat x)]
    (if (js/isNaN x') default x')))

(def float
  (memoize
   (fn [prec]
     #+clj  (let [fmt (str "%1." prec "f")]
              (fn [x] (clojure.core/format fmt (double x))))
     #+cljs (fn [x] (.toFixed (js/Number. x) prec)))))

(def pad-left
  (memoize
   (fn [len fill]
     (let [fill (apply str (repeat len fill))]
       (fn [x]
         (let [l (count x)]
           (if (< l len) (str (subs fill 0 (- len l)) x) x)))))))

(def pad-right
  (memoize
   (fn [len fill]
     (let [fill (apply str (repeat len fill))]
       (fn [x]
         (let [l (count x)]
           (if (< l len) (str x (subs fill 0 (- len l))) x)))))))

(def hex
  (memoize
   (fn [len]
     #+cljs
     (let [pad (pad-left len "0")]
       (fn [x] (pad (.toString (js/Number. (bit-or x 0)) 16))))
     #+clj
     (let [fmt (str "%0" len "x")]
       (fn [x] (clojure.core/format fmt (long x)))))))
       
(defn trunc-left
  [len]
  (fn [x]
    (let [l (count x)]
      (if (<= l len) x (subs x 0 len)))))

(defn trunc-right
  [len]
  (fn [x]
    (let [l (count x)]
      (if (<= l len) x (subs x (- l len))))))

(def date-formats
  (let [d2 (comp (pad-left 2 "0") str)]
    {:yyyy-mm-dd [str "-" d2 "-" d2]
     :yyyy-mm-dd-hh-mm-ss [str "-" d2 "-" d2 " " d2 ":" d2 ":" d2]
     :dd-mm-yyyy [d2 "/" d2 "/" str]
     :dd-mm-yy   [d2 "/" d2 "/" (comp (trunc-right 2) str)]}))

#+clj
(defn format
  [fmt & args]
  (loop [s (StringBuilder.), fmt fmt, args args]
    (if-let [f (first fmt)]
      (if (string? f)
        (recur (.append s f) (next fmt) args)
        (recur (.append s (f (first args))) (next fmt) (next args)))
      (.toString s))))

#+cljs
(defn format
  [fmt & args]
  (loop [s [], fmt fmt, args args]
    (if-let [f (first fmt)]
      (if (string? f)
        (recur (conj s f) (next fmt) args)
        (recur (conj s (f (first args))) (next fmt) (next args)))
      (apply str s))))
