(ns thi.ng.fabric.facts.core
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(declare index-selection make-index-selections add-query! add-query-join!)

(defprotocol IFactGraph
  (fact-indices [_])
  (add-fact! [_ t])
  (remove-fact! [_ t]))

(defprotocol ICache
  (cached [_ type k])
  (cache! [_ type k v])
  (expire! [_ type k]))

(defprotocol IFactQuery
  (raw-pattern [_])
  (result-vertex [_]))

  

(defn- signal-fact
  [vertex op] [op (f/vertex-id vertex) @vertex])

(defn- collect-index
  [spo]
  (f/collect-pure
   (fn [val incoming]
     ;;(debug :update-index spo incoming)
     (debug :old-index val)
     (let [val (transduce
                (map (fn [[op id t]] [op id (nth t spo)]))
                (completing
                 (fn [acc [op id x]]
                   (case op
                     :add    (update acc x (fnil conj #{}) id)
                     :remove (if-let [idx (acc x)]
                               (if (= #{id} idx)
                                 (dissoc acc x)
                                 (update acc x disj id))
                               acc)
                     (do (warn "ignoring unknown index signal op:" op)
                         acc))))
                val incoming)]
       (debug :new-index val)
       val))))

(defn- signal-index-select
  [vertex [idx sel]]
  [idx (if sel (@vertex sel [nil]) (->> @vertex vals (mapcat identity) (set)))])

(def ^:private collect-select
  (f/collect-pure
   (fn [val incoming]
     (let [val (reduce (fn [acc [idx res]] (assoc acc idx res)) val incoming)]
       ;;(debug :coll-select val incoming)
       val))))

(defn- score-collect-min-signal-vals
  [num]
  (fn [vertex]
    (if (> num (count (vals (peek (f/uncollected-signals vertex))))) 0 1)))

(defn- score-collect-min-signals
  [num]
  (fn [vertex]
    (if (> num (count (f/uncollected-signals vertex))) 0 1)))

(defn- aggregate-select
  [g]
  (f/collect-pure
   (fn [_ incoming]
     (let [res (vals (peek incoming))]
       ;;(debug :agg-incoming res)
       (if (every? #(not= [nil] %) res)
         (->> res
              (into #{} (map #(disj % nil)))
              (sort-by count)
              (reduce set/intersection)
              (into #{}
                    (comp (map #(f/vertex-for-id g %))
                          (filter identity)
                          (map deref))))
         #{})))))

(defn- score-collect-join
  [^thi.ng.fabric.core.Vertex vertex]
  (if (and (seq (f/uncollected-signals vertex))
           (== (count (f/signal-map vertex)) 2))
    1 0))

(defn- collect-inference
  [g production]
  (fn [^thi.ng.fabric.core.Vertex vertex]
    (let [prev @vertex
          in   (reduce into #{} (f/uncollected-signals vertex))
          adds (set/difference in prev)
          inferred (mapcat production adds)]
      (debug (f/vertex-id vertex) :additions adds)
      (doseq [[op t :as inf] inferred]
        (case op
          :+ (do (debug (f/vertex-id vertex) :rule-add-fact t)
                 (add-fact! g t))
          :- (do (debug (f/vertex-id vertex) :rule-remove-fact t)
                 (remove-fact! g t))
          (warn "invalid inference:" inf)))
      (swap! (.-value vertex) set/union adds))))

(defn- qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (= \? (.charAt ^String (name x) 0))))

(defn pattern-var-count
  [pattern]
  (count (filter qvar? pattern)))

(defn sort-patterns
  [patterns]
  (sort-by pattern-var-count patterns))

(defn select-keys*
  "Like c.c/select-keys, but doesn't retain map's meta"
  {:static true}
  [map keyseq]
  (loop [ret {} keys (seq keyseq)]
    (if keys
      #?(:clj
         (let [entry (. clojure.lang.RT (find map (first keys)))]
           (recur
            (if entry (conj ret entry) ret)
            (next keys)))
         :cljs
         (let [key   (first keys)
               entry (get map key ::not-found)]
           (recur
            (if (= entry ::not-found) ret (assoc ret key entry))
            (next keys))))
      ret)))

(defn index*
  "Like clojure.set/index, but using select-keys w/o metadata retention."
  [xrel ks]
  (persistent!
   (reduce
    (fn [m x]
      (let [ik (select-keys* x ks)]
        (assoc! m ik (conj (get m ik #{}) x))))
    (transient {}) xrel)))

(defn join
  "Optimized version of clojure.set/join (w/o key mapping & using
  transients). ~20% faster than original."
  [xrel yrel]
  (if (and (seq xrel) (seq yrel))
    (let [ks (set/intersection (set (keys (first xrel))) (set (keys (first yrel))))
          [r s] (if (<= (count xrel) (count yrel))
                  [xrel yrel]
                  [yrel xrel])
          idx (index* r ks)]
      (persistent!
       (reduce
        (fn [ret x]
          (let [found (idx (select-keys* x ks))]
            (if found
              (reduce #(conj! % (conj %2 x)) ret found)
              ret)))
        (transient #{}) s)))
    #{}))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb'       [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j   (first (join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))
(defmulti bind-translator
  (fn [qvars? pattern] (count pattern)))

(defmethod bind-translator 3
  [[vs? vp? vo?] [s p o]]
  (if vs?
    (if vp?
      (if vo?
        (fn [r] {s (r 0) p (r 1) o (r 2)})
        (fn [r] {s (r 0) p (r 1)}))
      (if vo?
        (fn [r] {s (r 0) o (r 2)})
        (fn [r] {s (r 0)})))
    (if vp?
      (if vo?
        (fn [r] {p (r 1) o (r 2)})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defmethod bind-translator 4
  [[vt? vs? vp? vo?] [t s p o]]
  (if vt?
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {t (r 0) s (r 1) p (r 2) o (r 3)})
          (fn [r] {t (r 0) s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) s (r 1) o (r 3)})
          (fn [r] {t (r 0) s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {t (r 0) p (r 2) o (r 3)})
          (fn [r] {t (r 0) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) o (r 3)})
          (fn [r] {t (r 0)}))))
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {s (r 1) p (r 2) o (r 3)})
          (fn [r] {s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {s (r 1) o (r 3)})
          (fn [r] {s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {p (r 2) o (r 3)})
          (fn [r] {p (r 2)}))
        (if vo?
          (fn [r] {o (r 3)})
          (fn [_] {}))))))
(defmulti fact-verifier
  (fn [qvars? pattern] (count pattern)))

(defmethod fact-verifier 3
  [[vs? vp? vo?] [s p o]]
  (cond
    (and vs? vp? vo?) (cond
                        (= s p o) #(= (% 0) (% 1) (% 2))
                        (= s p) #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
                        (= s o) #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
                        (= p o) #(and (= (% 1) (% 2)) (not= (% 0) (% 1)))
                        :else nil)
    (and vs? vp?)     (if (= s p) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vs? vo?)     (if (= s o) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vp? vo?)     (if (= p o) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    :else             nil))

(defmethod fact-verifier 4
  [[vt? vs? vp? vo?] [t s p o]]
  (cond
    (and vt? vs? vp? vo?)
    (cond
      (= t s p o)           #(= (% 0) (% 1) (% 2) (% 3))
      (= t s p)             #(and (not= (% 0) (% 3)) (= (% 0) (% 1) (% 2)))
      (= t s o)             #(and (not= (% 0) (% 2)) (= (% 0) (% 1) (% 3)))
      (= t p o)             #(and (not= (% 0) (% 1)) (= (% 0) (% 2) (% 3)))
      (= s p o)             #(and (not= (% 0) (% 1)) (= (% 1) (% 2) (% 3)))
      (and (= t s) (= p o)) #(and (= (% 0) (% 1)) (= (% 2) (% 3)))
      (and (= t p) (= s o)) #(and (= (% 0) (% 2)) (= (% 1) (% 3)))
      (and (= t o) (= s p)) #(and (= (% 0) (% 3)) (= (% 1) (% 2)))
      (= t s)               #(let [t (first %)] (and (= t (% 1)) (not= t (% 2)) (not= t (% 3))))
      (= t p)               #(let [t (first %)] (and (= t (% 2)) (not= t (% 1)) (not= t (% 3))))
      (= t o)               #(let [o (peek %)]  (and (= o (% 0)) (not= o (% 1)) (not= o (% 2))))
      (= s p)               #(let [s (nth % 1)] (and (= s (% 2)) (not= s (% 3)) (not= s (% 0))))
      (= s o)               #(let [o (peek %)]  (and (= o (% 1)) (not= o (% 2)) (not= o (% 0))))
      (= p o)               #(let [o (peek %)]  (and (= o (% 2)) (not= o (% 1)) (not= o (% 0))))
      :else                 nil)
    (and vt? vs? vp?)
    (cond
      (= t s p)             #(= (% 0) (% 1) (% 2))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vs? vo?)
    (cond
      (= t s o)             #(= (% 0) (% 1) (% 3))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vp? vo?)
    (cond
      (= t p o)             #(= (% 0) (% 2) (% 3))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 0)))
      :else                 nil)
    (and vs? vp? vo?)
    (cond
      (= s p o)             #(= (% 1) (% 2) (% 3))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 3)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 2)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 1)))
      :else                 nil)
    (and vt? vs?)           (if (= t s) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vt? vp?)           (if (= t p) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vt? vo?)           (if (= t o) #(= (% 0) (% 3)) #(not= (% 0) (% 3)))
    (and vs? vp?)           (if (= s p) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    (and vs? vo?)           (if (= s o) #(= (% 1) (% 3)) #(not= (% 1) (% 3)))
    (and vp? vo?)           (if (= p o) #(= (% 2) (% 3)) #(not= (% 2) (% 3)))
    :else                   nil))

(def ^:private fact-vertex-spec
  {::f/score-collect-fn (constantly 0)
   ::f/score-signal-fn  f/score-signal-with-new-edges})

(defn- index-vertex
  [g idx]
  (f/add-vertex!
   g {} {::f/collect-fn      (collect-index idx)
         ::f/score-signal-fn f/score-signal-with-new-edges}))

(defrecord FactGraph
    [g indices facts cache]
  f/IComputeGraph
  (add-vertex!
    [_ val vspec] (f/add-vertex! g val vspec))
  (remove-vertex!
    [_ v] (f/remove-vertex! g v))
  (vertex-for-id
    [_ id] (f/vertex-for-id g id))
  (vertices
    [_] (f/vertices g))
  (add-edge!
    [_ src dest sig opts] (f/add-edge! g src dest sig opts))
  f/IWatch
  (add-watch!
    [_ type id f] (f/add-watch! g type id f) _)
  (remove-watch!
    [_ type id] (f/remove-watch! g type id) _)
  IFactGraph
  (fact-indices
    [_] indices)
  (add-fact!
    [_ t]
    (or (@facts t)
        (let [v (f/add-vertex! g t fact-vertex-spec)]
          (run! #(f/add-edge! g v % signal-fact :add) indices)
          (swap! facts assoc t v)
          v)))
  (remove-fact!
    [_ t]
    (if-let [v (@facts t)]
      (do
        (run! #(f/add-edge! g v % signal-fact :remove) indices)
        (swap! facts dissoc t)
        (f/remove-vertex! g v)
        v)
      (warn "attempting to remove unknown fact:" t)))
  ICache
  (cached
    [_ type k] (get-in @cache [type k]))
  (cache!
    [_ type k v] (swap! cache assoc-in [type k] v) v)
  (expire!
    [_ type k] (swap! cache update type dissoc k) nil))
(defn fact-graph
  "Creates a new FactGraph instance with configurable backing
  IComputeGraph and fact length. If length is not specified, fact
  triples are used. Note that currently only triples or quads are
  supported! (len = 3 or 4)."
  ([]
   (fact-graph (f/compute-graph)))
  ([g]
   (fact-graph g 3))
  ([g len]
  (map->FactGraph
   {:indices (mapv #(index-vertex g %) (range len))
    :facts   (atom {})
    :cache   (atom {})
    :g       g})))

#?(:clj (prefer-method print-method clojure.lang.IRecord clojure.lang.IDeref))

(defrecord FactIndexSelection
    [index sel vertex]
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [v (f/add-vertex!
             g nil
             {::f/score-signal-fn f/score-signal-with-new-edges
              ::f/collect-fn      (f/collect-pure (fn [_ in] (peek in)))})]
      (f/add-edge! g index v signal-index-select sel)
      (cache! g ::index-sel sel (assoc _ :vertex v))))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? vertex parent)
      (do (warn :remove-index-sel sel)
          (f/disconnect-neighbor! index vertex)
          (f/remove-vertex! g vertex)
          true)
      (do (f/disconnect-neighbor! vertex parent)
          false))))
(defrecord BasicFactQuery
    [acc selections result pattern]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] pattern)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [[s p o] pattern
          selections (make-index-selections g pattern)
          acc   (f/add-vertex!
                 g {} {::f/collect-fn collect-select})
          res   (f/add-vertex!
                 g nil
                 {::f/collect-fn       (aggregate-select g)
                  ::f/score-signal-fn  f/score-signal-with-new-edges
                  ::f/score-collect-fn (score-collect-min-signal-vals (count selections))})]
      (run! #(f/add-edge! g (:vertex %) acc f/signal-forward nil) selections)
      (f/add-edge! g acc res f/signal-forward nil)
      (assoc _ :acc acc :result res :selections selections)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query pattern)
          (expire! g ::queries pattern)
          (f/remove-vertex! g result)
          (f/remove-vertex! g acc)
          (run! #(f/remove-from-graph! % g acc) selections)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord ParametricFactQuery
    [sub-query result pattern filter-fn]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] (mapv #(if-not (qvar? %) %) pattern))
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [qvars?     (mapv qvar? pattern)
          vmap       (bind-translator qvars? pattern)
          verify     (fact-verifier qvars? pattern)
          res-tx     (if verify
                       (comp (map #(if (verify %) (vmap %))) (filter filter-fn))
                       (comp (map vmap) (filter filter-fn)))
          collect-fn (f/collect-pure
                      (fn [_ incoming]
                        (if-let [res (seq (peek incoming))]
                          (into #{} res-tx res)
                          #{})))
          sub-q      (add-query! g (raw-pattern _))
          res        (f/add-vertex!
                      g nil
                      {::f/collect-fn      collect-fn
                       ::f/score-signal-fn f/score-signal-with-new-edges})
          this       (assoc _ :sub-query sub-q :result res)]
      (f/add-edge! g (result-vertex sub-q) res f/signal-forward nil)
      this))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-param-query pattern)
          (expire! g ::queries pattern)
          (f/remove-vertex! g result)
          (f/remove-from-graph! sub-query g result)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord FactQueryJoin
    [lhs rhs result join-fn]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] nil)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [lhs-v  (result-vertex lhs)
          rhs-v  (result-vertex rhs)
          lhs-id (f/vertex-id lhs-v)
          rhs-id (f/vertex-id rhs-v)
          result (f/add-vertex!
                  g nil
                  {::f/collect-fn
                   (fn [vertex]
                     (let [sig-map (f/signal-map vertex)
                           a (sig-map lhs-id)
                           b (sig-map rhs-id)]
                       (debug (f/vertex-id vertex) :join-sets a b)
                       (f/set-value! vertex (join-fn a b))))
                   ::f/score-collect-fn score-collect-join
                   ::f/score-signal-fn  f/score-signal-with-new-edges})]
      (f/add-edge! g lhs-v result f/signal-forward nil)
      (f/add-edge! g rhs-v result f/signal-forward nil)
      (assoc _ :result result)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query-join)
          (f/remove-vertex! g result)
          (f/remove-from-graph! lhs g result)
          (f/remove-from-graph! rhs g result)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord FactQueryUnion
    [sub-queries result]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] nil)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [res (f/add-vertex!
               g nil
               {::f/collect-fn
                (fn [vertex]
                  (let [subs (vals (f/signal-map vertex))]
                    (f/set-value! vertex (reduce (fn [acc res] (if (seq res) (into acc res) acc)) subs))))
                ::f/score-signal-fn f/score-signal-with-new-edges})]
      (run! #(f/add-edge! g (result-vertex %) res f/signal-forward nil) sub-queries)
      (assoc _ :result res)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query-union)
          (f/remove-vertex! g result)
          (run! #(f/remove-from-graph! % g result) sub-queries)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))

(defn index-selection
  [g sel]
  (if-let [sel (cached g ::index-sel sel)]
    (do (debug :reuse-index-sel (:sel sel)) sel)
    (f/add-to-graph!
     (map->FactIndexSelection
      {:index ((fact-indices g) (first sel))
       :sel   sel}) g)))

(defn make-index-selections
  [g pattern]
  (into [] (map-indexed #(index-selection g [% %2])) pattern))

(defn add-query!
  [g pattern]
  (if-let [q (cached g ::queries pattern)]
    (do (debug :reuse-basic-query pattern) q)
    (->> (f/add-to-graph!
          (map->BasicFactQuery {:pattern pattern}) g)
         (cache! g ::queries pattern))))

(defn add-param-query!
  ([g pattern]
   (if-let [q (cached g ::queries pattern)]
     (do (debug :reuse-param-query pattern) q)
     (->> (f/add-to-graph!
           (map->ParametricFactQuery {:filter-fn identity :pattern pattern}) g)
          (cache! g ::queries pattern))))
  ([g flt pattern]
   (f/add-to-graph!
    (map->ParametricFactQuery {:filter-fn flt :pattern pattern}) g)))

(defn add-join!
  ([g lhs rhs]
   (add-join! g join lhs rhs))
  ([g join-fn lhs rhs]
   (f/add-to-graph!
    (map->FactQueryJoin {:lhs lhs :rhs rhs :join-fn join-fn}) g)))

(defn add-query-join!
  [g & patterns]
  {:pre [(<= 2 (count patterns))]}
  (let [[a b & more :as p] (sort-patterns patterns)]
    (debug :sorted-join-patterns p)
    (reduce
     (fn [acc p] (add-join! g join acc (add-param-query! g p)))
     (add-join! g join (add-param-query! g a) (add-param-query! g b))
     more)))

(defn add-query-join-optional!
  [g a b & more]
  (reduce
   (fn [acc p] (add-join! g join-optional acc (add-param-query! g p)))
   (add-join! g join-optional (add-param-query! g a) (add-param-query! g b))
   more))

(defn add-query-union!
  [g a b & queries]
  (f/add-to-graph!
   (map->FactQueryUnion {:sub-queries (into [a b] queries)}) g))

(defn add-query-filter!
  [g flt q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (filter flt (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-query-group-by!
  [g gfn q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (group-by gfn (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-counter!
  [g src]
  (let [v (f/add-vertex!
           g nil
           {::f/collect-fn      (f/collect-pure (fn [_ in] (count (peek in))))
            ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g src v f/signal-forward nil)
    v))

(defrecord FactInferenceRule
    [id query patterns production inf]
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [q   (if (< 1 (count patterns))
                (apply add-query-join! g patterns)
                (add-param-query! g (first patterns)))
          inf (f/add-vertex!
               g #{} {::f/collect-fn (collect-inference g production)})]
      (f/add-edge! g (result-vertex q) inf f/signal-forward nil)
      (cache! g ::rules id (assoc _ :query q :inf inf))))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? inf parent)
      (do (warn :remove-rule id)
          (f/remove-vertex! g inf)
          (f/remove-from-graph! query g inf)
          true)
      (do (f/disconnect-neighbor! inf parent)
          false))))
(defn add-rule!
  [g id query production]
  (f/add-to-graph!
   (map->FactInferenceRule
    {:id id :patterns query :production production})
   g))

(def fact-log-transducer
  (comp
   (filter (fn [[op v]] (and (#{:add-vertex :remove-vertex} op) (vector? @v))))
   (map (fn [[op v]] [({:add-vertex :+ :remove-vertex :-} op) @v]))))

(defn add-fact-graph-logger
  [g log-fn]
  (let [ch        (chan 1024 fact-log-transducer)
        watch-id  (f/random-id)
        log->chan #(go (>! ch %))]
    (go-loop []
      (let [t (<! ch)]
        (when t
          (log-fn t)
          (recur))))
    (f/add-watch! g :add-vertex watch-id log->chan)
    (f/add-watch! g :remove-vertex watch-id log->chan)
    {:graph g :chan ch :watch-id watch-id}))

(defn remove-fact-graph-logger
  [{:keys [graph watch-id chan]}]
  (f/remove-watch! graph :add-vertex watch-id)
  (f/remove-watch! graph :remove-vertex watch-id)
  (close! chan))
