(ns thi.ng.fabric.facts.core
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [clojure.data.int-map :as imap]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(declare index-selection make-index-selections qvar? add-query! add-query-join!)

(defprotocol IFactGraph
  (fact-indices [_])
  (fact-transform [_])
  (add-fact! [_ t])
  (remove-fact! [_ t]))

(defprotocol ICache
  (cached [_ type k])
  (cache! [_ type k v])
  (expire! [_ type k]))

(defprotocol IFactQuery
  (raw-pattern [_])
  (result-vertex [_]))

(defprotocol ITwoWayTransform
  (transform [_ x])
  (untransform [_ x]))

  

(defn- signal-fact
  [vertex op] [op (f/vertex-id vertex) @vertex])

(defn- collect-index
  [spo]
  (let [tx (map (fn [[op id t]] [op id (nth t spo)]))
        rf (completing
            (fn [acc [op id x]]
              (case op
                :add    (assoc acc x (conj (or (acc x) #{}) id))
                :remove (if-let [idx (acc x)]
                          (if (= #{id} idx)
                            (dissoc acc x)
                            (assoc acc x (disj idx id)))
                          acc)
                (do (warn "ignoring unknown index signal op:" op)
                    acc))))]
    (f/collect-pure
     (fn [val incoming]
       ;;(debug :old-index val)
       (let [val (transduce tx rf val incoming)]
         ;;(debug :new-index val)
         val)))))

(defn- signal-index-select
  [vertex [idx sel]]
  [idx (if sel (@vertex sel [nil]) (->> @vertex vals (mapcat identity) (set)))])

(def ^:private collect-select
  (f/collect-pure
   (fn [val incoming]
     (let [val (reduce (fn [acc [idx res]] (assoc acc idx res)) val incoming)]
       ;;(debug :coll-select val incoming)
       val))))

(defn- score-collect-min-signal-vals
  [num]
  (fn [vertex]
    (if (> num (count (vals (peek (f/uncollected-signals vertex))))) 0 1)))

(defn- score-collect-min-signals
  [num]
  (fn [vertex]
    (if (> num (count (f/uncollected-signals vertex))) 0 1)))

(defn- collect-basic-query-results
  [g]
  (let [ftx    (fact-transform g)
        agg-tx (comp (map #(f/vertex-for-id g %))
                     (filter identity)
                     (map deref)
                     (map #(untransform ftx %)))]
    (f/collect-pure
     (fn [_ incoming]
       (let [res (vals (peek incoming))]
         ;;(debug :agg-incoming res)
         (if (every? #(not= [nil] %) res)
           (->> res
                (into #{} (map #(disj % nil)))
                (sort-by count)
                (reduce set/intersection)
                (into #{} agg-tx))
           #{}))))))

(defn- score-collect-join
  [vertex]
  (if (and (seq (f/uncollected-signals vertex))
           (== (count (f/signal-map vertex)) 2))
    1 0))

(defn- collect-inference
  [g production]
  (fn [vertex]
    (let [prev @vertex
          in   (reduce into #{} (f/uncollected-signals vertex))
          adds (set/difference in prev)]
      (debug (f/vertex-id vertex) :additions adds)
      (run! #(production g vertex %) adds)
      (f/update-value! vertex #(set/union % adds)))))

(def ^:dynamic *auto-qvar-prefix* "?__q")

(defn- qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (= \? (.charAt ^String (name x) 0))))

(defn auto-qvar?
  "Returns true, if x is an auto-generated qvar
  (a symbol prefixed with *auto-qvar-prefix*)"
  [x] (and (symbol? x) (zero? (.indexOf ^String (name x) ^String *auto-qvar-prefix*))))

(defn auto-qvar
  "Creates a new auto-named qvar (symbol)."
  [] (gensym *auto-qvar-prefix*))

(defn qvar-name
  [x] (-> x name (subs 1)))

(defn resolve-path-pattern
  "Takes a path triple pattern and max depth. The pattern's predicate
  must be a seq of preds. Returns a 2-elem vector [patterns vars],
  where `patterns` is a seq of query patterns with injected temp qvars
  for inbetween patterns and `vars` the autogenerated qvars themselves.
  Example:

      [?s [p1 p2 p3] ?o]
      => [([?s p1 ?__q0] [?__q0 p2 ?__q1] [?__q1 p3 ?o]) (?__q0 ?__q1)]"
  [[s p o] maxd]
  (let [avars (repeatedly maxd auto-qvar)
        vars  (cons s avars)]
    [(->> (concat (interleave vars (take maxd (cycle p))) [o])
          (partition 3 2))
     avars]))

(defn pattern-var-count
  [pattern]
  (count (filter qvar? pattern)))

(defn sort-patterns
  [patterns]
  (sort-by pattern-var-count patterns))

(defn select-keys*
  "Like c.c/select-keys, but doesn't retain map's meta"
  {:static true}
  [map keyseq]
  (loop [ret {} keys (seq keyseq)]
    (if keys
      #?(:clj
         (let [entry (. clojure.lang.RT (find map (first keys)))]
           (recur
            (if entry (conj ret entry) ret)
            (next keys)))
         :cljs
         (let [key   (first keys)
               entry (get map key ::not-found)]
           (recur
            (if (= entry ::not-found) ret (assoc ret key entry))
            (next keys))))
      ret)))

(defn index*
  "Like clojure.set/index, but using select-keys w/o metadata retention."
  [xrel ks]
  (persistent!
   (reduce
    (fn [m x]
      (let [ik (select-keys* x ks)]
        (assoc! m ik (conj (get m ik #{}) x))))
    (transient {}) xrel)))

(defn join
  "Optimized version of clojure.set/join. Does not join when there're
  no shared keys (w/o key mapping & using transients). ~20% faster
  than original."
  [xrel yrel]
  (if (and (seq xrel) (seq yrel))
    (let [ks (set/intersection (set (keys (first xrel))) (set (keys (first yrel))))]
      (if (seq ks)
        (let [[r s] (if (<= (count xrel) (count yrel))
                      [xrel yrel]
                      [yrel xrel])
              idx (index* r ks)]
          (persistent!
           (reduce
            (fn [ret x]
              (let [found (idx (select-keys* x ks))]
                (if found
                  (reduce #(conj! % (conj %2 x)) ret found)
                  ret)))
            (transient #{}) s)))
        #{}))
    #{}))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb'       [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j   (first (join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))
(defmulti bind-translator
  (fn [qvars? pattern] (count pattern)))

(defmethod bind-translator 3
  [[vs? vp? vo?] [s p o]]
  (if vs?
    (if vp?
      (if vo?
        (fn [r] {s (r 0) p (r 1) o (r 2)})
        (fn [r] {s (r 0) p (r 1)}))
      (if vo?
        (fn [r] {s (r 0) o (r 2)})
        (fn [r] {s (r 0)})))
    (if vp?
      (if vo?
        (fn [r] {p (r 1) o (r 2)})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defmethod bind-translator 4
  [[vt? vs? vp? vo?] [t s p o]]
  (if vt?
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {t (r 0) s (r 1) p (r 2) o (r 3)})
          (fn [r] {t (r 0) s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) s (r 1) o (r 3)})
          (fn [r] {t (r 0) s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {t (r 0) p (r 2) o (r 3)})
          (fn [r] {t (r 0) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) o (r 3)})
          (fn [r] {t (r 0)}))))
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {s (r 1) p (r 2) o (r 3)})
          (fn [r] {s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {s (r 1) o (r 3)})
          (fn [r] {s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {p (r 2) o (r 3)})
          (fn [r] {p (r 2)}))
        (if vo?
          (fn [r] {o (r 3)})
          (fn [_] {}))))))
(defmulti fact-verifier
  (fn [qvars? pattern] (count pattern)))

(defmethod fact-verifier 3
  [[vs? vp? vo?] [s p o]]
  (cond
    (and vs? vp? vo?) (cond
                        (= s p o) #(= (% 0) (% 1) (% 2))
                        (= s p) #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
                        (= s o) #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
                        (= p o) #(and (= (% 1) (% 2)) (not= (% 0) (% 1)))
                        :else nil)
    (and vs? vp?)     (if (= s p) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vs? vo?)     (if (= s o) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vp? vo?)     (if (= p o) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    :else             nil))

(defmethod fact-verifier 4
  [[vt? vs? vp? vo?] [t s p o]]
  (cond
    (and vt? vs? vp? vo?)
    (cond
      (= t s p o)           #(= (% 0) (% 1) (% 2) (% 3))
      (= t s p)             #(and (not= (% 0) (% 3)) (= (% 0) (% 1) (% 2)))
      (= t s o)             #(and (not= (% 0) (% 2)) (= (% 0) (% 1) (% 3)))
      (= t p o)             #(and (not= (% 0) (% 1)) (= (% 0) (% 2) (% 3)))
      (= s p o)             #(and (not= (% 0) (% 1)) (= (% 1) (% 2) (% 3)))
      (and (= t s) (= p o)) #(and (= (% 0) (% 1)) (= (% 2) (% 3)))
      (and (= t p) (= s o)) #(and (= (% 0) (% 2)) (= (% 1) (% 3)))
      (and (= t o) (= s p)) #(and (= (% 0) (% 3)) (= (% 1) (% 2)))
      (= t s)               #(let [t (first %)] (and (= t (% 1)) (not= t (% 2)) (not= t (% 3))))
      (= t p)               #(let [t (first %)] (and (= t (% 2)) (not= t (% 1)) (not= t (% 3))))
      (= t o)               #(let [o (peek %)]  (and (= o (% 0)) (not= o (% 1)) (not= o (% 2))))
      (= s p)               #(let [s (nth % 1)] (and (= s (% 2)) (not= s (% 3)) (not= s (% 0))))
      (= s o)               #(let [o (peek %)]  (and (= o (% 1)) (not= o (% 2)) (not= o (% 0))))
      (= p o)               #(let [o (peek %)]  (and (= o (% 2)) (not= o (% 1)) (not= o (% 0))))
      :else                 nil)
    (and vt? vs? vp?)
    (cond
      (= t s p)             #(= (% 0) (% 1) (% 2))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vs? vo?)
    (cond
      (= t s o)             #(= (% 0) (% 1) (% 3))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vp? vo?)
    (cond
      (= t p o)             #(= (% 0) (% 2) (% 3))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 0)))
      :else                 nil)
    (and vs? vp? vo?)
    (cond
      (= s p o)             #(= (% 1) (% 2) (% 3))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 3)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 2)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 1)))
      :else                 nil)
    (and vt? vs?)           (if (= t s) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vt? vp?)           (if (= t p) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vt? vo?)           (if (= t o) #(= (% 0) (% 3)) #(not= (% 0) (% 3)))
    (and vs? vp?)           (if (= s p) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    (and vs? vo?)           (if (= s o) #(= (% 1) (% 3)) #(not= (% 1) (% 3)))
    (and vp? vo?)           (if (= p o) #(= (% 2) (% 3)) #(not= (% 2) (% 3)))
    :else                   nil))

(def ^:private fact-vertex-spec
  {::f/score-collect-fn (constantly 0)
   ::f/score-signal-fn  f/score-signal-with-new-edges})

(defn- index-vertex
  [g idx]
  (f/add-vertex!
   g {} {::f/collect-fn      (collect-index idx)
         ::f/score-signal-fn f/score-signal-with-new-edges}))

(def identity-transform
  (reify ITwoWayTransform
    (transform [_ x] x)
    (untransform [_ x] x)))
(defn prefix-transform
  [prefixes]
  (reify ITwoWayTransform
    (transform
      [_ x]
      (if (string? x)
        (loop [[[pre pn] & more] (seq prefixes)]
          (if pre
            (if (zero? (.indexOf ^String x ^String pn))
              [pre (subs x (count pn))]
              (recur more))
            x))
        x))
    (untransform
      [_ x]
      (if (vector? x)
        (if-let [p (prefixes (first x))]
          (str p (nth x 1))
          x)
        x))))
(defn global-index-transform
  []
  (let [index (atom #?(:clj  {:fwd {} :rev (imap/int-map) :id 0}
                       :cljs {:fwd {} :rev {} :id 0}))]
    (reify ITwoWayTransform
      (transform
        [_ x]
        (if (or (nil? x) (qvar? x))
          x
          (or ((@index :fwd) x)
              (let [curr (volatile! nil)]
                (swap! index
                       #(let [id (:id %)]
                          (vreset! curr id)
                          (-> %
                              (update :id inc)
                              (update :fwd assoc x id)
                              (update :rev assoc id x))))
                @curr))))
      (untransform
        [_ id] ((@index :rev) id id)))))
(defn compose-transforms
  [& transforms]
  (let [rtx (reverse transforms)]
    (reify ITwoWayTransform
      (transform [_ x]
        (reduce #(transform %2 %) x transforms))
      (untransform [_ x]
        (reduce #(untransform %2 %) x rtx)))))

(defn one-way-transform
  [tx]
  (reify ITwoWayTransform
    (transform [_ x] (transform tx x))
    (untransform [_ x] x)))

(defn combine-transforms
  ([tx len]
   (apply combine-transforms (repeat len tx)))
  ([txs txp txo]
   (reify ITwoWayTransform
     (transform [_ fact]
       [(transform txs (first fact))
        (transform txp (nth fact 1))
        (transform txo (nth fact 2))])
     (untransform [_ fact]
       [(untransform txs (first fact))
        (untransform txp (nth fact 1))
        (untransform txo (nth fact 2))])))
  ([txt txs txp txo]
   (reify ITwoWayTransform
     (transform [_ fact]
       [(transform txt (first fact))
        (transform txs (nth fact 1))
        (transform txp (nth fact 2))
        (transform txo (nth fact 3))])
     (untransform [_ fact]
       [(untransform txt (first fact))
        (untransform txs (nth fact 1))
        (untransform txp (nth fact 2))
        (untransform txo (nth fact 3))]))))

(defrecord FactGraph
    [g indices facts cache ftx]
  f/IComputeGraph
  (add-vertex!
    [_ val vspec] (f/add-vertex! g val vspec))
  (remove-vertex!
    [_ v] (f/remove-vertex! g v))
  (vertex-for-id
    [_ id] (f/vertex-for-id g id))
  (vertices
    [_] (f/vertices g))
  (add-edge!
    [_ src dest sig opts] (f/add-edge! g src dest sig opts))
  f/IWatch
  (add-watch!
    [_ type id f] (f/add-watch! g type id f) _)
  (remove-watch!
    [_ type id] (f/remove-watch! g type id) _)
  IFactGraph
  (fact-indices
    [_] indices)
  (fact-transform
    [_] ftx)
  (add-fact!
    [_ f]
    (let [f' (transform ftx f)]
      (or (@facts f')
          (let [v (f/add-vertex! g f' fact-vertex-spec)]
            (debug :add-fact f f')
            (run! #(f/add-edge! g v % signal-fact :add) indices)
            (swap! facts assoc f' v)
            v))))
  (remove-fact!
    [_ f]
    (let [f' (transform ftx f)]
      (if-let [v (@facts f')]
        (do
          (debug :remove-fact f f')
          (run! #(f/add-edge! g v % signal-fact :remove) indices)
          (swap! facts dissoc f')
          (f/remove-vertex! g v)
          v)
        (warn "attempting to remove unknown fact:" f))))
  ICache
  (cached
    [_ type k] (get-in @cache [type k]))
  (cache!
    [_ type k v] (swap! cache assoc-in [type k] v) v)
  (expire!
    [_ type k] (swap! cache update type dissoc k) nil))
(defn fact-graph
  "Creates a new FactGraph instance configured with given options map:
  :graph     - backing IComputeGraph (default fabric.core/compute-graph)
  :len       - fact length (default 3)
  :index     - index vertex ctor (default index-vertex)
  :transform - fact transform (default none)"
  ([]
   (fact-graph {}))
  ([{:keys [graph len index transform]
     :or   {graph     (f/compute-graph)
            len       3
            index     index-vertex
            transform identity-transform}}]
   (map->FactGraph
    {:indices (mapv #(index graph %) (range len))
     :facts   (atom {})
     :cache   (atom {})
     :ftx     transform
     :g       graph})))

#?(:clj (prefer-method print-method clojure.lang.IRecord clojure.lang.IDeref))

(defn index-selection
  [g sel]
  (if-let [sel (cached g ::index-sel sel)]
    (do (debug :reuse-index-sel (:sel sel)) sel)
    (let [index  ((fact-indices g) (first sel))
          vertex (f/add-vertex!
                  g nil
                  {::f/score-signal-fn f/score-signal-with-new-edges
                   ::f/collect-fn      (f/collect-pure (fn [_ in] (peek in)))})
          isel   (reify
                   IFactQuery
                   (raw-pattern
                     [_] nil)
                   (result-vertex
                     [_] vertex)
                   f/IGraphComponent
                   (add-to-graph!
                     [_ g] _)
                   (remove-from-graph!
                     [_ g] (f/remove-from-graph! _ g nil))
                   (remove-from-graph!
                     [_ g parent]
                     (if (f/none-or-single-user? vertex parent)
                       (do (warn :remove-index-sel sel)
                           (f/disconnect-neighbor! index vertex)
                           (f/remove-vertex! g vertex)
                           true)
                       (do (f/disconnect-neighbor! vertex parent)
                           false))))]
      (f/add-edge! g index vertex signal-index-select sel)
      (cache! g ::index-sel sel isel))))

(defn make-index-selections
  [g pattern]
  (into [] (map-indexed #(index-selection g [% %2])) pattern))
(defn add-query!
  ([g pattern]
   (add-query! g (fact-transform g) pattern))
  ([g ptx pattern]
   (let [pattern (transform ptx pattern)]
     (if-let [q (cached g ::queries pattern)]
       (do (debug :reuse-basic-query pattern) q)
       (let [[s p o] pattern
             sels (make-index-selections g pattern)
             acc  (f/add-vertex!
                   g {} {::f/collect-fn collect-select})
             res  (f/add-vertex!
                   g nil
                   {::f/collect-fn       (collect-basic-query-results g)
                    ::f/score-signal-fn  f/score-signal-with-new-edges
                    ::f/score-collect-fn (score-collect-min-signal-vals (count sels))})
             q    (reify
                    #?@(:clj
                         [clojure.lang.IDeref (deref [_] (when res @res))]
                         :cljs
                         [IDeref (-deref [_] (when res @res))])
                    IFactQuery
                    (raw-pattern
                      [_] pattern)
                    (result-vertex
                      [_] res)
                    f/IGraphComponent
                    (add-to-graph!
                      [_ g] _)
                    (remove-from-graph!
                      [_ g] (f/remove-from-graph! _ g nil))
                    (remove-from-graph!
                      [_ g parent]
                      (if (f/none-or-single-user? res parent)
                        (do (warn :remove-query pattern)
                            (expire! g ::queries pattern)
                            (f/remove-vertex! g res)
                            (f/remove-vertex! g acc)
                            (run! #(f/remove-from-graph! % g acc) sels)
                            true)
                        (do (f/disconnect-neighbor! res parent)
                            false))))]
         (run! #(f/add-edge! g (result-vertex %) acc f/signal-forward nil) sels)
         (f/add-edge! g acc res f/signal-forward nil)
         (cache! g ::queries pattern q))))))
(defn add-param-query!
  ([g pattern]
   (add-param-query! g (fact-transform g) pattern))
  ([g ptx pattern]
   (add-param-query! g ptx identity pattern))
  ([g ptx flt pattern]
   (let [pattern (transform ptx pattern)]
     (if-let [q (cached g ::queries pattern)]
       (do (debug :reuse-param-query pattern) q)
       (let [qvars?  (mapv qvar? pattern)
             raw     (mapv #(if-not (qvar? %) %) pattern)
             vmap    (bind-translator qvars? pattern)
             verify  (fact-verifier qvars? pattern)
             res-tx  (if verify
                       (comp (map #(if (verify %) (vmap %))) (filter flt))
                       (comp (map vmap) (filter flt)))
             coll-fn (f/collect-pure
                      (fn [_ incoming]
                        (if-let [res (seq (peek incoming))]
                          (into #{} res-tx res)
                          #{})))
             sub-q   (add-query! g identity-transform raw)
             result  (f/add-vertex!
                      g nil
                      {::f/collect-fn      coll-fn
                       ::f/score-signal-fn f/score-signal-with-new-edges})
             q       (reify
                       #?@(:clj
                            [clojure.lang.IDeref (deref [_] (when result @result))]
                            :cljs
                            [IDeref (-deref [_] (when result @result))])
                       IFactQuery
                       (raw-pattern
                         [_] raw)
                       (result-vertex
                         [_] result)
                       f/IGraphComponent
                       (add-to-graph!
                         [_ g])
                       (remove-from-graph!
                         [_ g] (f/remove-from-graph! _ g nil))
                       (remove-from-graph!
                         [_ g parent]
                         (if (f/none-or-single-user? result parent)
                           (do (warn :remove-param-query pattern)
                               (expire! g ::queries pattern)
                               (f/remove-vertex! g result)
                               (f/remove-from-graph! sub-q g result)
                               true)
                           (do (f/disconnect-neighbor! result parent)
                               false))))]
         (f/add-edge! g (result-vertex sub-q) result f/signal-forward nil)
         (cache! g ::queries pattern q))))))
(defn add-join!
  ([g lhs rhs]
   (add-join! g join lhs rhs))
  ([g join-fn lhs rhs]
   (let [lhs-v  (result-vertex lhs)
         rhs-v  (result-vertex rhs)
         lhs-id (f/vertex-id lhs-v)
         rhs-id (f/vertex-id rhs-v)
         result (f/add-vertex!
                 g nil
                 {::f/collect-fn
                  (fn [vertex]
                    (let [sig-map (f/signal-map vertex)
                          a (sig-map lhs-id)
                          b (sig-map rhs-id)]
                      (debug (f/vertex-id vertex) :join-sets a b)
                      (f/set-value! vertex (join-fn a b))))
                  ::f/score-collect-fn score-collect-join
                  ::f/score-signal-fn  f/score-signal-with-new-edges})
         jq     (reify
                  #?@(:clj
                       [clojure.lang.IDeref (deref [_] (when result @result))]
                       :cljs
                       [IDeref (-deref [_] (when result @result))])
                  IFactQuery
                  (raw-pattern
                    [_] nil)
                  (result-vertex
                    [_] result)
                  f/IGraphComponent
                  (add-to-graph!
                    [_ g]
                    )
                  (remove-from-graph!
                    [_ g] (f/remove-from-graph! _ g nil))
                  (remove-from-graph!
                    [_ g parent]
                    (if (f/none-or-single-user? result parent)
                      (do (warn :remove-query-join)
                          (f/remove-vertex! g result)
                          (f/remove-from-graph! lhs g result)
                          (f/remove-from-graph! rhs g result)
                          true)
                      (do (f/disconnect-neighbor! result parent)
                          false))))]
     (f/add-edge! g lhs-v result f/signal-forward nil)
     (f/add-edge! g rhs-v result f/signal-forward nil)
     jq)))

(defn add-query-join!
  [g & patterns]
  (let [[ptx patterns] (if (satisfies? ITwoWayTransform (first patterns))
                         [(first patterns) (rest patterns)]
                         [(fact-transform g) patterns])
        [a b & more :as p] patterns] ;;(sort-patterns patterns)
    ;;(debug :sorted-join-patterns p)
    (reduce
     #(add-join! g join % (add-param-query! g ptx %2))
     (add-join! g join (add-param-query! g ptx a) (add-param-query! g ptx b))
     more)))

(defn add-query-join-optional!
  [g a b & more]
  (let [[ptx a b more] (if (satisfies? ITwoWayTransform a)
                         [a b (first more) (rest more)]
                         [(fact-transform g) a b more])]
    (assert b "Requires min. 2 query patterns")
    (reduce
     (fn [acc p] (add-join! g join-optional acc (add-param-query! g ptx p)))
     (add-join! g join-optional (add-param-query! g ptx a) (add-param-query! g ptx b))
     more)))
(defn add-query-union!
  [g & queries]
  (assert (< 1 (count queries)) "min. 2 queries required")
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn
              (fn [vertex]
                (let [subs (vals (f/signal-map vertex))]
                  (f/set-value! vertex (reduce (fn [acc res] (if (seq res) (into acc res) acc)) subs))))
              ::f/score-signal-fn f/score-signal-with-new-edges})
        q   (reify
              #?@(:clj
                   [clojure.lang.IDeref (deref [_] (when res @res))]
                   :cljs
                   [IDeref (-deref [_] (when res @res))])
              IFactQuery
              (raw-pattern
                [_] nil)
              (result-vertex
                [_] res)
              f/IGraphComponent
              (add-to-graph!
                [_ g])
              (remove-from-graph!
                [_ g] (f/remove-from-graph! _ g nil))
              (remove-from-graph!
                [_ g parent]
                (if (f/none-or-single-user? res parent)
                  (do (warn :remove-query-union)
                      (f/remove-vertex! g res)
                      (run! #(f/remove-from-graph! % g res) queries)
                      true)
                  (do (f/disconnect-neighbor! res parent)
                      false))))]
    (run! #(f/add-edge! g (result-vertex %) res f/signal-forward nil) queries)
    q))
(defn add-path-query!
  ([g path-pattern]
   (let [len (count (nth path-pattern 1))]
     (add-path-query! g path-pattern len len)))
  ([g path-pattern mind maxd]
   (add-path-query! g (fact-transform g) path-pattern mind maxd))
  ([g ftx path-pattern mind maxd]
   (assert (pos? mind) "min depth must be >= 1")
   (assert (<= mind maxd) "min depth must be <= max depth")
   (let [[patterns avars] (resolve-path-pattern path-pattern maxd)
         [?s _ ?o] path-pattern
         vs? (qvar? ?s)
         vo? (qvar? ?o)
         req (take mind patterns)
         opt (drop mind (take maxd patterns))
         req (if (seq req)
               (apply (if (== 1 (count req)) add-param-query! add-query-join!) g ftx req))
         opt (if (seq opt)
               (apply (if (== 1 (count opt)) add-param-query! add-query-join-optional!) g ftx opt))
         q   (if (and req opt)
               (add-join! g join-optional req opt)
               (or req opt))
         tx  (cond
               (or (= mind maxd) (and vs? (not vo?)))
               (let [qv (filter qvar? path-pattern)]
                 (map #(select-keys* % qv)))

               (and vo? (not vs?))
               (let [rv (take (dec mind) avars)]
                 (mapcat #(map (fn [v] {?o v}) (vals (apply dissoc % rv)))))

               :else
               (let [rv (cons ?s (take (dec mind) avars))]
                 (mapcat
                  #(let [s (% ?s)] (map (fn [v] {?s s ?o v}) (vals (apply dissoc % rv)))))))
         res (f/add-vertex!
              g #{}
              {::f/collect-fn (f/collect-pure (fn [_ in] (into #{} tx (peek in))))})
         pq  (reify
               #?@(:clj
                    [clojure.lang.IDeref (deref [_] (when res @res))]
                    :cljs
                    [IDeref (-deref [_] (when res @res))])
               IFactQuery
               (raw-pattern
                 [_] nil)
               (result-vertex
                 [_] res)
               f/IGraphComponent
               (add-to-graph!
                 [_ g] _)
               (remove-from-graph!
                 [_ g] (f/remove-from-graph! _ g nil))
               (remove-from-graph!
                 [_ g parent]
                 (if (f/none-or-single-user? res parent)
                   (do (warn :remove-path-query)
                       (f/remove-vertex! g res)
                       (f/remove-from-graph! q g res)
                       true)
                   (do (f/disconnect-neighbor! res parent)
                       false))))]
     (f/add-edge! g (result-vertex q) res f/signal-forward nil)
     pq)))
(defn add-query-filter!
  [g flt q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (filter flt (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-query-group-by!
  [g gfn q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (group-by gfn (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-counter!
  [g src]
  (let [v (f/add-vertex!
           g nil
           {::f/collect-fn      (f/collect-pure (fn [_ in] (count (peek in))))
            ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g src v f/signal-forward nil)
    v))

(defrecord FactInferenceRule
    [id query collect-fn inf]
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [inf (f/add-vertex! g #{} {::f/collect-fn collect-fn})]
      (f/add-edge! g (result-vertex query) inf f/signal-forward nil)
      (cache! g ::rules id (assoc _ :inf inf))))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? inf parent)
      (do (warn :remove-rule id)
          (f/remove-vertex! g inf)
          (f/remove-from-graph! query g inf)
          true)
      (do (f/disconnect-neighbor! inf parent)
          false))))
(defn add-rule!
  [g {:keys [id query patterns transform production collect-fn]}]
  (let [id         (or id (f/random-id))
        collect-fn (or collect-fn (collect-inference g production))
        query      (or query
                       (let [tx (or transform (fact-transform g))]
                         (if (< 1 (count patterns))
                           (apply add-query-join! g tx patterns)
                           (add-param-query! g tx (first patterns)))))]
    (f/add-to-graph!
     (map->FactInferenceRule
      {:id         id
       :query      query
       :collect-fn collect-fn})
     g)))

(def fact-log-transducer
  (comp
   (filter (fn [[op v]] (and (#{:add-vertex :remove-vertex} op) (vector? @v))))
   (map (fn [[op v]] [({:add-vertex :+ :remove-vertex :-} op) @v]))))

(defn add-fact-graph-logger
  [g log-fn]
  (let [ch        (chan 1024 fact-log-transducer)
        watch-id  (f/random-id)
        log->chan #(go (>! ch %))]
    (go-loop []
      (let [t (<! ch)]
        (when t
          (log-fn t)
          (recur))))
    (f/add-watch! g :add-vertex watch-id log->chan)
    (f/add-watch! g :remove-vertex watch-id log->chan)
    {:graph g :chan ch :watch-id watch-id}))

(defn remove-fact-graph-logger
  [{:keys [graph watch-id chan]}]
  (f/remove-watch! graph :add-vertex watch-id)
  (f/remove-watch! graph :remove-vertex watch-id)
  (close! chan))
