(ns thi.ng.fabric.facts.dsl
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [thi.ng.fabric.facts.core :as ff]
        [clojure.set :as set]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [thi.ng.fabric.facts.core :as ff]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(def vararg-ops
  {:= = :< < :> > :<= <= :>= >= :!= not=
   :+ + :- - :* * :div /})

(defmulti compile-expr
  (fn [expr]
    (let [op (cond
               (sequential? expr) (let [op (first expr)] (if (vararg-ops op) :varargs op))
               (ff/qvar? expr)    :qvar
               :else              :const)]
      ;;(prn op)
      op)))

(defmethod compile-expr :const
  [const] (fn [_] const))

(defmethod compile-expr :qvar
  [qvar] #(% qvar))

(defmethod compile-expr :varargs
  [[op & more]]
  (let [op   (vararg-ops op)
        args (mapv compile-expr more)]
    (fn [res] (apply op (sequence (comp (map #(% res)) (filter identity)) args)))))

(defmethod compile-expr :and
  [[_ & more]]
  (let [args (mapv compile-expr more)]
    (fn [res] (every? #(% res) args))))

(defmethod compile-expr :or
  [[_ & more]]
  (let [args (mapv compile-expr more)]
    (fn [res] (some #(% res) args))))

(defmethod compile-expr :match
  [[_ re x]]
  (let [x (compile-expr x)]
    (fn [res] (when-let [x' (x res)] (re-find re x')))))
(defn aggregation-with
  [op x]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (when (seq results)
        (transduce tx op results)))))

(def min* (fn ([] nil) ([x] x) ([x y] (if x (min x y) y))))
(def max* (fn ([] nil) ([x] x) ([x y] (if x (max x y) y))))

(defmethod compile-expr :sum
  [[_ x]]
  (aggregation-with + x))

(defmethod compile-expr :min
  [[_ x]]
  (aggregation-with min* x))

(defmethod compile-expr :max
  [[_ x]]
  (aggregation-with max* x))

(defmethod compile-expr :avg
  [[_ x]]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (let [res (sequence tx results)]
        (when (seq res)
          (double (/ (reduce + res) (count res))))))))

(defmethod compile-expr :mean
  [[_ x]]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (let [res (sequence tx results)]
        (nth (sort res) (bit-shift-right (count res) 1) nil)))))

(defmethod compile-expr :count
  [_] (fn [results] (count results)))

(defn compile-query-filter
  [flt] (if (fn? flt) flt (compile-expr flt)))

(defn compile-result-order
  [order]
  (if (fn? order)
    order
    (if (sequential? order)
      (fn [r] (reduce #(conj % (get r %2)) [] order))
      (fn [r] (get r order)))))

(defn compile-result-aggregation
  [agg]
  (let [agg (reduce-kv
             (fn [acc k v] (assoc acc k (compile-expr v)))
             {} agg)]
    (fn [results]
      (reduce-kv
       (fn [acc k afn]
         (assoc acc k (afn results)))
       {} agg))))

(defn compile-result-grouping
  [group]
  (if (fn? group)
    group
    (if (sequential? group)
      (fn [r] (reduce #(conj % (get r %2)) [] group))
      (fn [r] [(get r group)]))))

(defn sub-query-options
  [{:keys [filter limit]}]
  {:filter (if filter (compile-query-filter filter))
   :limit  limit})

(defmulti compile-sub-query
  (fn [g parent q spec] (some #{:where :optional :union} (keys q))))

(defmethod compile-sub-query :where
  [g parent q spec]
  (let [pat  (:where q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-join! g parent q {})
      q)))

(defmethod compile-sub-query :optional
  [g parent q spec]
  (let [pat  (:optional q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-join! g ff/join-optional parent q {})
      q)))

(defmethod compile-sub-query :union
  [g parent q spec]
  (let [pat  (:union q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-query-union! g [parent q] {})
      q)))

(defn compile-query
  [g {:keys [q] :as spec}]
  (let [spec (update spec :transform #(or % (ff/fact-transform g)))]
    (reduce
     (fn [acc sq] (compile-sub-query g acc sq spec))
     (compile-sub-query g nil (first q) spec)
     (rest q))))

(defn compile-select-qvars
  [result g spec]
  (let [spec (cond-> spec
               (:filter spec)    (update :filter compile-query-filter)
               (:order spec)     (update :order compile-result-order)
               (:group-by spec)  (update :group-by compile-result-grouping)
               (:aggregate spec) (update :aggregate compile-result-aggregation))]
    (ff/add-query-result! g spec result)))

(defn add-query-from-spec!
  [g spec]
  (-> (compile-query g spec)
      (compile-select-qvars g spec)))
