(ns thi.ng.fabric.facts.dsl
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [thi.ng.fabric.facts.core :as ff]
        [clojure.set :as set]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [thi.ng.fabric.facts.core :as ff]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(defn regexp?*
  [x] #?(:clj (= java.util.regex.Pattern (type x)) :cljs (regexp? x)))

(def vararg-ops
  {'+    [number? +]
   '-    [number? -]
   '*    [number? *]
   '/    [number? /]
   '=    [nil =]
   'not= [nil not=]
   'str  [nil str]})

(def unary-ops
  {'not   [nil not]
   'int   [number? int]
   'float [number? double]
   'abs   [number? #(Math/abs %)]
   'sqrt  [number? #(Math/sqrt %)]
   'exp   [number? #(Math/exp %)]
   'sin   [number? #(Math/sin %)]
   'asin  [number? #(Math/asin %)]
   'cos   [number? #(Math/cos %)]
   'acos  [number? #(Math/acos %)]
   'tan   [number? #(Math/tan %)]
   'atan  [number? #(Math/atan %)]
   'floor [number? #(long (Math/floor %))]
   'ceil  [number? #(long (Math/ceil %))]
   'round [number? #(Math/round (double %))]})

(def binary-ops
  {'<     [nil nil #(neg? (compare % %2))]
   '>     [nil nil #(pos? (compare % %2))]
   '<=    [nil nil #(<= (compare % %2) 0)]
   '>=    [nil nil #(>= (compare % %2) 0)]
   'match [regexp?* string? re-find]
   'pow   [number? number? #(Math/pow % %2)]
   'atan2 [number? number? #(Math/atan2 % %2)]
   'logn  [number? number? #(/ (Math/log %) (Math/log %2))]})

(defmulti compile-expr
  (fn [expr]
    (cond
      (sequential? expr) (let [op (first expr)]
                           (cond
                             (vararg-ops op) ::varargs
                             (unary-ops op)  ::unary
                             (binary-ops op) ::binary
                             :else           op))
      (ff/qvar? expr)    ::qvar
      :else              ::const)))

(defmethod compile-expr ::const
  [const] (fn [_] const))

(defmethod compile-expr ::qvar
  [qvar] #(% qvar))

(defmethod compile-expr ::varargs
  [[op & more]]
  (let [[check op] (vararg-ops op)
        args (mapv compile-expr more)]
    (if check
      (fn [res]
        (let [args' (sequence (comp (map #(% res)) (filter identity)) args)]
          (when (every? check args')
            (apply op args'))))
      (fn [res]
        (apply op (sequence (comp (map #(% res)) (filter identity)) args))))))

(defmethod compile-expr ::unary
  [[op x]]
  (let [[check op] (unary-ops op)
        x (compile-expr x)]
    (if check
      (fn [res] (let [x' (x res)] (when (check x') (op x'))))
      (fn [res] (op (x res))))))

(defmethod compile-expr ::binary
  [[op x y]]
  (let [[checkx checky op] (binary-ops op)
        x (compile-expr x)
        y (compile-expr y)]
    (cond
      (and checkx checky) (fn [res]
                            (let [x' (x res) y' (y res)]
                              (when (and (checkx x') (checky y')) (op x' y'))))
      checkx              (fn [res]
                            (let [x' (x res) y' (y res)]
                              (when (checkx x') (op x' y'))))
      checky              (fn [res]
                            (let [x' (x res) y' (y res)]
                              (when (checky y') (op x' y'))))
      :else               (fn [res] (op (x res) (y res))))))

(defmethod compile-expr 'and
  [[_ & more]]
  (let [args (mapv compile-expr more)]
    (fn [res] (every? #(% res) args))))

(defmethod compile-expr 'or
  [[_ & more]]
  (let [args (mapv compile-expr more)]
    (fn [res] (some #(% res) args))))

(defmethod compile-expr 'match
  [[_ re x]]
  (let [x (compile-expr x)]
    (fn [res] (let [x' (x res)] (when (string? x') (re-find re x'))))))

(defmethod compile-expr 'in-set?
  [[_ x & more]]
  (let [x       (compile-expr x)
        choices (mapv compile-expr more)]
    (fn [res] (let [x' (x res)] (some #(= (% res) x') choices)))))
(defn aggregation-with
  [op x]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (when (seq results)
        (transduce tx op results)))))

(def min* (fn ([] nil) ([x] x) ([x y] (if x (min x y) y))))
(def max* (fn ([] nil) ([x] x) ([x y] (if x (max x y) y))))

(defmethod compile-expr 'agg-sum
  [[_ x]]
  (aggregation-with + x))

(defmethod compile-expr 'agg-min
  [[_ x]]
  (aggregation-with min* x))

(defmethod compile-expr 'agg-max
  [[_ x]]
  (aggregation-with max* x))

(defmethod compile-expr 'agg-avg
  [[_ x]]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (let [res (sequence tx results)]
        (when (seq res)
          (double (/ (reduce + res) (count res))))))))

(defmethod compile-expr 'agg-mean
  [[_ x]]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results]
      (let [res (sequence tx results)]
        (nth (sort res) (bit-shift-right (count res) 1) nil)))))

(defmethod compile-expr 'agg-collect
  [[_ x]]
  (let [x  (compile-expr x)
        tx (comp (map x) (filter identity))]
    (fn [results] (into #{} tx results))))

(defmethod compile-expr 'agg-count
  [_] (fn [results] (count results)))
(defmethod compile-expr 'group-bins-of
  [[_ x n]]
  (let [x (compile-expr x)]
    (fn [res] (* (Math/floor (/ (x res) n)) n))))

(defn compile-query-filter
  [flt] (if (fn? flt) flt (compile-expr flt)))

(defn compile-result-order
  [order]
  (if (fn? order)
    order
    (if (sequential? order)
      (fn [r] (reduce #(conj % (get r %2)) [] order))
      (fn [r] (get r order)))))

(defn compile-result-aggregation
  [agg]
  (let [agg (reduce-kv
             (fn [acc k v] (assoc acc k (compile-expr v)))
             {} agg)]
    (fn [results]
      (reduce-kv
       (fn [acc k afn]
         (assoc acc k (afn results)))
       {} agg))))

(defn compile-result-grouping
  [group]
  (if (fn? group)
    group
    (cond
      (and (sequential? group) (every? ff/qvar? group))
      (fn [r] (reduce #(conj % (get r %2)) [] group))

      (ff/qvar? group)
      (fn [r] (get r group))

      :else
      (compile-expr group))))

(defn sub-query-options
  [opts]
  (-> opts
      (select-keys [:filter :limit :select])
      (update :filter #(if % (compile-query-filter %)))))

(defmulti compile-sub-query
  (fn [g parent q spec] (some #{:where :optional :union :path} (keys q))))

(defmethod compile-sub-query :where
  [g parent q spec]
  (let [pat  (:where q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-join! g parent q {})
      q)))

(defmethod compile-sub-query :optional
  [g parent q spec]
  (let [pat  (:optional q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-join! g ff/join-optional parent q {})
      q)))

(defmethod compile-sub-query :union
  [g parent q spec]
  (let [pat  (:union q)
        opts (sub-query-options q)
        q    (if (< 1 (count pat))
               (ff/add-query-join! g (:transform spec) pat opts)
               (ff/add-param-query! g (:transform spec) (first pat) opts))]
    (if parent
      (ff/add-query-union! g [parent q] {})
      q)))

(defmethod compile-sub-query :path
  [g parent q spec]
  (let [opts (merge (select-keys q [:min :max]) (sub-query-options q))
        q    (ff/add-path-query! g (:transform spec) (:path q) opts)]
    (if parent
      (ff/add-join! g parent q {})
      q)))

(defn compile-query
  [g {:keys [q] :as spec}]
  (let [spec (update spec :transform #(or % (ff/fact-transform g)))]
    (reduce
     (fn [acc sq] (compile-sub-query g acc sq spec))
     (compile-sub-query g nil (first q) spec)
     (rest q))))

(defn compile-select-qvars
  [result g spec]
  (let [spec (cond-> spec
               (:filter spec)    (update :filter compile-query-filter)
               (:order spec)     (update :order compile-result-order)
               (:group-by spec)  (update :group-by compile-result-grouping)
               (:aggregate spec) (-> (assoc :aggregate* (:aggregate spec))
                                     (update :aggregate compile-result-aggregation)))]
    (ff/add-query-result! g spec result)))

(defn add-query-from-spec!
  [g spec]
  (-> (compile-query g spec)
      (compile-select-qvars g spec)))
