(ns thi.ng.fabric.ld.core
  (:require
   [thi.ng.fabric.core :as f]
   [thi.ng.fabric.facts.core :as ff]
   [thi.ng.fabric.facts.dsl :as dsl]
   [thi.ng.fabric.facts.io.ntriples :as nt]
   [thi.ng.fabric.ld.version :refer [version]]
   [thi.ng.strf.core :as strf]
   [thi.ng.xerror.core :as err]
   [thi.ng.validate.core :as v]
   [com.stuartsierra.component :as comp]
   [clojure.tools.namespace.repl :refer (refresh)]
   [manifold.deferred :as d]
   [aleph.http :as http]
   [byte-streams :as bs]
   [compojure.core :as compojure :refer [GET POST DELETE]]
   [compojure.route :as route]
   [ring.middleware.defaults :refer [wrap-defaults site-defaults]]
   [ring.middleware.stacktrace :refer [wrap-stacktrace]]
   [ring.util.response :as resp]
   [clojure.java.io :as io]
   [clojure.edn :as edn]
   [clojure.data.csv :as csv]
   [clojure.data.json :as json]
   [clojure.string :as str]
   [clojure.walk :as walk]
   [clojure.core.async :as async :refer [go go-loop <! >!]]
   [taoensso.timbre :refer [debug info warn]])
  (:import
   java.io.PushbackReader))

(taoensso.timbre/set-level! :info)

(defprotocol IGraphAccess
  (get-graph [_])
  (get-facts [_]))

(defprotocol IGraphModel
  (trigger-update! [_])
  (result-event-bus [_])
  (add-facts! [_ facts])
  (remove-facts! [_ facts])
  (add-query! [_ q])
  (remove-query! [_ q])
  (transform-query [_ q])
  (add-rule! [_ rule])
  (remove-rule! [_ rule]))

(defprotocol IEntityRegistry
  (register! [_ id val])
  (deregister! [_ id])
  (lookup [_ id])
  (list-registry [_]))

(defprotocol IHandler
  (route-map [_]))

(defn find-cached
  [cache spec]
  (some (fn [[id q]] (if (= (:spec q) spec) id)) cache))

(defn compile-production-item
  [[op fact]]
  (let [fact (map #(if (ff/qvar? %) (fn [r] (r %)) (fn [_] %)) fact)]
    (case op
      :add    (fn [g res] (ff/add-fact! g (mapv #(% res) fact)))
      :remove (fn [g res] (ff/remove-fact! g (mapv #(% res) fact)))
      (throw (IllegalArgumentException. (str "invalid production op: " op))))))

(defn compile-rule-production
  [rule-id prod]
  (let [prods (mapv compile-production-item prod)]
    (fn [g _ res] (info :infer rule-id res) (run! #(% g res) prods))))

(defrecord PrefixRegistry [config registry]
  comp/Lifecycle
  (start
    [_]
    (info "starting prefix registry")
    (let [this (assoc _ :registry (atom {:prefixes (sorted-map)}))]
      (run! (fn [[pre uri]] (register! this pre uri)) (:prefixes config))
      this))
  (stop
    [_]
    (info "stopping prefix registry")
    (assoc _ :registry nil))
  IEntityRegistry
  (register!
    [_ prefix uri]
    (info "register! prefix:" prefix uri)
    (swap! registry
           (fn [reg]
             (let [reg (assoc-in reg [:prefixes prefix] uri)]
               (assoc reg :sorted (ff/sort-prefixes (:prefixes reg))))))
    _)
  (deregister!
    [_ prefix]
    (info "deregister! prefix:" prefix)
    (swap! registry
           (fn [reg]
             (let [reg (update reg :prefixes dissoc prefix)]
               (assoc reg :sorted (ff/sort-prefixes (:prefixes reg))))))
    _)
  (list-registry
    [_] (:prefixes @registry))
  (lookup
    [_ prefix] (get-in @registry [:prefixes prefix]))
  ff/ITwoWayTransform
  (transform
    [_ x] (ff/->prefix-string (:sorted @registry) x))
  (untransform
    [_ x] (ff/<-prefix-string (:prefixes @registry) x)))

(defn make-prefix-registry
  [config]
  (map->PrefixRegistry {:config config}))
(defrecord QueryRegistry [model registry config]
  comp/Lifecycle
  (start
    [_]
    (info "starting query registry")
    (let [this (assoc _ :registry (atom (sorted-map)))]
      (run! (fn [[id spec]] (register! this (name id) spec)) (:specs config))
      this))
  (stop
    [_]
    (info "stopping query registry")
    (assoc _ :registry nil :model nil))
  IEntityRegistry
  (register!
    [_ id spec]
    (if-let [id' (find-cached @registry spec)]
      id'
      (let [id (or id (str (java.util.UUID/randomUUID)))]
        (when-not (@registry id)
          (info "register! query:" id spec)
          (let [q (add-query! model spec)]
            (swap! registry assoc id {:spec spec :query q})
            id)))))
  (deregister!
    [_ id]
    (when-let [q (:query (@registry id))]
      (info "deregister! query:" id)
      (remove-query! model q)
      (swap! registry dissoc id)
      q))
  (lookup
    [_ id] (@registry id))
  (list-registry
    [_] @registry))

(defn make-query-registry
  [config]
  (map->QueryRegistry {:config config}))
(defrecord RuleRegistry [model registry config]
  comp/Lifecycle
  (start
    [_]
    (info "starting rule registry")
    (let [this (assoc _ :registry (atom (sorted-map)))]
      (run! (fn [[id spec]] (register! this (name id) spec)) (:specs config))
      this))
  (stop
    [_]
    (info "stopping rule registry")
    (assoc _ :registry nil :model nil))
  IEntityRegistry
  (register!
    [_ id spec]
    (if-let [id' (find-cached @registry spec)]
      id'
      (let [id (or id (str (java.util.UUID/randomUUID)))]
        (when-not (@registry id)
          (info "register! rule:" id spec)
          (let [prod  (compile-rule-production id (:prod spec))
                query (add-query! model (:q spec))
                rule  (add-rule! model {:id id :query query :production prod})]
            (swap! registry assoc id {:spec spec :rule rule})
            id)))))
  (deregister!
    [_ id]
    (when-let [r (:rule (@registry id))]
      (info "deregister! rule:" id)
      (remove-rule! model r)
      (swap! registry dissoc id)
      r))
  (lookup
    [_ id] (@registry id))
  (list-registry
    [_] @registry))

(defn make-rule-registry
  [config]
  (map->RuleRegistry {:config config}))

(defn- parse-edn-facts
  [body]
  (when-let [facts (try (edn/read-string body) (catch Exception e))]
    (cond
      (map? facts)  [(ff/map->facts facts)]
      (coll? facts) [facts]
      :else         [nil (str "EDN value is not a collection")])))

(defn load-facts-from-uri
  ([uri]
   (load-facts-from-uri uri 0 2))
  ([uri depth max-redirects]
   (try
     (info "loading facts from:" uri)
     (let [resp  @(http/get
                   uri
                   {:headers {"Accept"     "application/edn,application/n-triples"
                              "User-Agent" (str "thi.ng-fabric-ld/" version)}})
           body  (bs/to-string (:body resp))
           ctype (or (get-in resp [:headers "content-type"])
                     (get-in resp [:headers "Content-Type"]))]
       (if (== 303 (:status resp))
         (if (< depth max-redirects)
           (load-facts-from-uri
            (or (get-in resp [:headers "location"])
                (get-in resp [:headers "Location"]))
            (inc depth) max-redirects)
           [nil "Reached max number of redirects, giving up"])
         (condp = ctype
           "application/edn"       (parse-edn-facts body)
           "application/n-triples" [(nt/parse-ntriples body)]
           [nil (str "Remote server returned unsupported content type" ctype)])))
     (catch Exception e
       [nil (str "Error loading/parsing facts from remote server: " (.getMessage e))]))))

(defn- restore-graph-from-log
  "Reads graph log file and applies all fact additions/removals to graph."
  [g path]
  (try
    (let [facts (slurp path)
          facts (edn/read-string (str "[" facts "]"))]
      (loop [facts facts adds 0 dels 0]
        (if facts
          (let [[op f] (first facts)]
            (if (= :add-fact op)
              (do (ff/add-fact! g f)
                  (recur (next facts) (inc adds) dels))
              (do (ff/remove-fact! g f)
                  (recur (next facts) adds (inc dels)))))
          (info "added" adds "and removed" dels "facts from log:" path))))
    (catch Exception e
      (warn "failed to restore graph log:" path (.getMessage e)))))

(defrecord LDGraph [graph prefixes config]
  comp/Lifecycle
  (start
    [_]
    (info "initializing graph...")
    (let [opts (merge {:transform (ff/combine-transforms prefixes 3)} config)
          g    (ff/fact-graph opts)]
      (doseq [uri (:import config)]
        (try
          (let [^String uri' (.toString uri)
                [facts err]  (cond
                               (.startsWith uri' "http") (load-facts-from-uri uri)
                               (.endsWith uri' ".edn")   (parse-edn-facts (slurp uri))
                               (.endsWith uri' ".nt")    [(nt/parse-ntriples (slurp uri))]
                               :else                     [nil "Unsupported content type"])]
            (if facts
              (do (info "adding" (count facts) "facts from" uri)
                  (run! #(ff/add-fact! g %) facts))
              (warn uri err)))
          (catch Exception e)))
      (when-let [restore (:restore config)]
        (restore-graph-from-log g restore))
      (assoc _ :graph g)))
  (stop
    [_]
    (info "stop graph component")
    (assoc _ :graph nil))
  IGraphAccess
  (get-graph
    [_] graph)
  (get-facts
    [_]
    (let [ftx (ff/fact-transform graph)]
      (map #(ff/untransform ftx %) (ff/facts graph)))))

(defn make-graph
  [config]
  (map->LDGraph {:config config}))

(defrecord FactLog [config graph logger prefixes]
  comp/Lifecycle
  (start
    [_]
    (info "starting fact log")
    (let [log-fn ((:fn config) config prefixes)
          logger (ff/add-fact-graph-logger (get-graph graph) log-fn)]
      (run! #(log-fn [:add-fact %]) (get-facts graph))
      (assoc _ :logger logger)))
  (stop
    [_]
    (info "stop fact log")
    (ff/remove-fact-graph-logger logger)
    (assoc _ :graph nil :logger nil :prefixes nil)))

(defn make-fact-log
  [config]
  (map->FactLog {:config config}))

(defn default-logger
  [config prefixes]
  (let [path (strf/format (:path config) (strf/now))]
    (info "fact logger writing to:" path)
    (fn [evt]
      (info :fact-log evt)
      (spit path (str (pr-str evt) "\n") :append true))))

(defn- model-work-queue-processor
  [queue rbus]
  (let [res (async/tap rbus (async/chan))]
    (go-loop []
      (if-let [work (<! queue)]
        (let [[type f] work]
          (info "executing queue item:" type)
          (f)
          (if (<! res)
            (recur)
            (info "context finished")))
        (info "work queue closed")))))

(defn- queue-task
  [queue id f] (go (>! queue [id f])))

(defn- model-result-log
  [rbus]
  (let [logtap  (async/tap rbus (async/chan))]
    (go-loop []
      (when-let [res (<! logtap)]
        (info "context stats:" res)
        (recur)))))

(defn untransform-pnames-in-query
  [tx spec]
  (let [spec (reduce
              (fn [acc k]
                (let [v (acc k)]
                  (cond
                    (coll? v)   (assoc acc k (walk/postwalk #(ff/untransform tx %) v))
                    (string? v) (assoc acc k (ff/untransform tx v))
                    :else       acc)))
              spec [:q :bind :filter :aggregate :order :group-by])
        spec (if-let [q (:q spec)]
               (assoc spec :q
                      (reduce
                       (fn [acc sub]
                         (if-let [vals (:values sub)]
                           (conj acc (assoc sub :values (walk/postwalk #(ff/transform tx %) vals)))
                           (conj acc sub)))
                       [] q))
               spec)
        spec (if-let [vals (:values spec)]
               (assoc spec :values (walk/postwalk #(ff/transform tx %) vals))
               spec)]
    spec))

(defrecord LDGraphModel [graph prefixes ctx queue rbus rbus-in]
  comp/Lifecycle
  (start
    [_]
    (info "starting graph model")
    (let [queue   (async/chan)
          rbus-in (async/chan)
          rbus    (async/mult rbus-in)
          ctx     (f/async-execution-context {:graph (get-graph graph) :result rbus-in})
          this    (assoc _ :ctx ctx :queue queue :rbus rbus :rbus-in rbus-in)]
      (model-work-queue-processor queue rbus)
      (model-result-log rbus)
      (f/execute! ctx)
      this))
  (stop
    [_]
    (info "stop graph model")
    (f/stop! ctx)
    (async/close! rbus-in)
    (async/close! queue)
    (assoc _ :graph nil :ctx nil :queue nil :rbus nil :rbus-in nil))
  IGraphAccess
  (get-facts
    [_] (get-facts graph))
  (get-graph
    [_] (get-graph graph))
  IGraphModel
  (trigger-update!
    [_] (f/notify! ctx))
  (result-event-bus
    [_] rbus)
  (add-facts!
    [_ facts]
    (queue-task queue :add-facts
                (fn []
                  (let [g (get-graph graph)]
                    (run! #(ff/add-fact! g %) facts)
                    (f/notify! ctx))))
    _)
  (remove-facts!
    [_ facts]
    (queue-task queue :remove-facts
                (fn []
                  (let [g (get-graph graph)]
                    (run! #(ff/remove-fact! g %) facts)
                    (f/notify! ctx))))
    _)
  (transform-query
    [_ q] (untransform-pnames-in-query prefixes q))
  (add-query!
    [_ q]
    (let [out (promise)]
      (queue-task queue :add-query
                  (fn []
                    (let [q (dsl/add-query-from-spec! (get-graph graph) q)]
                      (f/notify! ctx)
                      (deliver out q))))
      @out))
  (remove-query!
    [_ q]
    (queue-task queue :remove-query
                (fn []
                  (f/remove-from-graph! q (get-graph graph))
                  (f/notify! ctx)))
    _)
  (add-rule!
    [_ rule]
    (let [out (promise)]
      (queue-task queue :add-rule
                  (fn []
                    (let [r (ff/add-rule! (get-graph graph) rule)]
                      (f/notify! ctx)
                      (deliver out r))))
      @out))
  (remove-rule!
    [_ rule]
    (queue-task queue :remove-rule
                (fn []
                  (f/remove-from-graph! rule (get-graph graph))
                  (f/notify! ctx)))
    _))

(defn make-graph-model
  [config]
  (map->LDGraphModel {}))

(defrecord AlephServer [config server handler]
  comp/Lifecycle
  (start [_]
    (if-not server
      (let [port   (:port config)
            routes (route-map handler)]
        (info "starting aleph server on port:" port)
        (assoc _ :server (http/start-server routes {:port port})))
      (do (warn "aleph server already running...")
          _)))
  (stop [_]
    (if server
      (do (info "stopping server...")
          (.close ^java.io.Closeable server)
          (assoc _ :server nil))
      (do (warn "aleph server already stopped!")
          _))))

(defn make-server
  [config]
  (map->AlephServer {:config config}))

(def fmt-iso8601 (java.text.SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ssX"))

(defn uri?
  [x] (and (string? x) (re-find #"^(https?|ftp|mailto|urn|uuid|isbn)://" x)))

(defmulti transform-body (fn [x] (type x)))

(defmethod transform-body :default
  [x] x)

(defmethod transform-body java.util.Date
  [x] (.format ^java.text.SimpleDateFormat fmt-iso8601 ^java.util.Date x))

(defn transform-body*
  [body] (walk/postwalk transform-body body))

(defmulti api-response (fn [type body rtype] [type rtype]))

(defmethod api-response :default
  [_ body _] nil)

(defn- edn-response
  [body] [(pr-str (dissoc body :columns)) "application/edn"])

(defmethod api-response ["*/*" :default]
  [_ body _] (edn-response body))

(defmethod api-response ["*/*" :query]
  [_ body _] (edn-response body))

(defmethod api-response ["*/*" :query-group]
  [_ body _] (edn-response body))

(defmethod api-response ["application/edn" :default]
  [_ body _] (edn-response body))

(defmethod api-response ["application/edn" :query]
  [_ body _] (edn-response body))

(defmethod api-response ["application/edn" :query-group]
  [_ body _] (edn-response body))

(defn- json-response
  ([body]
   (json-response body "application/json"))
  ([body mime]
   [(json/write-str (transform-body* (dissoc body :columns)) :escape-slash false)
    mime]))

(defmethod api-response ["application/json" :default]
  [_ body _] (json-response body))

(defmethod api-response ["application/json" :query]
  [_ body _] (json-response body))

(defmethod api-response ["application/json" :query-group]
  [_ body _] (json-response body))

(defn- ld-json-group
  [[s vals]]
  (reduce
   (fn [acc [_ p o]]
     (let [p (if (= "http://www.w3.org/1999/02/22-rdf-syntax-ns#type" p) "@type" (str p))
           o (if (= "@type" p) o (if (uri? o) {"@id" o} {"@value" o}))]
       (if-let [v (acc p)]
         (if (v "@set")
           (update-in acc [p "@set"] conj o)
           (assoc acc p {"@set" [v o]}))
         (assoc acc p o))))
   {"@id" s}
   vals))

;; http://www.w3.org/TR/json-ld/
(defmethod api-response ["application/ld+json" :default]
  [_ body _]
  (let [body' (map ld-json-group (group-by first (:body body)))
        body' {"@graph" body'}]
    (json-response (assoc body :body body'))))

;; http://www.w3.org/TR/sparql11-results-json/
(defmethod api-response ["application/sparql-results+json" :query]
  [_ body _]
  (let [qvars (if-let [sel (get-in body [:spec :select])]
                (if (sequential? sel) sel [sel])
                (into #{} (mapcat keys) (:body body)))
        qvars (mapv ff/qvar-name qvars)
        body  (map
               (fn [res]
                 (reduce-kv
                  #(assoc % (ff/qvar-name %2) {:type (if (uri? %3) :uri :literal) :value %3}) ;; TODO BNodes
                  {} res))
               (:body body))]
    (json-response
     {:head {:vars qvars}
      :results {:bindings body}}
     "application/sparql-results+json")))

(defn- csv-response
  [body & opts]
  [(with-open [out (java.io.StringWriter.)]
     (apply csv/write-csv out (transform-body* body) opts)
     (.toString out))
   "text/csv"])

(defn- sparql-csv-item
  [res qvar]
  (let [x (res qvar)]
    (if (uri? x) (str \< x \>) x)))

(defmethod api-response ["text/csv" :default]
  [_ body _]
  (if-let [body' (:body body)]
    (let [body' (if-let [cols (:columns body)] (cons cols body') body)]
      (csv-response body'))
    [(pr-str (:error body)) "text/csv"]))

;; http://www.w3.org/TR/sparql11-results-csv-tsv/
(defmethod api-response ["text/csv" :query]
  [_ body _]
  (let [qvars (if-let [sel (get-in body [:spec :select])]
                  (if (sequential? sel) sel [sel])
                  (into #{} (mapcat keys) (:body body)))
          qvars (vec (sort qvars))
          body  (into [(mapv ff/qvar-name qvars)]
                      (map (fn [res] (mapv #(sparql-csv-item res %) qvars)))
                      (:body body))]
      (csv-response body)))

(defmethod api-response ["application/sparql-results+csv" :query]
  [_ body _] (api-response "text/csv" body :query))

(defn parse-accept-header
  [accept]
  (->> accept
       (re-seq #"((([\w\*\-]+/[\w\*\-\+]+,?)+)(;q=(\d+(\.\d+)?))?)")
       (mapcat
        (fn [[_ _ a _ _ q]]
          (let [q (if q (strf/parse-float q 0) 1)]
            (map vector (str/split a #",") (repeat q)))))
       (sort-by peek)
       reverse))
(defn api-response*
  ([req body status]
   (api-response* req body :default status))
  ([req body rtype status]
   (let [accept             (get-in req [:headers "accept"])
         [resp type]        (->> accept
                                 (parse-accept-header)
                                 (some #(api-response (first %) body rtype)))
         [resp type status] (if resp
                              [resp type status]
                              [(str "Not acceptable: " accept) "text/plain" 406])]
     (-> (resp/response resp)
         (resp/status status)
         (resp/content-type type)))))

(def coercions
  {:edn (fn [x] (try (edn/read-string x) (catch Exception e x)))
   :int (fn [x] (strf/parse-int x 10 x))})

(def validations
  {:query   {:q [(v/sequential "must be a vector of sub-query maps") (v/min-length 1)]}
   :opt-int (v/optional (v/pos "must be positive integer"))})

(defn coerce-params
  [coercs params]
  (reduce-kv
   (fn [acc k v]
     (let [cfn (if (fn? v) v (coercions v))]
       (update acc k cfn)))
   params coercs))

(defn compile-validators
  [vspec]
  (reduce-kv
   (fn [acc k v]
     (if (keyword? v)
       (assoc acc k (validations v))
       acc))
   vspec vspec))

(defn validating-handler
  [req coerce vspec handler]
  (info "-------------------")
  (info :uri (:uri req) :params (:params req))
  ;; (info :cookies (:cookies req) :session (:session req))
  (let [params (:params req)
        params (if coerce
                 (coerce-params coerce params)
                 params)
        [params err] (if vspec
                       (v/validate params (compile-validators vspec))
                       [params])]
    (info :validated params err)
    (if (nil? err)
      (try
        (handler req params)
        (catch Exception e
          (.printStackTrace e)
          (api-response* req {:error "Error handling route"} 500)))
      (api-response* req {:error err} 400))))

(defn delayed-response-handler
  [model ready? handler]
  (d/->deferred
   (go
     (let [ch  (async/tap (result-event-bus model) (async/chan))]
       (loop []
         (let [res (<! ch)]
           (if (ready?)
             (do
               (async/untap (result-event-bus model) ch)
               (async/close! ch)
               (debug :result res)
               (handler))
             (recur))))))))
(defn list-registry-handler
  ([registry]
   (list-registry-handler registry ["id" "spec"] :spec))
  ([registry cols key-fn]
   (fn [req]
     (let [items (reduce-kv
                  (fn [acc k v] (assoc acc k (key-fn v)))
                  (sorted-map) (list-registry registry))]
       (api-response* req {:columns cols :body items} 200)))))
(defn add-facts-handler
  [model]
  (fn [req]
    (validating-handler
     req
     {:facts :edn}
     {:facts (v/optional (v/alts [(v/map) (v/sequential)]))
      :uri   (v/optional (v/url))}
     (fn [_ {:keys [facts uri] :as params}]
       (cond
         facts (let [facts (if (map? facts) (ff/map->facts facts) facts)]
                 (add-facts! model facts)
                 (api-response* req {:body (str "adding " (count facts) " facts")} 202))
         uri   (let [[facts err] (load-facts-from-uri uri)]
                 (if facts
                   (do (add-facts! model facts)
                       (api-response* req {:body (str "adding " (count facts) " facts from " uri)} 202))
                   (api-response* req {:error err} 400)))
         :else (api-response* req {:error "No facts or uri param given"} 400))))))

(defn list-facts-handler
  [model]
  (fn [req]
    (validating-handler
     req
     {:limit  :int
      :offset :int}
     {:limit  :opt-int
      :offset :opt-int}
     (fn [_ {:keys [limit offset]}]
       (let [limit  (or limit 100)
             offset (or offset 0)
             facts  (get-facts model)
             facts' (into [] (comp (drop offset) (take limit)) facts)]
         (api-response*
          req
          {:count (count facts')
           :total (count facts)
           :offset offset
           :columns ["subject" "predicate" "object"]
           :body facts'}
          200))))))
(defn one-off-query-handler
  [model]
  (fn [req]
    (validating-handler
     req
     {:spec   :edn
      :limit  :int
      :offset :int}
     {:spec   :query
      :limit  :opt-int
      :offset :opt-int}
     (fn [_ {:keys [spec limit offset] :as params}]
       (let [spec    (transform-query model spec)
             _       (info :one-off-query spec)
             q       (add-query! model spec)
             handler (delayed-response-handler
                      model
                      #(deref q)
                      #(let [limit  (or limit 100)
                             offset (or offset 0)
                             body   (into (empty @q) (comp (drop offset) (take limit)) @q)]
                         (remove-query! model q)
                         (api-response*
                          req
                          {:count  (count body)
                           :total  (count @q)
                           :offset offset
                           :spec   spec
                           :body   body}
                          (if (:group-by spec) :query-group :query)
                          200)))]
         ;;(trigger-update! model)
         handler)))))

(defn register-query-handler
  [model queries]
  (fn [req]
    (validating-handler
     req
     {:spec :edn}
     {:id   (v/required)
      :spec :query}
     (fn [_ {:keys [id spec] :as params}]
       (let [spec (transform-query model spec)
             id'  (register! queries id spec)]
         (if id'
           (if (= id id')
             (api-response* req {:id id' :body "New query registered"} 202)
             (api-response* req {:id id' :body "Query already registered using another ID"} 303))
           (api-response* req {:id id :error "Query ID conflict"} 409)))))))

(defn query-result-handler
  [model queries]
  (fn [req]
    (validating-handler
     req
     {:spec   :edn
      :limit  :int
      :offset :int}
     {:id     (v/required)
      :spec   {:q [(fn [_ v] (nil? v)) "spec must not have a :q key"]}
      :limit  :opt-int
      :offset :opt-int}
     (fn [_ {:keys [id spec limit offset] :as params}]
       (let [q (lookup queries id)]
         (if-let [qq (:query q)]
           (let [limit  (or limit 100)
                 offset (or offset 0)
                 res    (if spec
                          (let [spec (select-keys spec [:filter :aggregate :order :group-by :select :bind])
                                spec (transform-query model spec)
                                spec (merge (dissoc (:spec q) :q) spec)
                                cfn  (-> spec
                                         (dsl/compile-query-result-spec)
                                         (ff/make-query-result))]
                            (info "merged result using:" spec)
                            (cfn @(ff/pre-result-vertex qq)))
                          @qq)
                 body   (into (empty res) (comp (drop offset) (take limit)) res)]
             (api-response*
              req
              {:id id
               :count  (count body)
               :total  (count res)
               :offset offset
               :spec   (:spec q)
               :body   body}
              (if (:group-by spec) :query-group :query)
              200))
           (api-response* req {:id id} 404)))))))

(defn delete-query-handler
  [queries]
  (fn [req]
    (let [id (-> req :params :id)]
      (if (deregister! queries id)
        (api-response* req {:id id :body "Query scheduled for deletion"} 202)
        (api-response* req {:id id :error "Query not found"} 404)))))
(defn register-rule-handler
  [model rules]
  (fn [req]
    (validating-handler
     req
     {:q    :edn
      :prod :edn}
     {:id   (v/required)
      :q    :query
      :prod [(v/sequential) (v/min-length 1)]}
     (fn [_ {:keys [id q prod] :as params}]
       (let [spec {:q (transform-query model q) :prod (set prod)}
             id'   (register! rules id spec)]
         (if id'
           (if (= id id')
             (api-response* req {:id id' :body "New rule registered"} 202)
             (api-response* req {:id id' :body "Rule already registered using another ID"} 303))
           (api-response* req {:id id :error "Rule ID conflict"} 409)))))))

(defn rule-result-handler
  [rules]
  (fn [req]
    (validating-handler
     req
     {:limit  :int
      :offset :int}
     {:id     (v/required)
      :limit  :opt-int
      :offset :opt-int}
     (fn [_ {:keys [id limit offset] :as params}]
       (let [r (lookup rules id)]
         (if-let [rr (:rule r)]
           (let [limit  (or limit 100)
                 offset (or offset 0)
                 body   (into (empty @rr) (comp (drop offset) (take limit)) @rr)]
             (api-response*
              req
              {:id id
               :count  (count body)
               :total  (count @rr)
               :offset offset
               :spec   (:spec r)
               :body   body}
              :query
              200))
           (api-response* req {:id id :error "Rule not found"} 404)))))))

(defn delete-rule-handler
  [rules]
  (fn [req]
    (let [id (-> req :params :id)]
      (if (deregister! rules id)
        (api-response* req {:id id :body "Rule scheduled for deletion"} 202)
        (api-response* req {:id id :error "Rule not found"} 404)))))
(defn wrap-middleware
  [config routes]
  (let [defaults (assoc-in site-defaults [:security :anti-forgery] false)
        routes   (-> routes
                     (wrap-defaults defaults))
        routes   (if (:dev config)
                   (wrap-stacktrace routes)
                   routes)]
    routes))

(defn app-routes
  [config model prefixes queries rules]
  (wrap-middleware
   config
   (compojure/routes
    (GET    "/facts"       [] (list-facts-handler model))
    (POST   "/facts"       [] (add-facts-handler model))
    (POST   "/query"       [] (one-off-query-handler model))
    (POST   "/queries"     [] (register-query-handler model queries))
    (GET    "/queries"     [] (list-registry-handler queries))
    (GET    "/queries/:id" [] (query-result-handler model queries))
    (DELETE "/queries/:id" [] (delete-query-handler queries))
    (GET    "/rules"       [] (list-registry-handler rules))
    (POST   "/rules"       [] (register-rule-handler model rules))
    (GET    "/rules/:id"   [] (rule-result-handler rules))
    (DELETE "/rules/:id"   [] (delete-rule-handler rules))
    (GET    "/prefixes"    [] (list-registry-handler prefixes ["prefix" "uri"] identity))
    (route/not-found "404"))))

(defrecord Handler [config model prefixes queries rules routes]
  comp/Lifecycle
  (start [_]
    (info "starting handler...")
    (let [routes ((:routes config) config model prefixes queries rules)]
      (assoc _ :routes routes)))
  (stop [_]
    (info "stopping handler..."))
  IHandler
  (route-map [_]
    routes))

(defn make-handler
  [config] (map->Handler {:config config}))

(def inf-rules-props
  '{:prp-dom   {:q    {:q [{:where [[?p "rdfs:domain" ?d]
                                    [?x ?p ?y]]}]}
                :prod [[:add [?x "rdf:type" ?d]]]}

    :prp-rng   {:q    {:q [{:where [[?p "rdfs:range" ?r]
                                    [?x ?p ?y]]}]}
                :prod [[:add [?y "rdf:type" ?r]]]}

    :prp-fp    {:q    {:q [{:where [[?p "rdf:type" "owl:FunctionalProperty"]
                                    [?x ?p ?y1]
                                    [?x ?p ?y2]]}]}
                :prod [[:add [?y1 "owl:sameAs" ?y2]]]}

    :prp-ifp   {:q    {:q [{:where [[?p "rdf:type" "owl:InverseFunctionalProperty"]
                                    [?x1 ?p ?y]
                                    [?x2 ?p ?y]]}]}
                :prod [[:add [?x1 "owl:sameAs" ?x2]]]}

    :prp-irp   {:q    {:q [{:where [[?p "rdf:type" "owl:IrreflexiveProperty"]
                                    [?x ?p ?x]]}]}
                :prod [[:remove [?x ?p ?x]]]}

    :prp-symp  {:q    {:q [{:where [[?p "rdf:type" "owl:SymmetricProperty"]
                                    [?x ?p ?y]]}]}
                :prod [[:add [?y ?p ?x]]]}

    ;; :prp-asyp  {:q {:q [{:where [[?p "rdf:type" "owl:AsymmetricProperty"]
    ;;                               [?x ?p ?y]
    ;;                               [?y ?p ?x]]}]}
    ;;              :prod false}

    :prp-trp   {:q    {:q [{:where [[?p "rdf:type" "owl:TransitiveProperty"]
                                    [?x ?p ?y]
                                    [?y ?p ?z]]
                            :unique true}]}
                :prod [[:add [?x ?p ?z]]]}

    :prp-spo1  {:q    {:q [{:where [[?p1 "rdfs:subPropertyOf" ?p2]
                                    [?x ?p1 ?y]]}]}
                :prod [[:add [?x ?p2 ?y]]]}

    ;; :prp-spo2  nil ;; TODO requires property chains

    :prp-eqp1  {:q    {:q [{:where [[?p1 "owl:equivalentProperty" ?p2]
                                    [?x ?p1 ?y]]}]}
                :prod [[:add [?x ?p2 ?y]]]}

    :prp-eqp2  {:q    {:q [{:where [[?p1 "owl:equivalentProperty" ?p2]
                                    [?x ?p2 ?y]]}]}
                :prod [[:add [?x ?p1 ?y]]]}

    ;; :prp-pdw   {:q {:q [{:where [[?p1 "owl:propertyDisjointWith" ?p2]
    ;;                               [?x ?p1 ?y]
    ;;                               [?x ?p2 ?y]]}]}
    ;;              :prod false}

    ;; :prp-adp   nil ;; TODO required property chains

    :prp-inv1  {:q    {:q [{:where [[?p1 "owl:inverseOf" ?p2]
                                    [?x ?p1 ?y]]}]}
                :prod [[:add [?y ?p2 ?x]]]}

    :prp-inv2  {:q    {:q [{:where [[?p1 "owl:inverseOf" ?p2]
                                    [?x ?p2 ?y]]}]}
                :prod [[:add [?y ?p1 ?x]]]}})
(def inf-rules-class-semantics
  '{:cax-sco  {:q    {:q [{:where [[?c1 "rdfs:subClassOf" ?c2]
                                   [?x "rdf:type" ?c1]]}]}
               :prod [[:add [?x "rdf:type" ?c2]]]}

    :cax-eqc1 {:q    {:q [{:where [[?c1 "owl:equivalentClass" ?c2]
                                   [?x "rdf:type" ?c1]]}]}
               :prod [[:add [?x "rdf:type" ?c2]]]}

    :cax-eqc2 {:q    {:q [{:where [[?c1 "owl:equivalentClass" ?c2]
                                   [?x "rdf:type" ?c2]]}]}
               :prod [[:add [?x "rdf:type" ?c1]]]}

    ;; :cax-dw   {:q {:q [{:where [[?c1 "owl:disjointWith" ?c2]
    ;;                              [?x "rdf:type" ?c1]
    ;;                              [?x "rdf:type" ?c2]]}]}
    ;;             :prod false}

    ;; :cax-adc  nil ;; TODO
    })
(def inf-rules-schema
  '{:scm-cls  {:q    {:q [{:where [[?c "rdf:type" "owl:Class"]]}]}
               :prod [[:add [?c "rdfs:subClassOf" ?c]]
                      [:add [?c "owl:equivalentClass" ?c]]
                      [:add [?c "rdfs:subClassOf" "owl:Thing"]]
                      [:add ["owl:Nothing" "rdfs:subClassOf" ?c]]]}

    :scm-sco  {:q    {:q [{:where [[?c1 "rdfs:subClassOf" ?c2]
                                   [?c2 "rdfs:subClassOf" ?c3]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c3]]]}

    :scm-eqc1 {:q    {:q [{:where [[?c1 "owl:equivalentClass" ?c2]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c2]]
                      [:add [?c2 "rdfs:subClassOf" ?c1]]]}

    :scm-eqc2 {:q    {:q [{:where [[?c1 "rdfs:subClassOf" ?c2]
                                   [?c2 "rdfs:subClassOf" ?c1]]}]}
               :prod [[:add [?c1 "owl:equivalentClass" ?c2]]]}

    :scm-op   {:q    {:q [{:where [[?p "rdf:type" "owl:ObjectProperty"]]}]}
               :prod [[:add [?p "rdfs:subPropertyOf" ?p]]
                      [:add [?p "owl:equivalentProperty" ?p]]]}

    :scm-dp   {:q    {:q [{:where [[?p "rdf:type" "owl:DatatypeProperty"]]}]}
               :prod [[:add [?p "rdfs:subPropertyOf" ?p]]
                      [:add [?p "owl:equivalentProperty" ?p]]]}

    :scm-spo  {:q    {:q [{:where [[?p1 "rdfs:subPropertyOf" ?p2]
                                   [?p2 "rdfs:subPropertyOf" ?p3]]}]}
               :prod [[:add [?p1 "rdfs:subPropertyOf" ?p3]]]}

    :scm-eqp1 {:q    {:q [{:where [[?p1 "owl:equivalentProperty" ?p2]]}]}
               :prod [[:add [?p1 "rdfs:subPropertyOf" ?p2]]
                      [:add [?p2 "rdfs:subPropertyOf" ?p1]]]}

    :scm-eqp2 {:q    {:q [{:where [[?p1 "rdfs:subPropertyOf" ?p2]
                                   [?p2 "rdfs:subPropertyOf" ?p1]]}]}
               :prod [[:add [?p1 "owl:equivalentProperty" ?p2]]]}

    :scm-dom1 {:q    {:q [{:where [[?p "rdfs:domain" ?c1]
                                   [?c1 "rdfs:subClassOf" ?c2]]}]}
               :prod [[:add [?p "rdfs:domain" ?c2]]]}

    :scm-dom2 {:q    {:q [{:where [[?p2 "rdfs:domain" ?c]
                                   [?p1 "rdfs:subPropertyOf" ?p2]]}]}
               :prod [[:add [?p1 "rdfs:domain" ?c]]]}

    :scm-rng1 {:q    {:q [{:where [[?p "rdfs:range" ?c1]
                                   [?c1 "rdfs:subClassOf" ?c2]]}]}
               :prod [[:add [?p "rdfs:range" ?c2]]]}

    :scm-rng2 {:q    {:q [{:where [[?p2 "rdfs:range" ?c]
                                   [?p1 "rdfs:subPropertyOf" ?p2]]}]}
               :prod [[:add [?p1 "rdfs:range" ?c]]]}

    :scm-hv   {:q    {:q [{:where [[?c1 "owl:hasValue" ?i]
                                   [?c1 "owl:onProperty" ?p1]
                                   [?c2 "owl:hasValue" ?i]
                                   [?c2 "owl:onProperty" ?p2]
                                   [?p1 "rdfs:subPropertyOf" ?p2]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c2]]]}

    :scm-svf1 {:q    {:q [{:where [[?c1 "owl:someValuesFrom" ?y1]
                                   [?c1 "owl:onProperty" ?p]
                                   [?c2 "owl:someValuesFrom" ?y2]
                                   [?c2 "owl:onProperty" ?p]
                                   [?y1 "rdfs:subClassOf" ?y2]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c2]]]}

    :scm-svf2 {:q    {:q [{:where [[?c1 "owl:someValuesFrom" ?y]
                                   [?c1 "owl:onProperty" ?p1]
                                   [?c2 "owl:someValuesFrom" ?y]
                                   [?c2 "owl:onProperty" ?p2]
                                   [?p1 "rdfs:subPropertyOf" ?p2]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c2]]]}

    :scm-avf1 {:q    {:q [{:where [[?c1 "owl:allValuesFrom" ?y1]
                                   [?c1 "owl:onProperty" ?p]
                                   [?c2 "owl:allValuesFrom" ?y2]
                                   [?c2 "owl:onProperty" ?p]
                                   [?y1 "rdfs:subClassOf" ?y2]]}]}
               :prod [[:add [?c1 "rdfs:subClassOf" ?c2]]]}

    :scm-avf2 {:q    {:q [{:where [[?c1 "owl:allValuesFrom" ?y]
                                   [?c1 "owl:onProperty" ?p1]
                                   [?c2 "owl:allValuesFrom" ?y]
                                   [?c2 "owl:onProperty" ?p2]
                                   [?p1 "rdfs:subPropertyOf" ?p2]]}]}
               :prod [[:add [?c2 "rdfs:subClassOf" ?c1]]]}

    ;; :scm-int  nil ;; TODO

    ;; :scm-uni  nil ;; TODO
    })

(defn default-config
  []
  {:prefixes {:prefixes {"dcterms" "http://purl.org/dc/terms/"
                         "doap"    "http://usefulinc.com/ns/doap#"
                         "fabric"  "http://ns.thi.ng/fabric#"
                         "foaf"    "http://xmlns.com/foaf/0.1/"
                         "geo"     "http://www.w3.org/2003/01/geo/wgs84_pos#"
                         "owl"     "http://www.w3.org/2002/07/owl#"
                         "rdf"     "http://www.w3.org/1999/02/22-rdf-syntax-ns#"
                         "rdfs"    "http://www.w3.org/2000/01/rdf-schema#"
                         "schema"  "http://schema.org/"
                         "xsd"     "http://www.w3.org/2001/XMLSchema#"}}
   :log      {:fn       default-logger
              :path     ["session" #(strf/format-date :yyyymmdd-hhmmss %) ".edn"]}
   :graph    {:import   [(io/resource "fabric.edn")]
              ;;:restore  "session-20150910-222204.edn"
              :index    (ff/alias-index-vertex #{"owl:sameAs"})}
   :model    {}
   :queries  {:specs    {"types" '{:q [{:where [[?s "rdf:type" ?type]]}]}}}
   :rules    {:specs    (merge
                         inf-rules-props
                         inf-rules-class-semantics
                         inf-rules-schema)}
   :handler  {:routes   app-routes
              :dev      true}
   :server   {:port     8000}})

(defn make-system
  [config]
  (comp/system-map
   :graph    (comp/using
              (make-graph (:graph config))
              {:prefixes :prefixes})
   :log      (comp/using
              (make-fact-log (:log config))
              {:graph    :graph
               :prefixes :prefixes})
   :model    (comp/using
              (make-graph-model (:model config))
              {:graph    :graph
               :prefixes :prefixes})
   :prefixes (make-prefix-registry (:prefixes config))
   :queries  (comp/using
              (make-query-registry (:queries config))
              {:model :model})
   :rules    (comp/using
              (make-rule-registry (:rules config))
              {:model :model})
   :handler  (comp/using
              (make-handler (:handler config))
              {:model    :model
               :prefixes :prefixes
               :queries  :queries
               :rules    :rules})
   :server   (comp/using
              (make-server (:server config))
              {:handler :handler})))

(def system nil)

(defn init []
  (alter-var-root
   #'system
   (constantly (make-system (default-config)))))

(defn start []
  (alter-var-root #'system comp/start))

(defn stop []
  (alter-var-root
   #'system
   (fn [s] (when s (comp/stop s)))))

(defn launch []
  (init)
  (start))

(defn reset []
  (stop)
  (refresh :after 'thi.ng.fabric.ld.core/launch))
