(ns thi.ng.geom.mesh.io
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.core.utils :as gu]
   [thi.ng.geom.core.vector :as v :refer [vec3]]
   [thi.ng.geom.basicmesh :as bm]
   [thi.ng.geom.triangle :as t]
   [thi.ng.dstruct.streams :as streams]
   [thi.ng.strf.core :as f]
   [clojure.string :as str])
  #?(:clj
     (:import
      [java.io OutputStream InputStream])))

(def ^:dynamic *precision* 5)

(def wrapped-output-stream
  "Alias for thi.ng.dstruct.streams/output-stream"
  streams/output-stream)

(def wrapped-input-stream
  "Alias for thi.ng.dstruct.streams/input-stream"
  streams/input-stream)

(defn write-stl
  "Writes the given mesh as binary STL to output stream wrapper. Mesh
  is automatically tessellated into triangles. Returns stream wrapper"
  [out mesh]
  (let [faces    (g/faces mesh)
        fnormals (g/face-normals mesh true)]
    (streams/skip out 80)
    (streams/write-uint32-le out (count faces))
    (doseq [[a b c :as f] faces]
      (streams/write-vec3f-le out (or (get fnormals f) (gu/ortho-normal f)))
      (streams/write-vec3f-le out a)
      (streams/write-vec3f-le out b)
      (streams/write-vec3f-le out c)
      (streams/write-uint16-le out 0))
    out))

(defn read-stl
  "Reads faces from binary STL input stream wrapper and adds them to
  given (or new) mesh."
  ([in]
   (read-stl in (bm/basic-mesh)))
  ([in mesh]
   (streams/skip in 80)
   (loop [mesh mesh, i (streams/read-uint32-le in)]
     (if (pos? i)
       (let [_ (streams/read-vec3f-le in)
             a (vec3 (streams/read-vec3f-le in))
             b (vec3 (streams/read-vec3f-le in))
             c (vec3 (streams/read-vec3f-le in))]
         (streams/skip in 2)
         (recur (g/add-face mesh [a b c]) (dec i)))
       mesh))))

(defn write-ply
  "Writes the given mesh as binary PLY to output stream wrapper.
  For compatibility with some external tools, the mesh should already have been
  tessellated before calling this fn. Returns stream wrapper."
  [out mesh]
  (let [vertices    (g/vertices mesh)
        vindex      (zipmap vertices (range))
        vnormals    (g/vertex-normals mesh false)
        faces       (g/faces mesh)
        vnorms?     (not (nil? (seq vnormals)))
        write-props (fn [props]
                      (doseq [p props]
                        (streams/write-utf8-bytes
                         out (str "property float32 " p "\n"))))]
    (-> out
        (streams/write-utf8-bytes "ply\n")
        (streams/write-utf8-bytes "format binary_little_endian 1.0\n")
        (streams/write-utf8-bytes (str "element vertex " (count vertices) "\n")))
    (write-props ['x 'y 'z])
    (when vnorms?
      (write-props ['nx 'ny 'nz]))
    (-> out
        (streams/write-utf8-bytes (str "element face " (count faces) "\n"))
        (streams/write-utf8-bytes "property list uint8 uint32 vertex_indices\n")
        (streams/write-utf8-bytes "end_header\n"))
    (if vnorms?
      (doseq [v vertices]
        (streams/write-vec3f-le out v)
        (streams/write-vec3f-le out (get vnormals v)))
      (doseq [v vertices]
        (streams/write-vec3f-le out v)))
    (doseq [f (g/faces mesh)]
      (streams/write-uint8 out (unchecked-byte (count f)))
      (doseq [v f]
        (streams/write-uint32-le out (get vindex v))))
    out))

(defn write-obj
  "Writes mesh as Waveform OBJ format to output stream wrapper.
  Returns stream wrapper."
  [out mesh]
  (let [vertices    (g/vertices mesh)
        vindex      (zipmap vertices (range))
        vnormals    (g/vertex-normals mesh false)
        vnorms?     (not (nil? (seq vnormals)))
        nindex      (zipmap vnormals (range))
        faces       (g/faces mesh)
        ff          (f/float *precision*)
        fmt-fn      (fn [prefix]
                      (let [fmt [prefix " " ff " " ff " " ff "\n"]]
                        #(f/format fmt (double %1) (double %2) (double %3))))
        fmt-vertex  (fmt-fn "v")
        fmt-vnormal (fmt-fn "vn")
        fmt-face    (if vnorms?
                      (fn [verts normals]
                        (str "f " (str/join " " (map #(str % "//" %2) verts normals)) "\n"))
                      (fn [verts]
                        (str "f " (str/join " " verts) "\n")))]
    (doseq [[x y z] vertices]
      (streams/write-utf8-bytes
       out (fmt-vertex x y z)))
    (doseq [[x y z] (vals vnormals)]
      (streams/write-utf8-bytes out (fmt-vnormal x y z)))
    (streams/write-utf8-bytes out "g\n")
    (if vnorms?
      (doseq [fverts faces]
        (streams/write-utf8-bytes
         out (fmt-face
              (map #(inc (get vindex %)) fverts)
              (map #(inc (get nindex (get vnormals %))) fverts))))
      (doseq [fverts faces]
        (streams/write-utf8-bytes
         out (fmt-face (map #(inc (get vindex %)) fverts)))))
    out))

(defn write-off
  "Writes mesh as OFF format to output stream wrapper.
  Returns stream wrapper."
  [out mesh]
  (let [vertices   (g/vertices mesh)
        faces      (g/faces mesh)
        vindex     (zipmap vertices (range))
        fmt-float  (f/float *precision*)
        fmt-vertex [fmt-float " " fmt-float " " fmt-float "\n"]]
    (streams/write-utf8-bytes out "OFF\n")
    (streams/write-utf8-bytes out (str (count vertices) " " (count faces) " 0\n"))
    (doseq [[x y z] vertices]
      (streams/write-utf8-bytes
       out (f/format fmt-vertex (double x) (double y) (double z))))
    (doseq [fverts faces]
      (streams/write-utf8-bytes
       out
       (str (count fverts) " "
            (str/join " " (map vindex fverts))
            "\n")))
    out))
