(ns thi.ng.geom.mesh.slicing
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.core.utils :as gu]
   [thi.ng.geom.core.vector :as v :refer [vec3]]
   [thi.ng.geom.core.matrix :refer [M44]]
   [thi.ng.geom.gmesh :as gm]
   [thi.ng.geom.basicmesh :as bm]
   [thi.ng.geom.aabb :as a]
   [thi.ng.geom.circle :as c]
   [thi.ng.geom.rect :as r]
   [thi.ng.geom.polygon :as p]
   [thi.ng.geom.plane :as pl]
   [thi.ng.geom.triangle :as t]
   [thi.ng.geom.sphere :as s]
   [thi.ng.geom.mesh.io :as mio]
   [thi.ng.geom.mesh.ops :as ops]
   [thi.ng.geom.types.utils :as tu]
   [clojure.java.io :as io]
   [thi.ng.common.math.core :as m]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.data.intervaltree :as it]
   [thi.ng.macromath.core :as mm])
  (:import
   (java.awt.image BufferedImage)
   (java.awt Graphics2D RenderingHints Color Font GradientPaint)
   (java.awt.geom Path2D$Double Rectangle2D$Double)
   (javax.imageio ImageIO)))

(defn z-plane
  [z]
  (pl/plane-with-point (vec3 0 0 z) v/V3Z))

(defn z-slice-tree
  [tree z eps]
  (let [plane (z-plane z)
        tris (->> (it/query-interval tree [(- z eps) (+ z eps)] [])
                  (mapcat
                   (fn [t]
                     (let [t' (get (t/slice-with* t plane) 1)
                           tf (filter #(m/delta= z (:z %)) t)
                           c (count tf)]
                       ;;(prn t t' tf)
                       (or t' (if (m/in-range? 1 2 c) [t]))))))
        ;;_ (prn tris)
        mesh (g/into (gm/gmesh) tris)]
    (if (seq (g/faces mesh))
      (first (ops/canonicalize-vertices mesh 1e-3))
      mesh)))

(defn sorted-faces
  [mesh]
  (->> mesh
       (g/faces)
       (map #(:z (gu/centroid %)))
       (sort)))

(defn z-vertices
  [mesh z]
  (into (sorted-set) (filter #(m/delta= z (:z %)) (g/vertices mesh))))

(defn z-neighbors
  [mesh v z]
  (filter #(m/delta= z (:z %)) (d/value-set :next (:vertices mesh) v)))

(defn z-edges
  [mesh z]
  (filter #(let [[a b] (seq %)] (and (m/delta= z (:z a)) (m/delta= z (:z b)))) (g/edges mesh)))

(defn vertex-loop
  [m v z]
  ;;(prn "----")
  (loop [path [v]]
    (let [v (peek path)
          nf (z-neighbors m v z)
          nf2 (first (filter #(neg? (.indexOf path %)) nf))]
      ;;(prn v nf)
      ;;(prn (:xy v) :-> (:xy nf2))
      (if (and nf2 (not (m/delta= v nf2)))
        (recur (conj path nf2))
        path))))

(defn find-loops2
  [m z]
  (let [zverts (z-vertices m z)]
    (loop [paths [] vertices zverts]
      (if (seq vertices)
        (let [v (first vertices)
              p (vertex-loop m v z)]
          ;;(prn (map :xy p))
          (recur (conj paths p) (reduce disj (set vertices) p)))
        paths))))


(defn mesh->intervaltree
  [slice-dir mesh]
  (let [axis ({:x 0 :y 1 :z 2} slice-dir)
        faces (map #(vector (gu/axis-bounds axis %) %) (g/faces mesh))
        median (/ (reduce (fn [acc [b]] (+ acc (mm/addm (b 0) (b 1) 0.5))) 0 faces)
                  (count faces))]
    (it/interval-tree median faces)))

(defn save-mesh
  [m] (with-open [o (io/output-stream "p.ply")] (mio/write-ply o m)))


(defn new-image
  "Creates a new transparent BufferedImage and Graphics2D instance.
      Graphics are configured for anti-aliasing."
  [w h]
  (let[img (BufferedImage. w h BufferedImage/TYPE_INT_ARGB)
       g2d (.createGraphics img)]
    (doto g2d
      (.setRenderingHint RenderingHints/KEY_ANTIALIASING RenderingHints/VALUE_ANTIALIAS_ON)
      (.setRenderingHint RenderingHints/KEY_INTERPOLATION RenderingHints/VALUE_INTERPOLATION_BICUBIC)
      (.setRenderingHint RenderingHints/KEY_RENDERING RenderingHints/VALUE_RENDER_QUALITY))
    {:img img :g2d g2d}))

(defn clear-image
  "Takes an image spec returned by new-image and fills the image with transparent white."
  [{:keys[img g2d]}]
  (doto g2d
    (.setPaint Color/RED)
    (.fillRect 0 0 (.getWidth img) (.getHeight img))))

(defn save-image
  "Takes an image spec and saves it as PNG to the given path."
  [imgspec path]
  (ImageIO/write (:img imgspec) "PNG" (io/file path)))

(defn draw-poly
  [img points]
  (let [ccw? (neg? (g/area (p/polygon2 points)))
        [p & more :as points] (if ccw? (reverse points) points)
        path (Path2D$Double.)]
    ;;(prn :np (count points))
    (.moveTo path (:x p) (:y p))
    (doseq [p more]
      (.lineTo path (:x p) (:y p)))
    (.closePath path)
    (doto (:g2d img)
      (.setPaint (if ccw? Color/BLACK Color/WHITE))
      (.fill path)
      (.setPaint Color/GREEN))
    (doseq [p points]
      (.draw (:g2d img) (Rectangle2D$Double. (dec (:x p)) (dec (:y p)) 3.0 3.0)))))

(defn draw-slice-paths
  [paths z]
  (let [img (new-image 500 500)]
    (clear-image img)
    (doseq [p paths]
      ;;(prn p)
      (draw-poly img (map #(g/madd (:xy %) 4 [10 10]) p)))
    (save-image img (format "z-%3.3f.png" (double z)))))

(comment
  (def box
    (-> ;;(r/rect 100)
     ;;(c/circle [50 50] 50)
     ;;(g/as-polygon)
     (p/cog 50 10 [0.9 1 1 1 0.9])
     (g/center [50 50])
     (g/extrude-shell
      {:depth 100 :wall 2 :inset 5 :top? false :bottom? true :mesh (bm/basic-mesh)})
     (g/tessellate)))

  (def box
    (-> (a/aabb 100)
        (g/as-mesh)
        (g/tessellate)))

  (def box
    (let [;;path "canopy-flat-12.stl"
          path "hexsphere.stl"]
      (with-open [i (io/input-stream path)]
        (->> (mio/read-stl i)
             (vector)
             (tu/fit-all-into-bounds (a/aabb 100))
             (first)
             (#(ops/canonicalize-vertices % 1e-3))
             (first)))))

  (def sphere (-> (s/sphere 50)
                  (g/as-mesh {:res 15})
                  (g/transform (-> M44 (g/translate 50 50 50) (g/rotate-x m/HALF_PI)))
                  (g/tessellate)))

  (let [t (mesh->intervaltree :z box)
        [a b] (gu/axis-bounds 2 (seq (g/vertices box)))]
    (prn a :-> b)
    (doseq [z (range a b 0.5)]
      (prn z)
      (-> (z-slice-tree t z 0.01)
          (find-loops2 z)
          (draw-slice-paths z))))
  )
