(ns thi.ng.geom.webgl.shaders.image
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.vector :refer [vec2]]
   [thi.ng.geom.matrix :refer [M44]]
   [thi.ng.geom.rect :as r]
   [thi.ng.geom.webgl.core :as gl]
   [thi.ng.geom.webgl.constants :as glc]
   [thi.ng.geom.webgl.shaders :as sh]
   [thi.ng.typedarrays.core :as ta]
   [thi.ng.dstruct.core :as d]
   [thi.ng.dstruct.streams :as streams]
   [thi.ng.glsl.core :as glsl :include-macros true]
   [thi.ng.math.core :as m]))

(def shader-spec
  {:vs (glsl/minified "
void main() {
  vUV = uv;
  gl_Position = proj * model * vec4(position, 0.0, 1.0);
}")
   :fs (glsl/minified "
void main() {
  gl_FragColor = texture2D(tex, vUV);
}")
   :uniforms {:proj       :mat4
              :model      [:mat4 M44]
              :tex        [:sampler2D 0]}
   :attribs  {:position   :vec2
              :uv         :vec2}
   :varying  {:vUV        :vec2}
   :state    {:depth-test false
              :blend      true
              :blend-func [glc/src-alpha glc/one-minus-src-alpha]}})

(defn make-shader-spec
  ([^WebGLRenderingContext gl]
   (make-shader-spec gl nil))
  ([^WebGLRenderingContext gl opts]
   (let [r         (or (get opts :rect) (r/rect 1))
         [a b c d] (g/vertices r)
         verts     (ta/float32 8)]
     (streams/into-buffer [a b d c] verts 2 0)
     {:attribs      (gl/make-attribute-buffers
                     gl glc/static-draw
                     {:position {:data verts :size 2}
                      :uv       {:data (ta/float32 [0 0, 1 0, 0 1, 1 1]) :size 2}})
      :uniforms     {:tex  0
                     :proj (gl/ortho)}
      :shader       (d/merge-deep
                     (or (get opts :shader) (sh/make-shader-from-spec gl shader-spec))
                     {:state (get opts :state)})
      :view-port    (get opts :viewport)
      :pos          (get opts :pos (vec2))
      :width        (get opts :width 128)
      :height       (get opts :height 128)
      :mode         glc/triangle-strip
      :num-vertices 4})))

(defn draw
  [^WebGLRenderingContext gl {:keys [viewport pos width height] :as spec}]
  (let [[vw vh] (:size (or viewport (gl/get-viewport-rect gl)))
        x       (m/map-interval (nth pos 0) 0 vw -1 1)
        y       (m/map-interval (nth pos 1) 0 vh -1 1)
        s       (vec2 (* 2.0 (/ width vw)) (* 2.0 (/ height vh)))
        spec    (assoc-in spec [:uniforms :model] (-> M44 (g/translate x y 0) (g/scale s)))]
    (gl/draw-with-shader gl spec)))
