(ns thi.ng.geom.plane
  #?(:cljs (:require-macros [thi.ng.math.macros :as mm]))
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.utils :as gu]
   [thi.ng.geom.utils.intersect :as isec]
   [thi.ng.geom.vector :as v :refer [vec3]]
   [thi.ng.geom.quaternion :as q]
   [thi.ng.geom.basicmesh :as bm]
   [thi.ng.geom.attribs :as attr]
   [thi.ng.geom.types :as types]
   [thi.ng.xerror.core :as err]
   [thi.ng.math.core :as m :refer [*eps* INF+]]
   #?(:clj [thi.ng.math.macros :as mm])))

(defn plane
  [n w] (thi.ng.geom.types.Plane. (m/normalize (vec3 n)) w))

(defn plane-with-point
  [p n]
  (let [n (m/normalize (vec3 n))]
    (thi.ng.geom.types.Plane. n (- (m/dot n p)))))

(defn plane-from-points
  ([[a b c]] (plane-from-points a b c))
  ([a b c]
     (let [n (gu/ortho-normal a b c)]
       (thi.ng.geom.types.Plane. n (- (m/dot n a))))))




(extend-type thi.ng.geom.types.Plane
g/IArea
(area [_] INF+)
g/IBounds
(bounds
 [_]
 (let [s (vec3 (g/width _) (g/height _) (g/depth _))]
   (thi.ng.geom.types.AABB. (m/madd s -0.5 (g/centroid _)) s)))
(width
 [_] (if (m/delta= (m/abs (:n _)) v/V3X *eps*) 0.0 INF+))
(height
 [_] (if (m/delta= (m/abs (:n _)) v/V3Y *eps*) 0.0 INF+))
(depth
 [_] (if (m/delta= (m/abs (:n _)) v/V3Z *eps*) 0.0 INF+))
g/IBoundingSphere
(bounding-sphere
  [_] (thi.ng.geom.types.Sphere. (g/centroid _) INF+))
g/ICenter
(center
  ([_] (thi.ng.geom.types.Plane. (:n _) 0))
  ([_ o] (plane-with-point o (:n _))))
(centroid
  ([_] (m/* (:n _) (- (:w _)))))
g/IClassify
(classify-point
  [_ p]
  (-> (:n _) (m/dot p) (+ (:w _)) (m/signum *eps*)))
g/IDistance
(dist
 [_ p] (+ (m/dot (:n _) p) (:w _)))
(dist-squared
 [_ p] (let [d (g/dist _ p)] (* d d)))
g/IFlip
(flip
  [_] (thi.ng.geom.types.Plane. (m/- (:n _)) (- (:w _))))
g/IIntersect
(intersect-line
 ([_ l]
    (let [[p q] (if (map? l) (:points l) l)]
      (isec/intersect-ray-plane? p (m/- q p) (:n _) (:w _))))
 ([_ p q]
    (isec/intersect-ray-plane? p (m/- q p) (:n _) (:w _))))
(intersect-ray
 ([_ ray]
    (let [[p dir] (if (map? ray) [(:p ray) (:dir ray)] ray)]
      (isec/intersect-ray-plane? p dir (:n _) (:w _))))
 ([_ p dir]
    (isec/intersect-ray-plane? p dir (:n _) (:w _))))
(intersect-shape
 [_ s]
 (cond
  (instance? thi.ng.geom.types.Plane s)
  (isec/intersect-plane-plane? (:n _) (:w _) (:n s) (:w s))
  (instance? thi.ng.geom.types.Sphere s)
  (isec/intersect-plane-sphere? (:n _) (:w _) (:p s) (:r s))
  :default (err/illegal-arg! s)))
g/IMeshConvert
(as-mesh
 ([_] (g/as-mesh _ {}))
 ([_ {:keys [p width height size mesh attribs] :or {size 1.0}}]
  (let [w     (* (or width size) 0.5)
        h     (* (or height size) 0.5)
        flip? (m/delta= -1.0 (m/dot (:n _) v/V3Z))
        p     (g/closest-point _ (or p (vec3)))
        q     (if flip?
                (q/quat 0 0 0 1)
                (q/alignment-quat v/V3Z (:n _)))
        face  (mapv
               #(m/+ p (g/transform-vector q %))
               [(vec3 (- w) (- h) 0) (vec3 (- w) h 0)
                (vec3 w h 0) (vec3 w (- h) 0)])
        face  (attr/generate-face-attribs (if flip? face (rseq face)) 0 attribs nil)]
    (g/add-face (or mesh (bm/basic-mesh)) face))))
g/IProximity
(closest-point
  [_ p]
  (->> p
       (m/dot (:n _))
       (+ (:w _))
       (m/normalize (:n _))
       (m/- p)))
g/IScale
(scale
 ([_ s] (plane-with-point (m/* (g/centroid _) s) (:n _)))
 ([_ a b] (plane-with-point (m/* (g/centroid _) a b) (:n _)))
 ([_ a b c] (plane-with-point (m/* (g/centroid _) a b c) (:n _))))
(scale-size
 ([_ s] _))
g/ITranslate
(translate
 [_ t] (plane-with-point (m/+ (g/centroid _) t) (:n _)))
g/ITransform
(transform
 [_ m] (plane-with-point
        (g/transform-vector m (g/centroid _))
        (g/transform-vector m (:n _))))
g/IVolume
(volume [_] 0.0)
)
