(ns thi.ng.geom.sphere
  #?(:cljs (:require-macros [thi.ng.math.macros :as mm]))
  (:require
   [thi.ng.geom.core :as g :refer [*resolution*]]
   [thi.ng.geom.utils :as gu]
   [thi.ng.geom.vector :as v :refer [vec3]]
   [thi.ng.geom.utils.intersect :as isec]
   [thi.ng.geom.basicmesh :as bm]
   [thi.ng.geom.attribs :as attr]
   [thi.ng.geom.types :as types]
   [thi.ng.xerror.core :as err]
   [thi.ng.math.core :as m :refer [TWO_PI PI *eps*]]
   #?(:clj [thi.ng.math.macros :as mm]))
  #?(:clj
     (:import
      [thi.ng.geom.types AABB Sphere])))

(defn sphere
  ([] (thi.ng.geom.types.Sphere. (vec3) 1.0))
  ([r] (thi.ng.geom.types.Sphere. (vec3) #?(:clj (double r) :cljs r)))
  ([p r] (thi.ng.geom.types.Sphere. (vec3 p) #?(:clj (double r) :cljs r))))




(extend-type thi.ng.geom.types.Sphere
g/IArea
(area
 [{r :r}] (* 4.0 PI r r))
g/IBoundary
(contains-point?
 [{p :p r :r} q] (<= (g/dist-squared p q) (* r r)))
g/IBounds
(bounds
 [_] (thi.ng.geom.types.AABB. (m/- (:p _) (:r _)) (vec3 (* 2 (:r _)))))
(width  [_] (* 2.0 (:r _)))
(height [_] (* 2.0 (:r _)))
(depth  [_] (* 2.0 (:r _)))
g/IBoundingSphere
(bounding-sphere [_] _)
g/ICenter
(center
 ([_] (thi.ng.geom.types.Sphere. (vec3) (:r _)))
 ([_ p] (thi.ng.geom.types.Sphere. (vec3 p) (:r _))))
(centroid [_] (:p _))
g/IClassify
(classify-point
 [{p :p r :r} q]
 (m/signum (- (* r r) (g/dist-squared p q)) *eps*))


g/IIntersect
(intersect-ray
 [{p :p r :r} ray]
 (let [[rp dir] (if (map? ray) [(:p ray) (:dir ray)] ray)]
   (isec/intersect-ray-sphere? rp dir p r)))
(intersect-shape
 [_ s]
 (cond
  (instance? thi.ng.geom.types.AABB s)
  (isec/intersect-aabb-sphere? s _)
  (instance? thi.ng.geom.types.Sphere s)
  (isec/intersect-sphere-sphere? _ s)
  (instance? thi.ng.geom.types.Plane s)
  (isec/intersect-plane-sphere? (:n s) (:w s) (:p _) (:r _))
  :default (err/type-error! "Sphere" s)))
g/IMeshConvert
(as-mesh
 ([_] (g/as-mesh _ {}))
 ([{[x y z] :p r :r} {:keys [mesh res slices stacks attribs] :or {res *resolution*}}]
  (let [slices (or slices res), stacks (or stacks res)
        range-u (range slices), range-v (range stacks)]
    (->> (for [i range-u, j range-v
               :let [u (/ i slices)
                     v (/ j stacks)
                     u1 (/ (inc i) slices)
                     v1 (/ (inc j) stacks)
                     verts [[u v]]
                     verts (if (pos? j) (conj verts [u1 v]) verts)
                     verts (if (< j (dec stacks)) (conj verts [u1 v1]) verts)]]
           (conj verts [u v1]))
         ;; TODO transduce
         (map-indexed
          (fn [i fuvs]
            (let [fverts (mapv
                          (fn [uv]
                            (let [theta (* TWO_PI (uv 0)) ;; FIXME optimize trig
                                  phi (* PI (uv 1))
                                  st (Math/sin theta) ct (Math/cos theta)
                                  sp (Math/sin phi) cp (Math/cos phi)]
                              (vec3
                               (+ (mm/mul ct sp r) x)
                               (mm/madd cp r y)
                               (+ (mm/mul st sp r) z))))
                          fuvs)]
              (attr/generate-face-attribs fverts i attribs {:uv fuvs}))))
         (g/into (or mesh (bm/basic-mesh)))))))
g/IProximity
(closest-point
 [{p :p r :r} q]
 (m/+! (m/normalize (m/- q p) r) p))
g/ISample
(random-point-inside
 [_]
 (m/+ (:p _) (v/randvec3 (m/random (:r _)))))
(random-point
 [_]
 (m/+ (:p _) (v/randvec3 (:r _))))
g/ITessellate
(tessellate
 ([_] (g/tessellate _ {}))
 ([_ opts] (g/tessellate (g/as-mesh _ opts))))
g/IRotate
(rotate
 [_ theta] (thi.ng.geom.types.Sphere. (g/rotate-z (:p _) theta) (:r _)))
g/IRotate3D
(rotate-x
 [_ theta] (thi.ng.geom.types.Sphere. (g/rotate-x (:p _) theta) (:r _)))
(rotate-y
 [_ theta] (thi.ng.geom.types.Sphere. (g/rotate-y (:p _) theta) (:r _)))
(rotate-z
 [_ theta] (thi.ng.geom.types.Sphere. (g/rotate-z (:p _) theta) (:r _)))
(rotate-around-axis
 [_ axis theta]
 (thi.ng.geom.types.Sphere.
  (g/rotate-around-axis (:p _) axis theta) (:r _)))

g/IScale
(scale [_ s] (thi.ng.geom.types.Sphere. (m/* (:p _) s) (* (:r _) s)))
(scale-size [_ s] (thi.ng.geom.types.Sphere. (:p _) (* (:r _) s)))

g/ITranslate
(translate [_ t] (thi.ng.geom.types.Sphere. (m/+ (:p _) t) (:r _)))
g/IVolume
(volume [{r :r}] (mm/mul (/ 4.0 3.0) PI r r r))
)
