(ns thi.ng.geom.webgl.camera
  (:refer-clojure :exclude [apply])
  (:require
   [thi.ng.math.core :as m]
   [thi.ng.geom.vector :as v :refer [vec3]]
   [thi.ng.geom.matrix :as mat]
   [thi.ng.geom.webgl.core :as gl]))

(defn apply
  "Takes a webgl model spec map & camera, injects :view & :proj
  uniforms into spec."
  [spec cam]
  (update spec :uniforms merge {:view (:view cam) :proj (:proj cam)}))

(defn update-keys
  "Takes a map m, key seq and map of new vals, replaces keys in m with
  new vals. If a value in opts map is a function, applies fn to value
  of key in original map."
  [m ks opts]
  (reduce-kv
   (fn [acc k v] (assoc acc k (if (fn? v) (v (m k)) v)))
   m (select-keys opts ks)))

(defn set-view
  [cam opts]
  (let [cam (update-keys cam [:eye :target :up] opts)]
    (assoc cam :view (mat/look-at (:eye cam) (:target cam) (:up cam)))))

(defn set-projection
  [cam opts]
  (let [cam (update-keys cam  [:fov :aspect :near :far] opts)]
    (assoc cam :proj (gl/perspective (:fov cam) (:aspect cam) (:near cam) (:far cam)))))

(defn perspective-camera
  [opts]
  (-> (merge
       {:eye    (vec3 0.0 0.0 2.0)
        :target v/V3
        :up     v/V3Y
        :fov    45
        :near   0.1
        :far    100
        :aspect (/ 16.0 9.0)}
       opts)
      (set-view opts)
      (set-projection opts)))
