(ns thi.ng.geom.webgl.core
  (:require
   [thi.ng.math.core :as m]
   [thi.ng.geom.core :as g]
   [thi.ng.geom.utils :as gu]
   [thi.ng.geom.vector :as v :refer [vec2 vec3 V3Z]]
   [thi.ng.geom.matrix :as mat]
   [thi.ng.geom.types]
   [thi.ng.geom.gmesh :as gm]
   [thi.ng.geom.rect :as r]
   [thi.ng.geom.webgl.constants :as glc]
   [thi.ng.geom.webgl.shaders :as sh]
   [thi.ng.geom.webgl.utils :as glu]
   [thi.ng.color.core :as col]
   [thi.ng.typedarrays.core :as ta]
   [thi.ng.dstruct.streams :as streams]
   [thi.ng.xerror.core :as err]))

(declare into-buffer-vec3)

(defprotocol IWebGLConvert
  (as-webgl-buffer-spec [_ opts]))

(defprotocol IRelease
  (release [_]))

(defprotocol IBind
  (bind [_] [_ opts])
  (unbind [_] [_ opts]))

(defprotocol IConfigure
  (configure [_ opts]))

(defprotocol ITexture
  (set-texture-filter [_ min mag])
  (set-texture-wrap [_ wrap-s wrap-t]))

(defprotocol IFramebuffer
  (set-fbo-color-texture [_ tex])
  (set-fbo-depth-buffer [_ depth-buffer]))

(defn fill-buffer-vec2
  [buf idx v num stride]
  (let [vbuf (.-buf ^thi.ng.geom.vector.Vec2 v)]
    (loop [idx 0, num num]
      (when (pos? num)
        (.set buf vbuf idx)
        (recur (+ idx stride) (dec num)))))
  buf)

(defn fill-buffer-vec3
  [buf idx v num stride]
  (let [vbuf (.-buf ^thi.ng.geom.vector.Vec3 v)]
    (loop [idx 0, num num]
      (when (pos? num)
        (.set buf vbuf idx)
        (recur (+ idx stride) (dec num)))))
  buf)

(defn fill-buffer
  [buf idx coll n stride]
  (let [b  (ta/float32 coll)]
    (loop [i idx, n n]
      (when (pos? n)
        (.set buf b i)
        (recur (+ i stride) (dec n))))
    buf))

(defn fill-vertex-buffer
  [buf coll stride]
  (loop [i 0, coll (seq coll)]
    (if coll
      (recur (streams/into-buffer (first coll) buf stride i) (next coll))
      buf)))

(defn fill-vertex-buffer-3
  [f buf coll stride]
  (loop [i 0, coll (seq coll)]
    (if coll
      (let [[a b c] (first coll)]
        (recur (->> (f a buf stride i)
                    (f b buf stride)
                    (f c buf stride))
               (next coll)))
      buf)))

(defn face-normals-buffer
  [faces]
  (let [buf  (ta/float32 (* (count faces) 3 3))]
    (loop [idx 0, faces faces]
      (if faces
        (let [nbuf (.-buf ^thi.ng.geom.vector.Vec3 (gu/ortho-normal (first faces)))]
          (.set buf nbuf idx)
          (.set buf nbuf (+ idx 3))
          (.set buf nbuf (+ idx 6))
          (recur (+ idx 9) (next faces)))
        buf))))

(defn face-normals
  [m]
  (let [fnorms (g/face-normals m true)
        faces  (g/faces m)
        buf    (ta/float32 (* (count faces) 3 3))]
    (loop [idx 0, faces faces]
      (if faces
        (let [nbuf (.-buf ^thi.ng.geom.vector.Vec3 (fnorms (first faces)))]
          (.set buf nbuf idx)
          (.set buf nbuf (+ idx 3))
          (.set buf nbuf (+ idx 6))
          (recur (+ idx 9) (next faces)))
        buf))))

(defn vertex-normals
  [m]
  (let [vnorms (g/vertex-normals m true)
        faces  (g/faces m)
        buf    (ta/float32 (* (count faces) 3 3))]
    (loop [i 0, faces faces]
      (if faces
        (let [f (first faces)]
          (recur (->> (streams/into-buffer (vnorms (f 0)) buf 3 i)
                      (streams/into-buffer (vnorms (f 1)) buf 3)
                      (streams/into-buffer (vnorms (f 2)) buf 3))
                 (next faces)))
        buf))))
(defn common-attrib-buffer-specs
  [{:keys [num-vertices] :as acc} {:keys [normals fixed-normal uv colors single-color]}]
  (let [c-stride (if colors
                   (count (first colors))
                   (if single-color (count single-color)))]
    (cond->
     acc

     normals
     (assoc-in [:attribs :normal] {:data normals :size 3})

     fixed-normal
     (assoc-in
      [:attribs :normal]
      {:data (fill-buffer-vec3
              (ta/float32 (* num-vertices 3))
              0 fixed-normal num-vertices 3)
       :size 3})

     uv
     (assoc-in
      [:attribs :uv]
      {:data (fill-vertex-buffer
              (ta/float32 (* 2 (count uv))) uv 2)
       :size 2})

     colors
     (assoc-in
      [:attribs :color]
      {:data (fill-vertex-buffer
              (ta/float32 (* c-stride (count colors)))
              colors c-stride)
       :size c-stride})

     single-color
     (assoc-in
      [:attribs :color]
      {:data (fill-buffer
              (ta/float32 (* num-vertices c-stride))
              0 single-color num-vertices c-stride)
       :size c-stride}))))

(defn into-buffer-vec2
  [^thi.ng.geom.vector.Vec2 v buf stride idx]
  (.set buf (.-buf v) idx)
  (+ idx stride))

(defn into-buffer-vec3
  [^thi.ng.geom.vector.Vec3 v buf stride idx]
  (.set buf (.-buf v) idx)
  (+ idx stride))  
(extend-type cljs.core.PersistentVector
  streams/IBuffer
  (get-buffer
    [_] (ta/float32 _))
  (into-buffer
    [_ buf stride idx]
    (let [t     (type (first _))
          into* (case t
                 thi.ng.geom.vector.Vec2 into-buffer-vec2
                 thi.ng.geom.vector.Vec3 into-buffer-vec3
                 streams/into-buffer)]
      (case (count _)
        2 (let [idx (into* (first _) buf stride idx)]
            (into* (nth _ 1) buf stride idx))
        3 (let [idx (into* (first _) buf stride idx)
                idx (into* (nth _ 1) buf stride idx)]
            (into* (nth _ 2) buf stride idx))
        (loop [idx idx, xs _]
          (if xs
            (recur (into* (first xs) buf stride idx) (next xs))
            idx))))))
(extend-type thi.ng.geom.types.LineStrip2
  IWebGLConvert
  (as-webgl-buffer-spec
    [{:keys [points]} {:keys [stride] :or {stride 2} :as spec}]
    (let [numv (count points)]
      (common-attrib-buffer-specs
       {:attribs
        {:position    {:data (fill-vertex-buffer
                              (ta/float32 (* numv stride))
                              points stride)
                       :size stride}}
        :mode         glc/line-strip
        :num-vertices numv}
       spec))))
(extend-type thi.ng.geom.types.LineStrip3
  IWebGLConvert
  (as-webgl-buffer-spec
    [{:keys [points]} {:keys [stride] :or {stride 3} :as spec}]
    (let [numv (count points)]
      (common-attrib-buffer-specs
       {:attribs
        {:position    {:data (fill-vertex-buffer
                              (ta/float32 (* numv stride))
                              points stride)
                       :size stride}}
        :mode         glc/line-strip
        :num-vertices numv}
       spec))))
(extend-type thi.ng.geom.types.Rect2
  IWebGLConvert
  (as-webgl-buffer-spec
    [r {:keys [stride normals] :or {stride 2} :as spec}]
    (let [[a b c d] (g/vertices r)]
      (common-attrib-buffer-specs
       {:attribs
        {:position    {:data (fill-vertex-buffer
                              (ta/float32 (* 4 stride))
                              [a b d c] stride)
                       :size stride}}
        :mode         glc/triangle-strip
        :num-vertices 4}
       (if normals
         (-> spec (assoc :fixed-normal V3Z) (dissoc :normals))
         spec)))))
(extend-type thi.ng.geom.types.Polygon2
  IWebGLConvert
  (as-webgl-buffer-spec
    [_ {:keys [normals stride mode] :or {stride 2, normals true, mode glc/triangles} :as spec}]
    (if (= glc/triangles mode)
      (let [faces     (g/tessellate _)
            num-faces (count faces)
            num-verts (* num-faces 3)]
        (common-attrib-buffer-specs
         {:attribs {:position {:data (fill-vertex-buffer
                                      (ta/float32 (* num-verts stride))
                                      faces stride)
                               :size stride}}
          :mode         mode
          :num-vertices num-verts
          :num-faces    num-faces}
         (if normals
           (-> spec (assoc :fixed-normal V3Z) (dissoc :normals))
           spec)))
      (let [verts     (g/vertices _)
            verts     (conj verts (first verts))
            num-verts (count verts)]
        (common-attrib-buffer-specs
         {:attribs {:position {:data (fill-vertex-buffer
                                      (ta/float32 (* num-verts stride))
                                      verts stride)
                               :size stride}}
          :mode         mode
          :num-vertices num-verts}
         (dissoc spec :normals))))))
(extend-type thi.ng.geom.types.BasicMesh
  IWebGLConvert
  (as-webgl-buffer-spec
    [_ {:keys [fnormals tessellate stride]
        :or   {fnormals true, tessellate true, stride 3} :as spec}]
    (let [m         (if tessellate (g/tessellate _) _)
          faces     (map #(g/vertices % m) (g/faces m))
          num-faces (count faces)
          num-verts (* num-faces 3)]
      (common-attrib-buffer-specs
       {:attribs
        {:position    {:data (fill-vertex-buffer-3
                              into-buffer-vec3
                              (ta/float32 (* num-verts stride))
                              faces stride)
                       :size stride}}
        :mode         glc/triangles
        :num-vertices num-verts
        :num-faces    num-faces}
       (assoc spec :normals (if fnormals (face-normals m)))))))
(extend-type thi.ng.geom.types.GMesh
  IWebGLConvert
  (as-webgl-buffer-spec
    [_ {:keys [vnormals fnormals tessellate stride]
        :or   {fnormals true, tessellate true, stride 3} :as spec}]
    (let [m         (if tessellate (g/tessellate _) _)
          faces     (map #(g/vertices % m) (g/faces m))
          num-faces (count faces)
          num-verts (* num-faces 3)
          normals   (cond
                     vnormals (vertex-normals m)
                     fnormals (face-normals m)
                     :default nil)]
      (common-attrib-buffer-specs
       {:attribs
        {:position    {:data (fill-vertex-buffer-3
                              into-buffer-vec3
                              (ta/float32 (* num-verts stride))
                              faces stride)
                       :size stride}}
        :mode         glc/triangles
        :num-vertices num-verts
        :num-faces    num-faces}
       (assoc spec :normals normals)))))

(def context-default-attribs
  {:alpha true
   :antialias true
   :depth true
   :fail-if-major-performance-caveat false
   :prefer-low-power-to-high-performance false
   :premultiplied-alpha true
   :preserve-drawing-buffer false
   :stencil false})
(defn gl-context
  ([canvas] (gl-context canvas {}))
  ([canvas attribs]
     (let [canvas (if (string? canvas) (.getElementById js/document canvas) canvas)
           attribs (clj->js (merge context-default-attribs attribs))
           ctx (loop [ids ["webgl" "experimental-webgl" "webkit-3d" "moz-webgl"]]
                 (when ids
                   (try
                     (let [ctx (.getContext canvas (first ids) attribs)]
                       (set! (.-onselectstart canvas) (constantly false))
                       (if ctx ctx (recur (next ids))))
                     (catch js/Error e (recur (next ids))))))]
       (or ctx (err/unsupported! "WebGL not available")))))
(defn clear-color-buffer
  ([^WebGLRenderingContext gl col]
   (let [^thi.ng.color.core.RGBA c (col/as-rgba col)]
     (clear-color-buffer gl (.-r c) (.-g c) (.-b c) (.-a c))))
  ([^WebGLRenderingContext gl r g b a]
   (.clearColor gl r g b a)
   (.clear gl glc/color-buffer-bit)
   gl))

(defn clear-depth-buffer
  [^WebGLRenderingContext gl d]
  (.clearDepth gl d)
  (.clear gl glc/depth-buffer-bit)
  gl)

(defn clear-color-and-depth-buffer
  ([^WebGLRenderingContext gl col d]
   (let [^thi.ng.color.core.RGBA c (col/as-rgba col)]
     (clear-color-and-depth-buffer gl (.-r c) (.-g c) (.-b c) (.-a c) d)))
  ([^WebGLRenderingContext gl r g b a d]
   (.clearColor gl r g b a)
   (.clearDepth gl d)
   (.clear gl (bit-or glc/depth-buffer-bit glc/color-buffer-bit))
   gl))

(defn disable
  [^WebGLRenderingContext gl flag]
  (.disable gl flag)
  gl)

(defn enable
  [^WebGLRenderingContext gl flag]
  (.enable gl flag)
  gl)

(defn scissor-test
  ([^WebGLRenderingContext gl {[x y] :p [w h] :size}]
   (scissor-test gl x y w h))
  ([^WebGLRenderingContext gl x y w h]
   (.enable gl glc/scissor-test)
   (.scissor gl x y w h)
   gl))

(defn cull-faces
  [^WebGLRenderingContext gl side]
  (.enable gl glc/cull-face)
  (.cullFace gl side)
  gl)

(defn set-viewport
  ([^WebGLRenderingContext gl {[x y] :p [w h] :size}]
   (.viewport gl x y w h)
   gl)
  ([^WebGLRenderingContext gl x y w h]
   (.viewport gl x y w h)
   gl))

(defn get-viewport-rect
  [^WebGLRenderingContext gl]
  (let [b (.getParameter gl glc/viewport)]
    (r/rect (aget b 0) (aget b 1) (aget b 2) (aget b 3))))

(defn perspective
  [fovy aspect near far]
  (let [aspect (cond
                 (number? aspect) aspect
                 (map? aspect)    (let [{[w h] :size} aspect] (/ w h))
                 :else            (/ (first aspect) (nth aspect 1)))]
    (mat/perspective fovy aspect near far)))

(defn ortho
  ([] (mat/ortho -1 -1 1 1 -1 1))
  ([view-rect]
   (let [a (apply / (get view-rect :size))]
     (mat/ortho (- a) 1 a -1 -1 1))))

(def ^:private float-ext-ids
  ["OES_texture_float"
   "OES_texture_half_float"
   "OES_texture_float_linear"
   "OES_texture_half_float_linear"
   "WEBGL_color_buffer_float"
   "EXT_color_buffer_half_float"])

(defn get-extension
  [^WEBGL_color_buffer_float gl ext]
  (.getExtension gl ext))

(defn get-float-extension
  [^WebGLRenderingContext gl spec]
  (let [[s-tex h-tex s-lin h-lin s-fbo h-fbo :as ext] (map #(.getExtension gl %) float-ext-ids)]
    (->> [{:texture s-tex :filterable s-lin :renderable s-fbo
           :type glc/float
           :precision :single :single true}
          {:texture h-tex :filterable h-lin :renderable h-fbo
           :type (when h-tex (.-HALF_FLOAT_OES h-tex))
           :precision :half :half true}]
         (reduce
          (fn [acc c]
            (if (and (get c :texture) (every? c (get spec :require)))
              (conj acc c) acc))
          [])
         (map
          (fn [c]
            (assoc c :score
                   (apply + (map (fn [pref score] (if (c pref) score 0))
                                 (get spec :prefer) [0x80 0x40 0x20 0x10 0x8 0x4 0x2 0x1])))))
         (sort-by :score)
         (last))))

(defn get-supported-extensions
 [^WebGLRenderingContext gl]
 (.getSupportedExtensions gl))

(defn make-attribute-buffer
  [^WebGLRenderingContext gl target draw-type data]
  (let [buffer (.createBuffer gl)]
    (.bindBuffer gl target buffer)
    (.bufferData gl target data draw-type)
    buffer))

(defn make-attribute-buffers
  [^WebGLRenderingContext gl mode specs]
  (reduce-kv
   (fn [specs id {:keys [data target] :or {target glc/array-buffer}}]
     (update-in
      specs [id] merge
      {:buffer      (make-attribute-buffer gl target mode data)
       :target      target
       :buffer-mode mode}))
   specs specs))

(defn make-buffers-in-spec
  [spec ^WebGLRenderingContext gl mode]
  (let [spec (update spec :attribs #(make-attribute-buffers gl mode %))]
    (if (get spec :indices)
      (update spec :indices
              #(merge %
                      {:buffer (make-attribute-buffer gl glc/element-array-buffer mode (get % :data))
                       :buffer-mode mode}))
      spec)))

(defn update-buffer-in-spec
  [^WebGLRenderingContext gl spec id coll]
  (let [{:keys [target data buffer buffer-mode size]} (-> spec :attribs id)]
    (fill-vertex-buffer data coll size)
    (.bindBuffer gl target buffer)
    (.bufferData gl target data buffer-mode)
    gl))

(defn begin-shader
  [^WebGLRenderingContext gl shader uniforms attribs indices]
  (.useProgram gl (get shader :program))
  (sh/apply-default-uniforms shader uniforms)
  (reduce-kv #(sh/set-uniform shader %2 %3) nil uniforms)
  (reduce-kv #(sh/set-attribute gl shader %2 %3) nil attribs)
  (when indices
    (.bindBuffer gl glc/element-array-buffer (get indices :buffer))))

(defn end-shader
  [^WebGLRenderingContext gl shader]
  (reduce #(sh/disable-attribute gl shader (key %2)) nil (get shader :attribs)))
(defn bind-sequentially
  [coll]
  (loop [i 0, coll coll]
    (when coll
      (when-let [x (first coll)]
        (bind x i))
      (recur (inc i) (next coll)))))

(defn prepare-render-state
  "Takes a GL context and shader spec, sets GL render flags stored
  under :state key (only if :state is present)."
  [^WebGLRenderingContext gl {:keys [state]}]
  (when state
    (if (get state :depth-test)
      (enable gl glc/depth-test)
      (disable gl glc/depth-test))
    (if (get state :blend)
      (let [[src dest] (or (get state :blend-fn) [glc/src-alpha glc/one])]
        (doto gl
          (enable glc/blend)
          (.blendFunc src dest)))
      (disable gl glc/blend))
    (when-let [tex (get state :tex)]
      (if (sequential? tex)
        (bind-sequentially tex)
        (bind tex 0))))
  gl)
(defn compute-normal-matrix
  [m v] (-> v (m/* m) (m/invert) (m/transpose)))

(defn inject-normal-matrix
  [spec model-mat view-mat normal-mat-id]
  (let [model-mat (if (keyword? model-mat)
                    (-> spec :uniforms model-mat)
                    model-mat)
        view-mat (if (keyword? view-mat)
                    (-> spec :uniforms view-mat)
                    view-mat)]
    (assoc-in
     spec [:uniforms normal-mat-id]
     (compute-normal-matrix model-mat view-mat))))
(defn draw
  [^WebGLRenderingContext gl spec]
  (let [mode (get spec :mode glc/triangles)]
    (if (get spec :indices)
      (.drawElements gl mode (get spec :num-items) glc/unsigned-short 0)
      (.drawArrays gl mode 0 (get spec :num-vertices)))
    gl))

(defn draw-with-shader
  [^WebGLRenderingContext gl {:keys [shader] :as spec}]
  (prepare-render-state gl shader)
  (begin-shader gl shader (get spec :uniforms) (get spec :attribs) (get spec :indices))
  (draw gl spec)
  (end-shader gl shader)
  gl)
