(ns thi.ng.geom.gl.jogl.constants
  (:refer-clojure
   :exclude [byte compile double filter float
             int load max min reduce set short
             keep repeat replace and or type]))

;; Constant extraction from JOGL sources
;;
;; The code below is used to extract and convert OpenGL constants from
;; the Java JOGL sources in order to provide a better user experience
;; from Clojure. This is a superset of the WebGL constants defined in
;; `src/gl/webgl/constants.cljs`.
;;
;; The resulting constants are all in kebab case naming convention
;; with their `gl-` prefix removed, unless it would result in a symbol
;; with leading digits (in which case the `gl-` remains intact)

(comment

  (defn keep-gl?
    [x]
    (or (re-find #"^GL_[0-9]+" x)
        (= "GL_FALSE" x)
        (= "GL_TRUE" x)))

  (defn snake->kebab
    [x]
    (let [x (if (keep-gl? x) x (subs x 3))]
      (-> x str/lower-case (str/replace "_" "-"))))

  (defn extract-constants-from-file
    [path]
    (->> path
         (io/reader)
         (line-seq)
         (mapcat #(re-seq #"([A-Z0-9x_]+)\s*=\s*(0x[0-9a-f]+)" %))
         (filter identity)
         (map (fn [[_ k v]] (str "(def " (snake->kebab k) " " v ")")))))

  ;; jogl-all-2.3.2-sources/com/jogamp/opengl/
  (defn extract-constants
    [base]
    (->> ["GL" "GL2" "GL2ES1" "GL2ES2" "GL2ES3" "GL2GL3" "GL3" "GL3ES3" "GL4" "GL4ES3"]
         (mapcat #(extract-constants-from-file (str base % ".java")))
         (into #{})
         (sort)
         (interpose "\n")
         (apply str)))
  )

;; OpenGL constants (combined versions 1.2 - 4.0)

(def abgr-ext 0x8000)
(def accum 0x100)
(def accum-alpha-bits 0xd5b)
(def accum-blue-bits 0xd5a)
(def accum-buffer-bit 0x200)
(def accum-clear-value 0xb80)
(def accum-green-bits 0xd59)
(def accum-red-bits 0xd58)
(def active-atomic-counter-buffers 0x92d9)
(def active-attribute-max-length 0x8b8a)
(def active-attributes 0x8b89)
(def active-program 0x8259)
(def active-resources 0x92f5)
(def active-stencil-face-ext 0x8911)
(def active-subroutine-max-length 0x8e48)
(def active-subroutine-uniform-locations 0x8e47)
(def active-subroutine-uniform-max-length 0x8e49)
(def active-subroutine-uniforms 0x8de6)
(def active-subroutines 0x8de5)
(def active-texture 0x84e0)
(def active-uniform-block-max-name-length 0x8a35)
(def active-uniform-blocks 0x8a36)
(def active-uniform-max-length 0x8b87)
(def active-uniforms 0x8b86)
(def active-variables 0x9305)
(def active-vertex-units-arb 0x86a5)
(def add 0x104)
(def add-signed 0x8574)
(def aliased-line-width-range 0x846e)
(def aliased-point-size-range 0x846d)
(def all-attrib-bits 0xfffff)
(def all-barrier-bits 0xffffffff)
(def all-client-attrib-bits 0xffffffff)
(def all-shader-bits 0xffffffff)
(def allow-draw-frg-hint-pgi 0x1a210)
(def allow-draw-mem-hint-pgi 0x1a211)
(def allow-draw-obj-hint-pgi 0x1a20e)
(def allow-draw-win-hint-pgi 0x1a20f)
(def alpha 0x1906)
(def alpha-bias 0xd1d)
(def alpha-bits 0xd55)
(def alpha-float16-apple 0x881c)
(def alpha-float16-ati 0x881c)
(def alpha-float32-apple 0x8816)
(def alpha-float32-ati 0x8816)
(def alpha-integer 0x8d97)
(def alpha-max-clamp-ingr 0x8567)
(def alpha-min-clamp-ingr 0x8563)
(def alpha-ref-command-nv 0xf)
(def alpha-scale 0xd1c)
(def alpha-snorm 0x9010)
(def alpha-test 0xbc0)
(def alpha-test-func 0xbc1)
(def alpha-test-ref 0xbc2)
(def alpha12 0x803d)
(def alpha16 0x803e)
(def alpha16-snorm 0x9018)
(def alpha16f 0x881c)
(def alpha16i 0x8d8a)
(def alpha16ui 0x8d78)
(def alpha32f 0x8816)
(def alpha32i 0x8d84)
(def alpha32ui 0x8d72)
(def alpha4 0x803b)
(def alpha8 0x803c)
(def alpha8-snorm 0x9014)
(def alpha8i 0x8d90)
(def alpha8ui 0x8d7e)
(def already-signaled 0x911a)
(def always 0x207)
(def always-fast-hint-pgi 0x1a20c)
(def always-soft-hint-pgi 0x1a20d)
(def and 0x1501)
(def and-inverted 0x1504)
(def and-reverse 0x1502)
(def any-samples-passed 0x8c2f)
(def any-samples-passed-conservative 0x8d6a)
(def array-buffer 0x8892)
(def array-buffer-binding 0x8894)
(def array-element-lock-count-ext 0x81a9)
(def array-element-lock-first-ext 0x81a8)
(def array-size 0x92fb)
(def array-stride 0x92fe)
(def atomic-counter-barrier-bit 0x1000)
(def atomic-counter-buffer 0x92c0)
(def atomic-counter-buffer-active-atomic-counter-indices 0x92c6)
(def atomic-counter-buffer-active-atomic-counters 0x92c5)
(def atomic-counter-buffer-binding 0x92c1)
(def atomic-counter-buffer-data-size 0x92c4)
(def atomic-counter-buffer-index 0x9301)
(def atomic-counter-buffer-referenced-by-compute-shader 0x90ed)
(def atomic-counter-buffer-referenced-by-fragment-shader 0x92cb)
(def atomic-counter-buffer-referenced-by-geometry-shader 0x92ca)
(def atomic-counter-buffer-referenced-by-tess-control-shader 0x92c8)
(def atomic-counter-buffer-referenced-by-tess-evaluation-shader 0x92c9)
(def atomic-counter-buffer-referenced-by-vertex-shader 0x92c7)
(def atomic-counter-buffer-size 0x92c3)
(def atomic-counter-buffer-start 0x92c2)
(def attached-shaders 0x8b85)
(def attenuation-ext 0x834d)
(def attrib-stack-depth 0xbb0)
(def attribute-address-command-nv 0x9)
(def auto-generate-mipmap 0x8295)
(def auto-normal 0xd80)
(def aux-buffers 0xc00)
(def aux-depth-stencil-apple 0x8a14)
(def aux0 0x409)
(def aux1 0x40a)
(def aux2 0x40b)
(def aux3 0x40c)
(def average-ext 0x8335)
(def back 0x405)
(def back-left 0x402)
(def back-normals-hint-pgi 0x1a223)
(def back-right 0x403)
(def bgr 0x80e0)
(def bgr-integer 0x8d9a)
(def bgra 0x80e1)
(def bgra-integer 0x8d9b)
(def bgra8 0x93a1)
(def bitmap 0x1a00)
(def bitmap-token 0x704)
(def blend 0xbe2)
(def blend-advanced-coherent-khr 0x9285)
(def blend-color 0x8005)
(def blend-color-command-nv 0xb)
(def blend-dst 0x0)
(def blend-dst-alpha 0x80ca)
(def blend-dst-rgb 0x80c8)
(def blend-equation 0x8009)
(def blend-equation-alpha 0x883d)
(def blend-equation-rgb 0x8009)
(def blend-src 0x0)
(def blend-src-alpha 0x80cb)
(def blend-src-rgb 0x80c9)
(def block-index 0x92fd)
(def blue 0x1905)
(def blue-bias 0xd1b)
(def blue-bits 0xd54)
(def blue-integer 0x8d96)
(def blue-max-clamp-ingr 0x8566)
(def blue-min-clamp-ingr 0x8562)
(def blue-scale 0xd1a)
(def bool 0x8b56)
(def bool-arb 0x8b56)
(def bool-vec2 0x8b57)
(def bool-vec2-arb 0x8b57)
(def bool-vec3 0x8b58)
(def bool-vec3-arb 0x8b58)
(def bool-vec4 0x8b59)
(def bool-vec4-arb 0x8b59)
(def buffer 0x82e0)
(def buffer-access 0x88bb)
(def buffer-access-flags 0x911f)
(def buffer-binding 0x9302)
(def buffer-data-size 0x9303)
(def buffer-flushing-unmap 0x8a13)
(def buffer-gpu-address-nv 0x8f1d)
(def buffer-immutable-storage 0x821f)
(def buffer-map-length 0x9120)
(def buffer-map-offset 0x9121)
(def buffer-map-pointer 0x88bd)
(def buffer-mapped 0x88bc)
(def buffer-object-apple 0x85b3)
(def buffer-serialized-modify 0x8a12)
(def buffer-size 0x8764)
(def buffer-storage-flags 0x8220)
(def buffer-update-barrier-bit 0x200)
(def buffer-usage 0x8765)
(def buffer-variable 0x92e5)
(def byte 0x1400)
(def c3f-v3f 0x2a24)
(def c4f-n3f-v3f 0x2a26)
(def c4ub-v2f 0x2a22)
(def c4ub-v3f 0x2a23)
(def caveat-support 0x82b8)
(def ccw 0x901)
(def clamp 0x2900)
(def clamp-fragment-color 0x891b)
(def clamp-read-color 0x891c)
(def clamp-to-border 0x812d)
(def clamp-to-edge 0x812f)
(def clamp-vertex-color 0x891a)
(def clear 0x1500)
(def clear-buffer 0x82b4)
(def clear-texture 0x9365)
(def client-active-texture 0x84e1)
(def client-all-attrib-bits 0xffffffff)
(def client-attrib-stack-depth 0xbb1)
(def client-mapped-buffer-barrier-bit 0x4000)
(def client-pixel-store-bit 0x1)
(def client-storage-bit 0x200)
(def client-vertex-array-bit 0x2)
(def clip-depth-mode 0x935d)
(def clip-distance0 0x3000)
(def clip-distance1 0x3001)
(def clip-distance2 0x3002)
(def clip-distance3 0x3003)
(def clip-distance4 0x3004)
(def clip-distance5 0x3005)
(def clip-distance6 0x3006)
(def clip-distance7 0x3007)
(def clip-far-hint-pgi 0x1a221)
(def clip-near-hint-pgi 0x1a220)
(def clip-origin 0x935c)
(def clip-plane0 0x3000)
(def clip-plane1 0x3001)
(def clip-plane2 0x3002)
(def clip-plane3 0x3003)
(def clip-plane4 0x3004)
(def clip-plane5 0x3005)
(def clip-volume-clipping-hint-ext 0x80f0)
(def clipping-input-primitives-arb 0x82f6)
(def clipping-output-primitives-arb 0x82f7)
(def cmyk-ext 0x800c)
(def cmyka-ext 0x800d)
(def coeff 0xa00)
(def color 0x1800)
(def color-array-address-nv 0x8f23)
(def color-array-buffer-binding 0x8898)
(def color-array-length-nv 0x8f2d)
(def color-array-pointer 0x8090)
(def color-array-size 0x8081)
(def color-array-stride 0x8083)
(def color-array-type 0x8082)
(def color-attachment0 0x8ce0)
(def color-attachment1 0x8ce1)
(def color-attachment10 0x8cea)
(def color-attachment11 0x8ceb)
(def color-attachment12 0x8cec)
(def color-attachment13 0x8ced)
(def color-attachment14 0x8cee)
(def color-attachment15 0x8cef)
(def color-attachment16 0x8cf0)
(def color-attachment17 0x8cf1)
(def color-attachment18 0x8cf2)
(def color-attachment19 0x8cf3)
(def color-attachment2 0x8ce2)
(def color-attachment20 0x8cf4)
(def color-attachment21 0x8cf5)
(def color-attachment22 0x8cf6)
(def color-attachment23 0x8cf7)
(def color-attachment24 0x8cf8)
(def color-attachment25 0x8cf9)
(def color-attachment26 0x8cfa)
(def color-attachment27 0x8cfb)
(def color-attachment28 0x8cfc)
(def color-attachment29 0x8cfd)
(def color-attachment3 0x8ce3)
(def color-attachment30 0x8cfe)
(def color-attachment31 0x8cff)
(def color-attachment4 0x8ce4)
(def color-attachment5 0x8ce5)
(def color-attachment6 0x8ce6)
(def color-attachment7 0x8ce7)
(def color-attachment8 0x8ce8)
(def color-attachment9 0x8ce9)
(def color-buffer-bit 0x4000)
(def color-clear-unclamped-value-ati 0x8835)
(def color-clear-value 0xc22)
(def color-components 0x8283)
(def color-encoding 0x8296)
(def color-float-apple 0x8a0f)
(def color-index 0x1900)
(def color-indexes 0x1603)
(def color-logic-op 0x0)
(def color-material-face 0xb55)
(def color-material-parameter 0xb56)
(def color-matrix 0x80b1)
(def color-matrix-stack-depth 0x80b2)
(def color-renderable 0x8286)
(def color-samples-nv 0x8e20)
(def color-sum 0x8458)
(def color-sum-arb 0x8458)
(def color-table 0x80d0)
(def color-table-alpha-size 0x80dd)
(def color-table-bias 0x80d7)
(def color-table-blue-size 0x80dc)
(def color-table-format 0x80d8)
(def color-table-green-size 0x80db)
(def color-table-intensity-size 0x80df)
(def color-table-luminance-size 0x80de)
(def color-table-red-size 0x80da)
(def color-table-scale 0x80d6)
(def color-table-width 0x80d9)
(def color-writemask 0xc23)
(def color3-bit-pgi 0x10000)
(def color4-bit-pgi 0x20000)
(def colorburn 0x929a)
(def colordodge 0x9299)
(def combine 0x8570)
(def combine-alpha 0x8572)
(def combine-rgb 0x8571)
(def command-barrier-bit 0x40)
(def compare-r-to-texture 0x884e)
(def compare-ref-depth-to-texture 0x884e)
(def compare-ref-to-texture 0x884e)
(def compatible-subroutines 0x8e4b)
(def compile 0x1300)
(def compile-and-execute 0x1301)
(def compile-status 0x8b81)
(def completion-status-arb 0x91b1)
(def compressed-alpha 0x84e9)
(def compressed-intensity 0x84ec)
(def compressed-luminance 0x84ea)
(def compressed-luminance-alpha 0x84eb)
(def compressed-luminance-alpha-latc2-ext 0x8c72)
(def compressed-luminance-latc1-ext 0x8c70)
(def compressed-r11-eac 0x9270)
(def compressed-red 0x8225)
(def compressed-red-rgtc1 0x8dbb)
(def compressed-rg 0x8226)
(def compressed-rg-rgtc2 0x8dbd)
(def compressed-rg11-eac 0x9272)
(def compressed-rgb 0x84ed)
(def compressed-rgb-bptc-signed-float 0x8e8e)
(def compressed-rgb-bptc-signed-float-arb 0x8e8e)
(def compressed-rgb-bptc-unsigned-float 0x8e8f)
(def compressed-rgb-bptc-unsigned-float-arb 0x8e8f)
(def compressed-rgb-s3tc-dxt1-ext 0x83f0)
(def compressed-rgb8-etc2 0x9274)
(def compressed-rgb8-punchthrough-alpha1-etc2 0x9276)
(def compressed-rgba 0x84ee)
(def compressed-rgba-astc-10x10 0x93bb)
(def compressed-rgba-astc-10x10-khr 0x93bb)
(def compressed-rgba-astc-10x5 0x93b8)
(def compressed-rgba-astc-10x5-khr 0x93b8)
(def compressed-rgba-astc-10x6 0x93b9)
(def compressed-rgba-astc-10x6-khr 0x93b9)
(def compressed-rgba-astc-10x8 0x93ba)
(def compressed-rgba-astc-10x8-khr 0x93ba)
(def compressed-rgba-astc-12x10 0x93bc)
(def compressed-rgba-astc-12x10-khr 0x93bc)
(def compressed-rgba-astc-12x12 0x93bd)
(def compressed-rgba-astc-12x12-khr 0x93bd)
(def compressed-rgba-astc-4x4 0x93b0)
(def compressed-rgba-astc-4x4-khr 0x93b0)
(def compressed-rgba-astc-5x4 0x93b1)
(def compressed-rgba-astc-5x4-khr 0x93b1)
(def compressed-rgba-astc-5x5 0x93b2)
(def compressed-rgba-astc-5x5-khr 0x93b2)
(def compressed-rgba-astc-6x5 0x93b3)
(def compressed-rgba-astc-6x5-khr 0x93b3)
(def compressed-rgba-astc-6x6 0x93b4)
(def compressed-rgba-astc-6x6-khr 0x93b4)
(def compressed-rgba-astc-8x5 0x93b5)
(def compressed-rgba-astc-8x5-khr 0x93b5)
(def compressed-rgba-astc-8x6 0x93b6)
(def compressed-rgba-astc-8x6-khr 0x93b6)
(def compressed-rgba-astc-8x8 0x93b7)
(def compressed-rgba-astc-8x8-khr 0x93b7)
(def compressed-rgba-bptc-unorm 0x8e8c)
(def compressed-rgba-bptc-unorm-arb 0x8e8c)
(def compressed-rgba-s3tc-dxt1-ext 0x83f1)
(def compressed-rgba-s3tc-dxt3-ext 0x83f2)
(def compressed-rgba-s3tc-dxt5-ext 0x83f3)
(def compressed-rgba8-etc2-eac 0x9278)
(def compressed-signed-luminance-alpha-latc2-ext 0x8c73)
(def compressed-signed-luminance-latc1-ext 0x8c71)
(def compressed-signed-r11-eac 0x9271)
(def compressed-signed-red-rgtc1 0x8dbc)
(def compressed-signed-rg-rgtc2 0x8dbe)
(def compressed-signed-rg11-eac 0x9273)
(def compressed-sluminance 0x8c4a)
(def compressed-sluminance-alpha 0x8c4b)
(def compressed-srgb 0x8c48)
(def compressed-srgb-alpha 0x8c49)
(def compressed-srgb-alpha-bptc-unorm 0x8e8d)
(def compressed-srgb-alpha-bptc-unorm-arb 0x8e8d)
(def compressed-srgb-alpha-s3tc-dxt1 0x8c4d)
(def compressed-srgb-alpha-s3tc-dxt3 0x8c4e)
(def compressed-srgb-alpha-s3tc-dxt5 0x8c4f)
(def compressed-srgb-s3tc-dxt1 0x8c4c)
(def compressed-srgb8-alpha8-astc-10x10 0x93db)
(def compressed-srgb8-alpha8-astc-10x10-khr 0x93db)
(def compressed-srgb8-alpha8-astc-10x5 0x93d8)
(def compressed-srgb8-alpha8-astc-10x5-khr 0x93d8)
(def compressed-srgb8-alpha8-astc-10x6 0x93d9)
(def compressed-srgb8-alpha8-astc-10x6-khr 0x93d9)
(def compressed-srgb8-alpha8-astc-10x8 0x93da)
(def compressed-srgb8-alpha8-astc-10x8-khr 0x93da)
(def compressed-srgb8-alpha8-astc-12x10 0x93dc)
(def compressed-srgb8-alpha8-astc-12x10-khr 0x93dc)
(def compressed-srgb8-alpha8-astc-12x12 0x93dd)
(def compressed-srgb8-alpha8-astc-12x12-khr 0x93dd)
(def compressed-srgb8-alpha8-astc-4x4 0x93d0)
(def compressed-srgb8-alpha8-astc-4x4-khr 0x93d0)
(def compressed-srgb8-alpha8-astc-5x4 0x93d1)
(def compressed-srgb8-alpha8-astc-5x4-khr 0x93d1)
(def compressed-srgb8-alpha8-astc-5x5 0x93d2)
(def compressed-srgb8-alpha8-astc-5x5-khr 0x93d2)
(def compressed-srgb8-alpha8-astc-6x5 0x93d3)
(def compressed-srgb8-alpha8-astc-6x5-khr 0x93d3)
(def compressed-srgb8-alpha8-astc-6x6 0x93d4)
(def compressed-srgb8-alpha8-astc-6x6-khr 0x93d4)
(def compressed-srgb8-alpha8-astc-8x5 0x93d5)
(def compressed-srgb8-alpha8-astc-8x5-khr 0x93d5)
(def compressed-srgb8-alpha8-astc-8x6 0x93d6)
(def compressed-srgb8-alpha8-astc-8x6-khr 0x93d6)
(def compressed-srgb8-alpha8-astc-8x8 0x93d7)
(def compressed-srgb8-alpha8-astc-8x8-khr 0x93d7)
(def compressed-srgb8-alpha8-etc2-eac 0x9279)
(def compressed-srgb8-etc2 0x9275)
(def compressed-srgb8-punchthrough-alpha1-etc2 0x9277)
(def compressed-texture-formats 0x86a3)
(def compute-program-nv 0x90fb)
(def compute-program-parameter-buffer-nv 0x90fc)
(def compute-shader 0x91b9)
(def compute-shader-bit 0x20)
(def compute-shader-invocations-arb 0x82f5)
(def compute-subroutine 0x92ed)
(def compute-subroutine-uniform 0x92f3)
(def compute-texture 0x82a0)
(def compute-work-group-size 0x8267)
(def condition-satisfied 0x911c)
(def conservative-raster-dilate-granularity-nv 0x937b)
(def conservative-raster-dilate-nv 0x9379)
(def conservative-raster-dilate-range-nv 0x937a)
(def conservative-rasterization-nv 0x9346)
(def conserve-memory-hint-pgi 0x1a1fd)
(def const-eye-nv 0x86e5)
(def constant 0x8576)
(def constant-alpha 0x8003)
(def constant-border 0x8151)
(def constant-color 0x8001)
(def context-compatibility-profile-bit 0x2)
(def context-core-profile-bit 0x1)
(def context-flag-debug-bit 0x2)
(def context-flag-forward-compatible-bit 0x1)
(def context-flag-no-error-bit-khr 0x8)
(def context-flag-robust-access-bit 0x4)
(def context-flags 0x821e)
(def context-lost 0x507)
(def context-profile-mask 0x9126)
(def context-release-behavior 0x82fb)
(def context-release-behavior-flush 0x82fc)
(def context-robust-access 0x90f3)
(def continuous-amd 0x9007)
(def convolution-1d 0x8010)
(def convolution-2d 0x8011)
(def convolution-border-color 0x8154)
(def convolution-border-mode 0x8013)
(def convolution-filter-bias 0x8015)
(def convolution-filter-scale 0x8014)
(def convolution-format 0x8017)
(def convolution-height 0x8019)
(def convolution-width 0x8018)
(def coord-replace 0x8862)
(def copy 0x1503)
(def copy-inverted 0x150)
(def copy-pixel-token 0x706)
(def copy-read-buffer 0x8f36)
(def copy-read-buffer-binding 0x8f36)
(def copy-write-buffer 0x8f37)
(def copy-write-buffer-binding 0x8f37)
(def counter-range-amd 0x8bc1)
(def counter-type-amd 0x8bc0)
(def coverage-modulation-nv 0x9332)
(def coverage-modulation-table-nv 0x9331)
(def coverage-modulation-table-size-nv 0x9333)
(def cubic-ext 0x8334)
(def cull-face 0xb44)
(def cull-face-mode 0xb45)
(def cull-fragment-nv 0x86e7)
(def cull-modes-nv 0x86e0)
(def cull-vertex-ext 0x81aa)
(def cull-vertex-eye-position-ext 0x81ab)
(def cull-vertex-object-position-ext 0x81ac)
(def current-bit 0x1)
(def current-color 0xb00)
(def current-fog-coord 0x8453)
(def current-fog-coordinate 0x8453)
(def current-index 0xb01)
(def current-matrix-arb 0x8641)
(def current-matrix-index-arb 0x8845)
(def current-matrix-stack-depth-arb 0x8640)
(def current-normal 0xb02)
(def current-occlusion-query-id-nv 0x8865)
(def current-palette-matrix-arb 0x8843)
(def current-program 0x8b8d)
(def current-query 0x8865)
(def current-raster-color 0xb04)
(def current-raster-distance 0xb09)
(def current-raster-index 0xb05)
(def current-raster-position 0xb07)
(def current-raster-position-valid 0xb08)
(def current-raster-secondary-color 0x845f)
(def current-raster-texture-coords 0xb06)
(def current-secondary-color 0x8459)
(def current-texture-coords 0xb03)
(def current-vertex-attrib 0x8626)
(def current-vertex-attrib-arb 0x8626)
(def current-vertex-ext 0x87e2)
(def current-vertex-weight-ext 0x850b)
(def current-weight-arb 0x86a8)
(def cw 0x900)
(def darken 0x9297)
(def data-buffer-amd 0x9151)
(def debug-callback-function 0x8244)
(def debug-callback-user-param 0x8245)
(def debug-category-api-error-amd 0x9149)
(def debug-category-application-amd 0x914f)
(def debug-category-deprecation-amd 0x914b)
(def debug-category-other-amd 0x9150)
(def debug-category-performance-amd 0x914d)
(def debug-category-shader-compiler-amd 0x914e)
(def debug-category-undefined-behavior-amd 0x914c)
(def debug-category-window-system-amd 0x914a)
(def debug-group-stack-depth 0x826d)
(def debug-logged-messages 0x9145)
(def debug-logged-messages-amd 0x9145)
(def debug-next-logged-message-length 0x8243)
(def debug-output 0x92e0)
(def debug-output-synchronous 0x8242)
(def debug-severity-high 0x9146)
(def debug-severity-high-amd 0x9146)
(def debug-severity-low 0x9148)
(def debug-severity-low-amd 0x9148)
(def debug-severity-medium 0x9147)
(def debug-severity-medium-amd 0x9147)
(def debug-severity-notification 0x826b)
(def debug-source-api 0x8246)
(def debug-source-application 0x824a)
(def debug-source-other 0x824b)
(def debug-source-shader-compiler 0x8248)
(def debug-source-third-party 0x8249)
(def debug-source-window-system 0x8247)
(def debug-type-deprecated-behavior 0x824d)
(def debug-type-error 0x824c)
(def debug-type-marker 0x8268)
(def debug-type-other 0x8251)
(def debug-type-performance 0x8250)
(def debug-type-pop-group 0x826a)
(def debug-type-portability 0x824f)
(def debug-type-push-group 0x8269)
(def debug-type-undefined-behavior 0x824e)
(def decal 0x2101)
(def decode-ext 0x8a49)
(def decr 0x1e03)
(def decr-wrap 0x8508)
(def delete-status 0x8b80)
(def dependent-ar-texture-2d-nv 0x86e9)
(def dependent-gb-texture-2d-nv 0x86ea)
(def dependent-hilo-texture-2d-nv 0x8858)
(def dependent-rgb-texture-3d-nv 0x8859)
(def dependent-rgb-texture-cube-map-nv 0x885a)
(def depth 0x1801)
(def depth-attachment 0x8d00)
(def depth-bias 0xd1f)
(def depth-bits 0xd56)
(def depth-bounds-ext 0x8891)
(def depth-bounds-test-ext 0x8890)
(def depth-buffer-bit 0x100)
(def depth-clamp 0x864f)
(def depth-clamp-far-amd 0x901f)
(def depth-clamp-near-amd 0x901e)
(def depth-clamp-nv 0x864f)
(def depth-clear-value 0xb73)
(def depth-component 0x1902)
(def depth-component16 0x81a5)
(def depth-component24 0x81a6)
(def depth-component32 0x81a7)
(def depth-component32f 0x8cac)
(def depth-components 0x8284)
(def depth-func 0xb74)
(def depth-range 0xb70)
(def depth-renderable 0x8287)
(def depth-samples-nv 0x932d)
(def depth-scale 0xd1e)
(def depth-stencil 0x84f9)
(def depth-stencil-attachment 0x821a)
(def depth-stencil-texture-mode 0x90ea)
(def depth-stencil-to-bgra-nv 0x886f)
(def depth-stencil-to-rgba-nv 0x886e)
(def depth-test 0xb71)
(def depth-texture-mode 0x884b)
(def depth-writemask 0xb72)
(def depth24-stencil8 0x88f0)
(def depth32f-stencil8 0x8cad)
(def difference 0x929e)
(def discrete-amd 0x9006)
(def dispatch-indirect-buffer 0x90ee)
(def dispatch-indirect-buffer-binding 0x90ef)
(def display-list 0x82e7)
(def dither 0xbd0)
(def domain 0xa02)
(def dont-care 0x1100)
(def dot-product-affine-depth-replace-nv 0x885d)
(def dot-product-const-eye-reflect-cube-map-nv 0x86f3)
(def dot-product-depth-replace-nv 0x86ed)
(def dot-product-diffuse-cube-map-nv 0x86f1)
(def dot-product-nv 0x86ec)
(def dot-product-pass-through-nv 0x885b)
(def dot-product-reflect-cube-map-nv 0x86f2)
(def dot-product-texture-1d-nv 0x885c)
(def dot-product-texture-2d-nv 0x86ee)
(def dot-product-texture-3d-nv 0x86ef)
(def dot-product-texture-cube-map-nv 0x86f0)
(def dot-product-texture-rectangle-nv 0x864e)
(def dot3-rgb 0x86ae)
(def dot3-rgba 0x86af)
(def double 0x140a)
(def double-mat2 0x8f46)
(def double-mat2x3 0x8f49)
(def double-mat2x4 0x8f4a)
(def double-mat3 0x8f47)
(def double-mat3x2 0x8f4b)
(def double-mat3x4 0x8f4c)
(def double-mat4 0x8f48)
(def double-mat4x2 0x8f4d)
(def double-mat4x3 0x8f4e)
(def double-vec2 0x8ffc)
(def double-vec3 0x8ffd)
(def double-vec4 0x8ffe)
(def doublebuffer 0xc32)
(def draw-arrays-command-nv 0x3)
(def draw-arrays-instanced-command-nv 0x7)
(def draw-arrays-strip-command-nv 0x5)
(def draw-buffer 0xc01)
(def draw-buffer0 0x8825)
(def draw-buffer0-ati 0x8825)
(def draw-buffer1 0x8826)
(def draw-buffer1-ati 0x8826)
(def draw-buffer10 0x882f)
(def draw-buffer10-ati 0x882f)
(def draw-buffer11 0x8830)
(def draw-buffer11-ati 0x8830)
(def draw-buffer12 0x8831)
(def draw-buffer12-ati 0x8831)
(def draw-buffer13 0x8832)
(def draw-buffer13-ati 0x8832)
(def draw-buffer14 0x8833)
(def draw-buffer14-ati 0x8833)
(def draw-buffer15 0x8834)
(def draw-buffer15-ati 0x8834)
(def draw-buffer2 0x8827)
(def draw-buffer2-ati 0x8827)
(def draw-buffer3 0x8828)
(def draw-buffer3-ati 0x8828)
(def draw-buffer4 0x8829)
(def draw-buffer4-ati 0x8829)
(def draw-buffer5 0x882a)
(def draw-buffer5-ati 0x882a)
(def draw-buffer6 0x882b)
(def draw-buffer6-ati 0x882b)
(def draw-buffer7 0x882c)
(def draw-buffer7-ati 0x882c)
(def draw-buffer8 0x882d)
(def draw-buffer8-ati 0x882d)
(def draw-buffer9 0x882e)
(def draw-buffer9-ati 0x882e)
(def draw-elements-command-nv 0x2)
(def draw-elements-instanced-command-nv 0x6)
(def draw-elements-strip-command-nv 0x4)
(def draw-framebuffer 0x8ca9)
(def draw-framebuffer-binding 0x8ca6)
(def draw-indirect-address-nv 0x8f41)
(def draw-indirect-buffer 0x8f3f)
(def draw-indirect-buffer-binding 0x8f43)
(def draw-indirect-length-nv 0x8f42)
(def draw-indirect-unified-nv 0x8f40)
(def draw-pixel-token 0x705)
(def ds-bias-nv 0x8716)
(def ds-scale-nv 0x8710)
(def dsdt-mag-intensity-nv 0x86dc)
(def dsdt-mag-nv 0x86f6)
(def dsdt-mag-vib-nv 0x86f7)
(def dsdt-nv 0x86f5)
(def dsdt8-mag8-intensity8-nv 0x870b)
(def dsdt8-mag8-nv 0x870a)
(def dsdt8-nv 0x8709)
(def dst-alpha 0x304)
(def dst-color 0x306)
(def dt-bias-nv 0x8717)
(def dt-scale-nv 0x8711)
(def dynamic-copy 0x88ea)
(def dynamic-draw 0x88e8)
(def dynamic-read 0x88e9)
(def dynamic-storage-bit 0x100)
(def edge-flag 0xb43)
(def edge-flag-array 0x8079)
(def edge-flag-array-address-nv 0x8f26)
(def edge-flag-array-buffer-binding 0x889b)
(def edge-flag-array-length-nv 0x8f30)
(def edge-flag-array-pointer 0x8093)
(def edge-flag-array-stride 0x808c)
(def edgeflag-bit-pgi 0x40000)
(def effective-raster-samples-ext 0x932c)
(def element-address-command-nv 0x8)
(def element-array-address-nv 0x8f29)
(def element-array-barrier-bit 0x2)
(def element-array-buffer 0x8893)
(def element-array-buffer-binding 0x8895)
(def element-array-length-nv 0x8f33)
(def element-array-unified-nv 0x8f1f)
(def emboss-constant-nv 0x855e)
(def emboss-light-nv 0x855d)
(def emboss-map-nv 0x855f)
(def enable-bit 0x2000)
(def equal 0x202)
(def equiv 0x1509)
(def eval-2d-nv 0x86c0)
(def eval-bit 0x10000)
(def eval-fractional-tessellation-nv 0x86c5)
(def eval-triangular-2d-nv 0x86c1)
(def eval-vertex-attrib0-nv 0x86c6)
(def eval-vertex-attrib1-nv 0x86c7)
(def eval-vertex-attrib10-nv 0x86d0)
(def eval-vertex-attrib11-nv 0x86d1)
(def eval-vertex-attrib12-nv 0x86d2)
(def eval-vertex-attrib13-nv 0x86d3)
(def eval-vertex-attrib14-nv 0x86d4)
(def eval-vertex-attrib15-nv 0x86d5)
(def eval-vertex-attrib2-nv 0x86c8)
(def eval-vertex-attrib3-nv 0x86c9)
(def eval-vertex-attrib4-nv 0x86ca)
(def eval-vertex-attrib5-nv 0x86cb)
(def eval-vertex-attrib6-nv 0x86cc)
(def eval-vertex-attrib7-nv 0x86cd)
(def eval-vertex-attrib8-nv 0x86ce)
(def eval-vertex-attrib9-nv 0x86cf)
(def exclusion 0x92a0)
(def exp 0x800)
(def exp2 0x801)
(def extensions 0x1f03)
(def external-virtual-memory-buffer-amd 0x9160)
(def eye-linear 0x2400)
(def eye-plane 0x2502)
(def eye-plane-absolute-nv 0x855c)
(def eye-radial-nv 0x855b)
(def factor-max-amd 0x901d)
(def factor-min-amd 0x901c)
(def failure-nv 0x9030)
(def gl-false 0x0)
(def fastest 0x1101)
(def feedback 0x1c01)
(def feedback-buffer-pointer 0xdf0)
(def feedback-buffer-size 0xdf1)
(def feedback-buffer-type 0xdf2)
(def field-lower-nv 0x9023)
(def field-upper-nv 0x9022)
(def fill 0x1b02)
(def fill-rectangle-nv 0x933c)
(def filter 0x829a)
(def first-vertex-convention 0x8e4d)
(def first-vertex-convention-ext 0x8e4d)
(def fixed 0x140c)
(def fixed-only 0x891d)
(def float 0x1406)
(def float-32-unsigned-int-24-8-rev 0x8dad)
(def float-clear-color-value-nv 0x888d)
(def float-mat2 0x8b5a)
(def float-mat2-arb 0x8b5a)
(def float-mat2x3 0x8b65)
(def float-mat2x4 0x8b66)
(def float-mat3 0x8b5b)
(def float-mat3-arb 0x8b5b)
(def float-mat3x2 0x8b67)
(def float-mat3x4 0x8b68)
(def float-mat4 0x8b5c)
(def float-mat4-arb 0x8b5c)
(def float-mat4x2 0x8b69)
(def float-mat4x3 0x8b6a)
(def float-r-nv 0x8880)
(def float-r16-nv 0x8884)
(def float-r32-nv 0x8885)
(def float-rg-nv 0x8881)
(def float-rg16-nv 0x8886)
(def float-rg32-nv 0x8887)
(def float-rgb-nv 0x8882)
(def float-rgb16-nv 0x8888)
(def float-rgb32-nv 0x8889)
(def float-rgba-mode-nv 0x888e)
(def float-rgba-nv 0x8883)
(def float-rgba16-nv 0x888a)
(def float-rgba32-nv 0x888b)
(def float-vec2 0x8b50)
(def float-vec2-arb 0x8b50)
(def float-vec3 0x8b51)
(def float-vec3-arb 0x8b51)
(def float-vec4 0x8b52)
(def float-vec4-arb 0x8b52)
(def float16-nv 0x8ff8)
(def float16-vec2-nv 0x8ff9)
(def float16-vec3-nv 0x8ffa)
(def float16-vec4-nv 0x8ffb)
(def fog 0xb60)
(def fog-bit 0x80)
(def fog-color 0xb66)
(def fog-coord 0x8451)
(def fog-coord-array 0x8457)
(def fog-coord-array-address-nv 0x8f28)
(def fog-coord-array-buffer-binding 0x889d)
(def fog-coord-array-length-nv 0x8f32)
(def fog-coord-array-pointer 0x8456)
(def fog-coord-array-stride 0x8455)
(def fog-coord-array-type 0x8454)
(def fog-coord-src 0x8450)
(def fog-coordinate 0x8451)
(def fog-coordinate-array 0x8457)
(def fog-coordinate-array-buffer-binding 0x889d)
(def fog-coordinate-array-pointer 0x8456)
(def fog-coordinate-array-stride 0x8455)
(def fog-coordinate-array-type 0x8454)
(def fog-coordinate-source 0x8450)
(def fog-density 0xb62)
(def fog-distance-mode-nv 0x855a)
(def fog-end 0xb64)
(def fog-hint 0xc54)
(def fog-index 0xb61)
(def fog-mode 0xb65)
(def fog-start 0xb63)
(def force-blue-to-one-nv 0x8860)
(def format-subsample-24-24-oml 0x8982)
(def format-subsample-244-244-oml 0x8983)
(def fractional-even 0x8e7c)
(def fractional-odd 0x8e7b)
(def fragment-color-ext 0x834c)
(def fragment-coverage-color-nv 0x92de)
(def fragment-coverage-to-color-nv 0x92dd)
(def fragment-depth 0x8452)
(def fragment-interpolation-offset-bits 0x8e5d)
(def fragment-material-ext 0x8349)
(def fragment-normal-ext 0x834a)
(def fragment-program-arb 0x8804)
(def fragment-program-interpolation-offset-bits-nv 0x8e5d)
(def fragment-program-parameter-buffer-nv 0x8da4)
(def fragment-shader 0x8b30)
(def fragment-shader-bit 0x2)
(def fragment-shader-derivative-hint 0x8b8b)
(def fragment-shader-invocations-arb 0x82f4)
(def fragment-subroutine 0x92ec)
(def fragment-subroutine-uniform 0x92f2)
(def fragment-texture 0x829f)
(def framebuffer 0x8d40)
(def framebuffer-attachment-alpha-size 0x8215)
(def framebuffer-attachment-blue-size 0x8214)
(def framebuffer-attachment-color-encoding 0x8210)
(def framebuffer-attachment-component-type 0x8211)
(def framebuffer-attachment-depth-size 0x8216)
(def framebuffer-attachment-green-size 0x8213)
(def framebuffer-attachment-layered 0x8da7)
(def framebuffer-attachment-layered-arb 0x8da7)
(def framebuffer-attachment-layered-ext 0x8da7)
(def framebuffer-attachment-object-name 0x8cd1)
(def framebuffer-attachment-object-type 0x8cd0)
(def framebuffer-attachment-red-size 0x8212)
(def framebuffer-attachment-stencil-size 0x8217)
(def framebuffer-attachment-texture-3d-zoffset 0x8cd4)
(def framebuffer-attachment-texture-base-view-index-ovr 0x9632)
(def framebuffer-attachment-texture-cube-map-face 0x8cd3)
(def framebuffer-attachment-texture-layer 0x8cd4)
(def framebuffer-attachment-texture-layer-ext 0x8cd4)
(def framebuffer-attachment-texture-level 0x8cd2)
(def framebuffer-attachment-texture-num-views-ovr 0x9630)
(def framebuffer-barrier-bit 0x400)
(def framebuffer-binding 0x8ca6)
(def framebuffer-blend 0x828b)
(def framebuffer-complete 0x8cd5)
(def framebuffer-default 0x8218)
(def framebuffer-default-fixed-sample-locations 0x9314)
(def framebuffer-default-height 0x9311)
(def framebuffer-default-layers 0x9312)
(def framebuffer-default-samples 0x9313)
(def framebuffer-default-width 0x9310)
(def framebuffer-incomplete-attachment 0x8cd6)
(def framebuffer-incomplete-dimensions 0x8cd9)
(def framebuffer-incomplete-draw-buffer 0x8cdb)
(def framebuffer-incomplete-formats 0x8)
(def framebuffer-incomplete-layer-count-arb 0x8da9)
(def framebuffer-incomplete-layer-count-ext 0x8da9)
(def framebuffer-incomplete-layer-targets 0x8da8)
(def framebuffer-incomplete-layer-targets-arb 0x8da8)
(def framebuffer-incomplete-layer-targets-ext 0x8da8)
(def framebuffer-incomplete-missing-attachment 0x8cd7)
(def framebuffer-incomplete-multisample 0x8d56)
(def framebuffer-incomplete-read-buffer 0x8cdc)
(def framebuffer-programmable-sample-locations-arb 0x9342)
(def framebuffer-programmable-sample-locations-nv 0x9342)
(def framebuffer-renderable 0x8289)
(def framebuffer-renderable-layered 0x828a)
(def framebuffer-sample-location-pixel-grid-arb 0x9343)
(def framebuffer-sample-location-pixel-grid-nv 0x9343)
(def framebuffer-srgb 0x8db9)
(def framebuffer-srgb-capable 0x8dba)
(def framebuffer-undefined 0x8219)
(def framebuffer-unsupported 0x8cdd)
(def front 0x404)
(def front-and-back 0x408)
(def front-face 0xb46)
(def front-face-command-nv 0x12)
(def front-left 0x400)
(def front-right 0x401)
(def full-range-ext 0x87e1)
(def full-stipple-hint-pgi 0x1a219)
(def full-support 0x82b7)
(def func-add 0x8006)
(def func-reverse-subtract 0x800b)
(def func-subtract 0x800a)
(def generate-mipmap 0x8191)
(def generate-mipmap-hint 0x8192)
(def geometry-input-type 0x8917)
(def geometry-input-type-arb 0x8ddb)
(def geometry-input-type-ext 0x8ddb)
(def geometry-output-type 0x8918)
(def geometry-output-type-arb 0x8ddc)
(def geometry-output-type-ext 0x8ddc)
(def geometry-program-nv 0x8c26)
(def geometry-program-parameter-buffer-nv 0x8da3)
(def geometry-shader 0x8dd9)
(def geometry-shader-arb 0x8dd9)
(def geometry-shader-bit 0x4)
(def geometry-shader-invocations 0x887f)
(def geometry-shader-primitives-emitted-arb 0x82f3)
(def geometry-subroutine 0x92eb)
(def geometry-subroutine-uniform 0x92f1)
(def geometry-texture 0x829e)
(def geometry-vertices-out 0x8916)
(def geometry-vertices-out-arb 0x8dda)
(def geometry-vertices-out-ext 0x8dda)
(def gequal 0x206)
(def get-texture-image-format 0x8291)
(def get-texture-image-type 0x8292)
(def gl-2-bytes 0x1407)
(def gl-2d 0x600)
(def gl-3-bytes 0x1408)
(def gl-3d 0x601)
(def gl-3d-color 0x602)
(def gl-3d-color-texture 0x603)
(def gl-4-bytes 0x1409)
(def gl-422-average-ext 0x80ce)
(def gl-422-ext 0x80cc)
(def gl-422-rev-average-ext 0x80cf)
(def gl-422-rev-ext 0x80cd)
(def gl-4d-color-texture 0x604)
(def gpu-address-nv 0x8f34)
(def gpu-disjoint 0x8fbb)
(def gpu-memory-info-current-available-vidmem-nvx 0x9049)
(def gpu-memory-info-dedicated-vidmem-nvx 0x9047)
(def gpu-memory-info-evicted-memory-nvx 0x904b)
(def gpu-memory-info-eviction-count-nvx 0x904a)
(def gpu-memory-info-total-available-memory-nvx 0x9048)
(def greater 0x204)
(def green 0x1904)
(def green-bias 0xd19)
(def green-bits 0xd53)
(def green-integer 0x8d95)
(def green-max-clamp-ingr 0x8565)
(def green-min-clamp-ingr 0x8561)
(def green-scale 0xd18)
(def guilty-context-reset 0x8253)
(def half-apple 0x140b)
(def half-float 0x140)
(def hardlight 0x929b)
(def hi-bias-nv 0x8714)
(def hi-scale-nv 0x870e)
(def high-float 0x8df2)
(def high-int 0x8df5)
(def hilo-nv 0x86f4)
(def hilo16-nv 0x86f8)
(def hilo8-nv 0x885e)
(def hint-bit 0x8000)
(def histogram 0x8024)
(def histogram-alpha-size 0x802b)
(def histogram-blue-size 0x802a)
(def histogram-format 0x8027)
(def histogram-green-size 0x8029)
(def histogram-luminance-size 0x802c)
(def histogram-red-size 0x8028)
(def histogram-sink 0x802d)
(def histogram-width 0x8026)
(def hsl-color 0x92af)
(def hsl-hue 0x92ad)
(def hsl-luminosity 0x92b0)
(def hsl-saturation 0x92ae)
(def image-1d 0x904c)
(def image-1d-array 0x9052)
(def image-2d 0x904d)
(def image-2d-array 0x9053)
(def image-2d-multisample 0x9055)
(def image-2d-multisample-array 0x9056)
(def image-2d-rect 0x904f)
(def image-3d 0x904e)
(def image-binding-access 0x8f3e)
(def image-binding-format 0x906e)
(def image-binding-layer 0x8f3d)
(def image-binding-layered 0x8f3c)
(def image-binding-level 0x8f3b)
(def image-binding-name 0x8f3a)
(def image-buffer 0x9051)
(def image-class-1-x-16 0x82be)
(def image-class-1-x-32 0x82bb)
(def image-class-1-x-8 0x82c1)
(def image-class-10-10-10-2 0x82c3)
(def image-class-11-11-10 0x82c2)
(def image-class-2-x-16 0x82bd)
(def image-class-2-x-32 0x82ba)
(def image-class-2-x-8 0x82c0)
(def image-class-4-x-16 0x82bc)
(def image-class-4-x-32 0x82b9)
(def image-class-4-x-8 0x82bf)
(def image-compatibility-class 0x82a8)
(def image-cube 0x9050)
(def image-cube-map-array 0x9054)
(def image-format-compatibility-by-class 0x90c9)
(def image-format-compatibility-by-size 0x90c8)
(def image-format-compatibility-type 0x90c7)
(def image-pixel-format 0x82a9)
(def image-pixel-type 0x82aa)
(def image-texel-size 0x82a7)
(def implementation-color-read-format 0x8b9b)
(def implementation-color-read-type 0x8b9a)
(def incr 0x1e02)
(def incr-wrap 0x8507)
(def index 0x8222)
(def index-array 0x8077)
(def index-array-address-nv 0x8f24)
(def index-array-buffer-binding 0x8899)
(def index-array-length-nv 0x8f2e)
(def index-array-pointer 0x8091)
(def index-array-stride 0x8086)
(def index-array-type 0x8085)
(def index-bit-pgi 0x80000)
(def index-bits 0xd51)
(def index-clear-value 0xc20)
(def index-logic-op 0xbf1)
(def index-material-ext 0x81b8)
(def index-material-face-ext 0x81ba)
(def index-material-parameter-ext 0x81b9)
(def index-mode 0xc30)
(def index-offset 0xd13)
(def index-shift 0xd12)
(def index-test-ext 0x81b5)
(def index-test-func-ext 0x81b6)
(def index-test-ref-ext 0x81b7)
(def index-writemask 0xc21)
(def info-log-length 0x8b84)
(def innocent-context-reset 0x8254)
(def int 0x1404)
(def int-10-10-10-2 0x8df7)
(def int-2-10-10-10-rev 0x8d9f)
(def int-image-1d 0x9057)
(def int-image-1d-array 0x905d)
(def int-image-2d 0x9058)
(def int-image-2d-array 0x905e)
(def int-image-2d-multisample 0x9060)
(def int-image-2d-multisample-array 0x9061)
(def int-image-2d-rect 0x905a)
(def int-image-3d 0x9059)
(def int-image-buffer 0x905c)
(def int-image-cube 0x905b)
(def int-image-cube-map-array 0x905f)
(def int-sampler-1d 0x8dc9)
(def int-sampler-1d-array 0x8dce)
(def int-sampler-2d 0x8dca)
(def int-sampler-2d-array 0x8dcf)
(def int-sampler-2d-multisample 0x9109)
(def int-sampler-2d-multisample-array 0x910c)
(def int-sampler-2d-rect 0x8dcd)
(def int-sampler-3d 0x8dcb)
(def int-sampler-buffer 0x8dd0)
(def int-sampler-buffer-amd 0x9002)
(def int-sampler-cube 0x8dcc)
(def int-sampler-cube-map-array 0x900e)
(def int-sampler-renderbuffer-nv 0x8e57)
(def int-vec2 0x8b53)
(def int-vec2-arb 0x8b53)
(def int-vec3 0x8b54)
(def int-vec3-arb 0x8b54)
(def int-vec4 0x8b55)
(def int-vec4-arb 0x8b55)
(def int16-nv 0x8fe4)
(def int16-vec2-nv 0x8fe5)
(def int16-vec3-nv 0x8fe6)
(def int16-vec4-nv 0x8fe7)
(def int64-arb 0x140e)
(def int64-nv 0x140e)
(def int64-vec2-arb 0x8fe9)
(def int64-vec2-nv 0x8fe9)
(def int64-vec3-arb 0x8fea)
(def int64-vec3-nv 0x8fea)
(def int64-vec4-arb 0x8feb)
(def int64-vec4-nv 0x8feb)
(def int8-nv 0x8fe0)
(def int8-vec2-nv 0x8fe1)
(def int8-vec3-nv 0x8fe2)
(def int8-vec4-nv 0x8fe3)
(def intensity 0x8049)
(def intensity-float16-apple 0x881d)
(def intensity-float16-ati 0x881d)
(def intensity-float32-apple 0x8817)
(def intensity-float32-ati 0x8817)
(def intensity-snorm 0x9013)
(def intensity12 0x804c)
(def intensity16 0x804d)
(def intensity16-snorm 0x901b)
(def intensity16f 0x881d)
(def intensity16i 0x8d8b)
(def intensity16ui 0x8d79)
(def intensity32f 0x8817)
(def intensity32i 0x8d85)
(def intensity32ui 0x8d73)
(def intensity4 0x804a)
(def intensity8 0x804b)
(def intensity8-snorm 0x9017)
(def intensity8i 0x8d91)
(def intensity8ui 0x8d7f)
(def interlace-oml 0x8980)
(def interlace-read-ingr 0x8568)
(def interlace-read-oml 0x8981)
(def interleaved-attribs 0x8c8c)
(def internalformat-alpha-size 0x8274)
(def internalformat-alpha-type 0x827b)
(def internalformat-blue-size 0x8273)
(def internalformat-blue-type 0x827a)
(def internalformat-depth-size 0x8275)
(def internalformat-depth-type 0x827c)
(def internalformat-green-size 0x8272)
(def internalformat-green-type 0x8279)
(def internalformat-preferred 0x8270)
(def internalformat-red-size 0x8271)
(def internalformat-red-type 0x8278)
(def internalformat-shared-size 0x8277)
(def internalformat-stencil-size 0x8276)
(def internalformat-stencil-type 0x827d)
(def internalformat-supported 0x826f)
(def interpolate 0x8575)
(def invalid-enum 0x500)
(def invalid-framebuffer-operation 0x506)
(def invalid-operation 0x502)
(def invalid-value 0x501)
(def invariant-datatype-ext 0x87eb)
(def invariant-ext 0x87c2)
(def invariant-value-ext 0x87ea)
(def invert 0x150a)
(def inverted-screen-w-rend 0x8491)
(def is-per-patch 0x92e7)
(def is-row-major 0x9300)
(def isolines 0x8e7a)
(def iui-n3f-v2f-ext 0x81af)
(def iui-n3f-v3f-ext 0x81b0)
(def iui-v2f-ext 0x81ad)
(def iui-v3f-ext 0x81ae)
(def keep 0x1e00)
(def last-vertex-convention 0x8e4e)
(def last-vertex-convention-ext 0x8e4e)
(def last-video-capture-status-nv 0x9027)
(def layer-provoking-vertex 0x825e)
(def layout-default-intel 0x0)
(def layout-linear-cpu-cached-intel 0x2)
(def layout-linear-intel 0x1)
(def left 0x406)
(def lequal 0x203)
(def less 0x201)
(def light-model-ambient 0xb53)
(def light-model-color-control 0x81f8)
(def light-model-local-viewer 0xb51)
(def light-model-specular-vector-apple 0x85b0)
(def light-model-two-side 0xb52)
(def lighten 0x9298)
(def lighting-bit 0x40)
(def line 0x1b01)
(def line-bit 0x4)
(def line-loop 0x2)
(def line-reset-token 0x707)
(def line-smooth 0x0)
(def line-smooth-hint 0x0)
(def line-stipple 0xb24)
(def line-stipple-pattern 0xb25)
(def line-stipple-repeat 0xb26)
(def line-strip 0x3)
(def line-strip-adjacency 0xb)
(def line-strip-adjacency-arb 0xb)
(def line-strip-adjacency-ext 0xb)
(def line-token 0x702)
(def line-width 0xb21)
(def line-width-command-nv 0xd)
(def line-width-granularity 0xb23)
(def line-width-range 0xb22)
(def linear 0x2601)
(def linear-mipmap-linear 0x2703)
(def linear-mipmap-nearest 0x2701)
(def lines 0x1)
(def lines-adjacency 0xa)
(def lines-adjacency-arb 0xa)
(def lines-adjacency-ext 0xa)
(def link-status 0x8b82)
(def list-base 0xb32)
(def list-bit 0x20000)
(def list-index 0xb33)
(def list-mode 0xb30)
(def lo-bias-nv 0x8715)
(def lo-scale-nv 0x870f)
(def load 0x101)
(def local-constant-datatype-ext 0x87ed)
(def local-constant-ext 0x87c3)
(def local-constant-value-ext 0x87ec)
(def local-ext 0x87c4)
(def location 0x930e)
(def location-component 0x934a)
(def location-index 0x930f)
(def logic-op 0xbf1)
(def logic-op-mode 0x0)
(def lose-context-on-reset 0x8252)
(def low-float 0x8df0)
(def low-int 0x8df3)
(def lower-left 0x8ca1)
(def luminance 0x1909)
(def luminance-alpha 0x190a)
(def luminance-alpha-float16-apple 0x881f)
(def luminance-alpha-float16-ati 0x881f)
(def luminance-alpha-float32-apple 0x8819)
(def luminance-alpha-float32-ati 0x8819)
(def luminance-alpha-integer 0x8d9d)
(def luminance-alpha-snorm 0x9012)
(def luminance-alpha16f 0x881f)
(def luminance-alpha16i 0x8d8d)
(def luminance-alpha16ui 0x8d7b)
(def luminance-alpha32f 0x8819)
(def luminance-alpha32i 0x8d87)
(def luminance-alpha32ui 0x8d75)
(def luminance-alpha8i 0x8d93)
(def luminance-alpha8ui 0x8d81)
(def luminance-float16-apple 0x881e)
(def luminance-float16-ati 0x881e)
(def luminance-float32-apple 0x8818)
(def luminance-float32-ati 0x8818)
(def luminance-integer 0x8d9c)
(def luminance-snorm 0x9011)
(def luminance12 0x8041)
(def luminance12-alpha12 0x8047)
(def luminance12-alpha4 0x8046)
(def luminance16 0x8042)
(def luminance16-alpha16 0x8048)
(def luminance16-alpha16-snorm 0x901a)
(def luminance16-snorm 0x9019)
(def luminance16f 0x881e)
(def luminance16i 0x8d8c)
(def luminance16ui 0x8d7a)
(def luminance32f 0x8818)
(def luminance32i 0x8d86)
(def luminance32ui 0x8d74)
(def luminance4 0x803f)
(def luminance4-alpha4 0x8043)
(def luminance6-alpha2 0x8044)
(def luminance8 0x8040)
(def luminance8-alpha8 0x8045)
(def luminance8-alpha8-snorm 0x9016)
(def luminance8-snorm 0x9015)
(def luminance8i 0x8d92)
(def luminance8ui 0x8d80)
(def magnitude-bias-nv 0x8718)
(def magnitude-scale-nv 0x8712)
(def major-version 0x821b)
(def manual-generate-mipmap 0x8294)
(def map-attrib-u-order-nv 0x86c3)
(def map-attrib-v-order-nv 0x86c4)
(def map-coherent-bit 0x80)
(def map-color 0xd10)
(def map-flush-explicit-bit 0x10)
(def map-invalidate-buffer-bit 0x8)
(def map-invalidate-range-bit 0x4)
(def map-persistent-bit 0x40)
(def map-read-bit 0x1)
(def map-stencil 0xd11)
(def map-tessellation-nv 0x86c2)
(def map-unsynchronized-bit 0x20)
(def map-write-bit 0x2)
(def map1-color-4 0xd90)
(def map1-grid-domain 0xdd0)
(def map1-grid-segments 0xdd1)
(def map1-index 0xd91)
(def map1-normal 0xd92)
(def map1-texture-coord-1 0xd93)
(def map1-texture-coord-2 0xd94)
(def map1-texture-coord-3 0xd95)
(def map1-texture-coord-4 0xd96)
(def map1-vertex-3 0xd97)
(def map1-vertex-4 0xd98)
(def map2-color-4 0xdb0)
(def map2-grid-domain 0xdd2)
(def map2-grid-segments 0xdd3)
(def map2-index 0xdb1)
(def map2-normal 0xdb2)
(def map2-texture-coord-1 0xdb3)
(def map2-texture-coord-2 0xdb4)
(def map2-texture-coord-3 0xdb5)
(def map2-texture-coord-4 0xdb6)
(def map2-vertex-3 0xdb7)
(def map2-vertex-4 0xdb8)
(def mat-ambient-and-diffuse-bit-pgi 0x200000)
(def mat-ambient-bit-pgi 0x100000)
(def mat-color-indexes-bit-pgi 0x1000000)
(def mat-diffuse-bit-pgi 0x400000)
(def mat-emission-bit-pgi 0x800000)
(def mat-shininess-bit-pgi 0x2000000)
(def mat-specular-bit-pgi 0x4000000)
(def material-side-hint-pgi 0x1a22c)
(def matrix-ext 0x87c0)
(def matrix-index-array-arb 0x8844)
(def matrix-index-array-pointer-arb 0x8849)
(def matrix-index-array-size-arb 0x8846)
(def matrix-index-array-stride-arb 0x8848)
(def matrix-index-array-type-arb 0x8847)
(def matrix-palette-arb 0x8840)
(def matrix-stride 0x92ff)
(def matrix0-arb 0x88c0)
(def matrix1-arb 0x88c1)
(def matrix10-arb 0x88ca)
(def matrix11-arb 0x88cb)
(def matrix12-arb 0x88cc)
(def matrix13-arb 0x88cd)
(def matrix14-arb 0x88ce)
(def matrix15-arb 0x88cf)
(def matrix16-arb 0x88d0)
(def matrix17-arb 0x88d1)
(def matrix18-arb 0x88d2)
(def matrix19-arb 0x88d3)
(def matrix2-arb 0x88c2)
(def matrix20-arb 0x88d4)
(def matrix21-arb 0x88d5)
(def matrix22-arb 0x88d6)
(def matrix23-arb 0x88d7)
(def matrix24-arb 0x88d8)
(def matrix25-arb 0x88d9)
(def matrix26-arb 0x88da)
(def matrix27-arb 0x88db)
(def matrix28-arb 0x88dc)
(def matrix29-arb 0x88dd)
(def matrix3-arb 0x88c3)
(def matrix30-arb 0x88de)
(def matrix31-arb 0x88df)
(def matrix4-arb 0x88c4)
(def matrix5-arb 0x88c5)
(def matrix6-arb 0x88c6)
(def matrix7-arb 0x88c7)
(def matrix8-arb 0x88c8)
(def matrix9-arb 0x88c9)
(def max 0x8008)
(def max-3d-texture-size 0x8073)
(def max-array-texture-layers 0x88ff)
(def max-atomic-counter-buffer-bindings 0x92dc)
(def max-atomic-counter-buffer-size 0x92d8)
(def max-attrib-stack-depth 0xd35)
(def max-bindable-uniform-size-ext 0x8ded)
(def max-client-attrib-stack-depth 0xd3b)
(def max-clip-distances 0xd32)
(def max-clip-planes 0xd32)
(def max-color-attachments 0x8cdf)
(def max-color-matrix-stack-depth 0x80b3)
(def max-color-texture-samples 0x910e)
(def max-combined-atomic-counter-buffers 0x92d1)
(def max-combined-atomic-counters 0x92d7)
(def max-combined-clip-and-cull-distances 0x82fa)
(def max-combined-compute-uniform-components 0x8266)
(def max-combined-dimensions 0x8282)
(def max-combined-fragment-uniform-components 0x8a33)
(def max-combined-geometry-uniform-components 0x8a32)
(def max-combined-image-uniforms 0x90cf)
(def max-combined-image-units-and-fragment-outputs 0x8f39)
(def max-combined-shader-output-resources 0x8f39)
(def max-combined-shader-storage-blocks 0x90dc)
(def max-combined-tess-control-uniform-components 0x8e1e)
(def max-combined-tess-evaluation-uniform-components 0x8e1f)
(def max-combined-texture-image-units 0x8b4d)
(def max-combined-uniform-blocks 0x8a2e)
(def max-combined-vertex-uniform-components 0x8a31)
(def max-compute-atomic-counter-buffers 0x8264)
(def max-compute-atomic-counters 0x8265)
(def max-compute-fixed-group-invocations-arb 0x90eb)
(def max-compute-fixed-group-size-arb 0x91bf)
(def max-compute-image-uniforms 0x91bd)
(def max-compute-shader-storage-blocks 0x90db)
(def max-compute-shared-memory-size 0x8262)
(def max-compute-texture-image-units 0x91bc)
(def max-compute-uniform-blocks 0x91bb)
(def max-compute-uniform-components 0x8263)
(def max-compute-variable-group-invocations-arb 0x9344)
(def max-compute-variable-group-size-arb 0x9345)
(def max-compute-work-group-count 0x91be)
(def max-compute-work-group-invocations 0x90eb)
(def max-compute-work-group-size 0x91bf)
(def max-convolution-height 0x801b)
(def max-convolution-width 0x801a)
(def max-cube-map-texture-size 0x851c)
(def max-cull-distances 0x82f9)
(def max-debug-group-stack-depth 0x826c)
(def max-debug-logged-messages 0x9144)
(def max-debug-logged-messages-amd 0x9144)
(def max-debug-message-length 0x9143)
(def max-debug-message-length-amd 0x9143)
(def max-deep-3d-texture-depth-nv 0x90d1)
(def max-deep-3d-texture-width-height-nv 0x90d0)
(def max-depth 0x8280)
(def max-depth-texture-samples 0x910f)
(def max-draw-buffers 0x8824)
(def max-draw-buffers-ati 0x8824)
(def max-dual-source-draw-buffers 0x88fc)
(def max-element-index 0x8d6b)
(def max-elements-indices 0x80e9)
(def max-elements-vertices 0x80e8)
(def max-eval-order 0xd30)
(def max-fragment-atomic-counter-buffers 0x92d0)
(def max-fragment-atomic-counters 0x92d6)
(def max-fragment-bindable-uniforms-ext 0x8de3)
(def max-fragment-image-uniforms 0x90ce)
(def max-fragment-input-components 0x9125)
(def max-fragment-interpolation-offset 0x8e5c)
(def max-fragment-interpolation-offset-nv 0x8e5c)
(def max-fragment-shader-storage-blocks 0x90da)
(def max-fragment-uniform-blocks 0x8a2d)
(def max-fragment-uniform-components 0x8b49)
(def max-fragment-uniform-vectors 0x8dfd)
(def max-framebuffer-height 0x9316)
(def max-framebuffer-layers 0x9317)
(def max-framebuffer-samples 0x9318)
(def max-framebuffer-width 0x9315)
(def max-geometry-atomic-counter-buffers 0x92cf)
(def max-geometry-atomic-counters 0x92d5)
(def max-geometry-bindable-uniforms-ext 0x8de4)
(def max-geometry-image-uniforms 0x90cd)
(def max-geometry-input-components 0x9123)
(def max-geometry-output-components 0x9124)
(def max-geometry-output-vertices 0x8de0)
(def max-geometry-output-vertices-arb 0x8de0)
(def max-geometry-program-invocations-nv 0x8e5a)
(def max-geometry-shader-invocations 0x8e5a)
(def max-geometry-shader-storage-blocks 0x90d7)
(def max-geometry-texture-image-units 0x8c29)
(def max-geometry-texture-image-units-arb 0x8c29)
(def max-geometry-texture-image-units-ext 0x8c29)
(def max-geometry-total-output-components 0x8de1)
(def max-geometry-total-output-components-arb 0x8de1)
(def max-geometry-uniform-blocks 0x8a2c)
(def max-geometry-uniform-components 0x8ddf)
(def max-geometry-uniform-components-arb 0x8ddf)
(def max-geometry-varying-components-arb 0x8ddd)
(def max-height 0x827f)
(def max-image-samples 0x906d)
(def max-image-units 0x8f38)
(def max-integer-samples 0x9110)
(def max-label-length 0x82e8)
(def max-layers 0x8281)
(def max-lights 0xd31)
(def max-list-nesting 0xb31)
(def max-map-tessellation-nv 0x86d6)
(def max-matrix-palette-stack-depth-arb 0x8841)
(def max-modelview-stack-depth 0xd36)
(def max-multisample-coverage-modes-nv 0x8e11)
(def max-name-length 0x92f6)
(def max-name-stack-depth 0xd37)
(def max-num-active-variables 0x92f7)
(def max-num-compatible-subroutines 0x92f8)
(def max-optimized-vertex-shader-instructions-ext 0x87ca)
(def max-optimized-vertex-shader-invariants-ext 0x87cd)
(def max-optimized-vertex-shader-local-constants-ext 0x87cc)
(def max-optimized-vertex-shader-locals-ext 0x87ce)
(def max-optimized-vertex-shader-variants-ext 0x87cb)
(def max-palette-matrices-arb 0x8842)
(def max-patch-vertices 0x8e7d)
(def max-pixel-map-table 0xd34)
(def max-pixel-transform-2d-stack-depth-ext 0x8337)
(def max-pn-triangles-tesselation-level-ati 0x87f1)
(def max-program-address-registers-arb 0x88b1)
(def max-program-alu-instructions-arb 0x880b)
(def max-program-attrib-components-nv 0x8908)
(def max-program-attribs-arb 0x88ad)
(def max-program-env-parameters-arb 0x88b5)
(def max-program-generic-attribs-nv 0x8da5)
(def max-program-generic-results-nv 0x8da6)
(def max-program-instructions-arb 0x88a1)
(def max-program-local-parameters-arb 0x88b4)
(def max-program-matrices-arb 0x862f)
(def max-program-matrix-stack-depth-arb 0x862e)
(def max-program-native-address-registers-arb 0x88b3)
(def max-program-native-alu-instructions-arb 0x880e)
(def max-program-native-attribs-arb 0x88af)
(def max-program-native-instructions-arb 0x88a3)
(def max-program-native-parameters-arb 0x88ab)
(def max-program-native-temporaries-arb 0x88a7)
(def max-program-native-tex-indirections-arb 0x8810)
(def max-program-native-tex-instructions-arb 0x880f)
(def max-program-output-vertices-nv 0x8c27)
(def max-program-parameter-buffer-bindings-nv 0x8da0)
(def max-program-parameter-buffer-size-nv 0x8da1)
(def max-program-parameters-arb 0x88a9)
(def max-program-patch-attribs-nv 0x86d8)
(def max-program-result-components-nv 0x8909)
(def max-program-subroutine-num-nv 0x8f45)
(def max-program-subroutine-parameters-nv 0x8f44)
(def max-program-temporaries-arb 0x88a5)
(def max-program-tex-indirections-arb 0x880d)
(def max-program-tex-instructions-arb 0x880c)
(def max-program-texel-offset 0x8905)
(def max-program-texel-offset-nv 0x8905)
(def max-program-texture-gather-components 0x8f9f)
(def max-program-texture-gather-offset 0x8e5f)
(def max-program-texture-gather-offset-nv 0x8e5f)
(def max-program-total-output-components-nv 0x8c28)
(def max-projection-stack-depth 0xd38)
(def max-raster-samples-ext 0x9329)
(def max-rational-eval-order-nv 0x86d7)
(def max-rectangle-texture-size 0x84f8)
(def max-rectangle-texture-size-arb 0x84f8)
(def max-renderbuffer-size 0x84e8)
(def max-sample-mask-words 0x8e59)
(def max-sample-mask-words-nv 0x8e59)
(def max-samples 0x8d57)
(def max-server-wait-timeout 0x9111)
(def max-shader-buffer-address-nv 0x8f35)
(def max-shader-compiler-threads-arb 0x91b0)
(def max-shader-storage-block-size 0x90de)
(def max-shader-storage-buffer-bindings 0x90dd)
(def max-shininess-nv 0x8504)
(def max-sparse-3d-texture-size-amd 0x9199)
(def max-sparse-3d-texture-size-arb 0x9199)
(def max-sparse-array-texture-layers 0x919a)
(def max-sparse-array-texture-layers-arb 0x919a)
(def max-sparse-texture-size-amd 0x9198)
(def max-sparse-texture-size-arb 0x9198)
(def max-spot-exponent-nv 0x8505)
(def max-subpixel-precision-bias-bits-nv 0x9349)
(def max-subroutine-uniform-locations 0x8de8)
(def max-subroutines 0x8de7)
(def max-tess-control-atomic-counter-buffers 0x92cd)
(def max-tess-control-atomic-counters 0x92d3)
(def max-tess-control-image-uniforms 0x90cb)
(def max-tess-control-input-components 0x886c)
(def max-tess-control-output-components 0x8e83)
(def max-tess-control-shader-storage-blocks 0x90d8)
(def max-tess-control-texture-image-units 0x8e81)
(def max-tess-control-total-output-components 0x8e85)
(def max-tess-control-uniform-blocks 0x8e89)
(def max-tess-control-uniform-components 0x8e7f)
(def max-tess-evaluation-atomic-counter-buffers 0x92ce)
(def max-tess-evaluation-atomic-counters 0x92d4)
(def max-tess-evaluation-image-uniforms 0x90cc)
(def max-tess-evaluation-input-components 0x886d)
(def max-tess-evaluation-output-components 0x8e86)
(def max-tess-evaluation-shader-storage-blocks 0x90d9)
(def max-tess-evaluation-texture-image-units 0x8e82)
(def max-tess-evaluation-uniform-blocks 0x8e8a)
(def max-tess-evaluation-uniform-components 0x8e80)
(def max-tess-gen-level 0x8e7e)
(def max-tess-patch-components 0x8e84)
(def max-texture-buffer-size 0x8c2b)
(def max-texture-coords 0x8871)
(def max-texture-coords-arb 0x8871)
(def max-texture-image-units 0x8872)
(def max-texture-image-units-arb 0x8872)
(def max-texture-lod-bias 0x84fd)
(def max-texture-max-anisotropy-ext 0x84ff)
(def max-texture-size 0xd33)
(def max-texture-stack-depth 0xd39)
(def max-texture-units 0x84e2)
(def max-transform-feedback-buffers 0x8e70)
(def max-transform-feedback-interleaved-components 0x8c8a)
(def max-transform-feedback-separate-attribs 0x8c8b)
(def max-transform-feedback-separate-components 0x8c80)
(def max-uniform-block-size 0x8a30)
(def max-uniform-buffer-bindings 0x8a2f)
(def max-uniform-locations 0x826e)
(def max-varying-components 0x8b4b)
(def max-varying-floats 0x8b4b)
(def max-varying-vectors 0x8dfc)
(def max-vertex-atomic-counter-buffers 0x92cc)
(def max-vertex-atomic-counters 0x92d2)
(def max-vertex-attrib-bindings 0x82da)
(def max-vertex-attrib-relative-offset 0x82d9)
(def max-vertex-attrib-stride 0x82e5)
(def max-vertex-attribs 0x8869)
(def max-vertex-attribs-arb 0x8869)
(def max-vertex-bindable-uniforms-ext 0x8de2)
(def max-vertex-hint-pgi 0x1a22d)
(def max-vertex-image-uniforms 0x90ca)
(def max-vertex-output-components 0x9122)
(def max-vertex-shader-instructions-ext 0x87c5)
(def max-vertex-shader-invariants-ext 0x87c7)
(def max-vertex-shader-local-constants-ext 0x87c8)
(def max-vertex-shader-locals-ext 0x87c9)
(def max-vertex-shader-storage-blocks 0x90d6)
(def max-vertex-shader-variants-ext 0x87c6)
(def max-vertex-streams 0x8e71)
(def max-vertex-texture-image-units 0x8b4c)
(def max-vertex-uniform-blocks 0x8a2b)
(def max-vertex-uniform-components 0x8b4a)
(def max-vertex-uniform-vectors 0x8dfb)
(def max-vertex-units-arb 0x86a4)
(def max-vertex-varying-components-arb 0x8dde)
(def max-viewport-dims 0xd3a)
(def max-viewports 0x825b)
(def max-views-ovr 0x9631)
(def max-width 0x827e)
(def medium-float 0x8df1)
(def medium-int 0x8df4)
(def min 0x8007)
(def min-fragment-interpolation-offset 0x8e5b)
(def min-fragment-interpolation-offset-nv 0x8e5b)
(def min-lod-warning-amd 0x919c)
(def min-map-buffer-alignment 0x90bc)
(def min-program-texel-offset 0x8904)
(def min-program-texel-offset-nv 0x8904)
(def min-program-texture-gather-offset 0x8e5e)
(def min-program-texture-gather-offset-nv 0x8e5e)
(def min-sample-shading-value 0x8c37)
(def min-sparse-level-amd 0x919b)
(def minmax 0x802e)
(def minmax-format 0x802f)
(def minmax-sink 0x8030)
(def minor-version 0x821c)
(def mipmap 0x8293)
(def mirror-clamp-ext 0x8742)
(def mirror-clamp-to-border-ext 0x8912)
(def mirror-clamp-to-edge 0x8743)
(def mirror-clamp-to-edge-ext 0x8743)
(def mirrored-repeat 0x8370)
(def mixed-depth-samples-supported-nv 0x932f)
(def mixed-stencil-samples-supported-nv 0x9330)
(def modelview-stack-depth 0xba3)
(def modelview0-arb 0x1700)
(def modelview0-ext 0x1700)
(def modelview0-matrix-ext 0xba6)
(def modelview0-stack-depth-ext 0xba3)
(def modelview1-arb 0x850a)
(def modelview1-ext 0x850a)
(def modelview1-matrix-ext 0x8506)
(def modelview1-stack-depth-ext 0x8502)
(def modelview10-arb 0x872a)
(def modelview11-arb 0x872b)
(def modelview12-arb 0x872c)
(def modelview13-arb 0x872d)
(def modelview14-arb 0x872e)
(def modelview15-arb 0x872f)
(def modelview16-arb 0x8730)
(def modelview17-arb 0x8731)
(def modelview18-arb 0x8732)
(def modelview19-arb 0x8733)
(def modelview2-arb 0x8722)
(def modelview20-arb 0x8734)
(def modelview21-arb 0x8735)
(def modelview22-arb 0x8736)
(def modelview23-arb 0x8737)
(def modelview24-arb 0x8738)
(def modelview25-arb 0x8739)
(def modelview26-arb 0x873a)
(def modelview27-arb 0x873b)
(def modelview28-arb 0x873c)
(def modelview29-arb 0x873d)
(def modelview3-arb 0x8723)
(def modelview30-arb 0x873e)
(def modelview31-arb 0x873f)
(def modelview4-arb 0x8724)
(def modelview5-arb 0x8725)
(def modelview6-arb 0x8726)
(def modelview7-arb 0x8727)
(def modelview8-arb 0x8728)
(def modelview9-arb 0x8729)
(def modulate 0x2100)
(def mult 0x103)
(def multiply 0x9294)
(def multisample 0x809)
(def multisample-bit 0x20000000)
(def multisample-coverage-modes-nv 0x8e12)
(def multisample-filter-hint-nv 0x8534)
(def multisample-line-width-granularity 0x9382)
(def multisample-line-width-range 0x9381)
(def multisample-rasterization-allowed-ext 0x932b)
(def mvp-matrix-ext 0x87e3)
(def n3f-v3f 0x2a25)
(def name-length 0x92f9)
(def name-stack-depth 0xd70)
(def named-string-length-arb 0x8de9)
(def named-string-type-arb 0x8dea)
(def nand 0x150)
(def native-graphics-begin-hint-pgi 0x1a203)
(def native-graphics-end-hint-pgi 0x1a204)
(def native-graphics-handle-pgi 0x1a202)
(def nearest 0x2600)
(def nearest-mipmap-linear 0x2702)
(def nearest-mipmap-nearest 0x2700)
(def negative-one-ext 0x87df)
(def negative-one-to-one 0x935e)
(def negative-w-ext 0x87dc)
(def negative-x-ext 0x87d9)
(def negative-y-ext 0x87da)
(def negative-z-ext 0x87db)
(def never 0x200)
(def next-video-capture-buffer-status-nv 0x9025)
(def nicest 0x1102)
(def no-error 0x0)
(def no-reset-notification 0x8261)
(def none 0x0)
(def noop 0x1505)
(def nop-command-nv 0x1)
(def nor 0x1508)
(def normal-array-address-nv 0x8f22)
(def normal-array-buffer-binding 0x8897)
(def normal-array-length-nv 0x8f2c)
(def normal-array-pointer 0x808f)
(def normal-array-stride 0x807f)
(def normal-array-type 0x807e)
(def normal-bit-pgi 0x8000000)
(def normal-map 0x8511)
(def normalized-range-ext 0x87e0)
(def notequal 0x205)
(def num-active-variables 0x9304)
(def num-compatible-subroutines 0x8e4a)
(def num-compressed-texture-formats 0x86a2)
(def num-extensions 0x821d)
(def num-program-binary-formats 0x87fe)
(def num-sample-counts 0x9380)
(def num-shader-binary-formats 0x8df9)
(def num-shading-language-versions 0x82e9)
(def num-sparse-levels-arb 0x91aa)
(def num-video-capture-streams-nv 0x9024)
(def num-virtual-page-sizes-arb 0x91a8)
(def object-active-uniform-max-length-arb 0x8b87)
(def object-active-uniforms-arb 0x8b86)
(def object-attached-objects-arb 0x8b85)
(def object-compile-status-arb 0x8b81)
(def object-delete-status-arb 0x8b80)
(def object-info-log-length-arb 0x8b84)
(def object-linear 0x2401)
(def object-link-status-arb 0x8b82)
(def object-plane 0x2501)
(def object-shader-source-length-arb 0x8b88)
(def object-subtype-arb 0x8b4f)
(def object-type 0x9112)
(def object-type-arb 0x8b4e)
(def object-validate-status-arb 0x8b83)
(def occlusion-query-event-mask-amd 0x874f)
(def offset 0x92fc)
(def offset-hilo-projective-texture-2d-nv 0x8856)
(def offset-hilo-projective-texture-rectangle-nv 0x8857)
(def offset-hilo-texture-2d-nv 0x8854)
(def offset-hilo-texture-rectangle-nv 0x8855)
(def offset-projective-texture-2d-nv 0x8850)
(def offset-projective-texture-2d-scale-nv 0x8851)
(def offset-projective-texture-rectangle-nv 0x8852)
(def offset-projective-texture-rectangle-scale-nv 0x8853)
(def offset-texture-2d-bias-nv 0x86e3)
(def offset-texture-2d-matrix-nv 0x86e1)
(def offset-texture-2d-nv 0x86e8)
(def offset-texture-2d-scale-nv 0x86e2)
(def offset-texture-bias-nv 0x86e3)
(def offset-texture-matrix-nv 0x86e1)
(def offset-texture-rectangle-nv 0x864c)
(def offset-texture-rectangle-scale-nv 0x864d)
(def offset-texture-scale-nv 0x86e2)
(def one 0x1)
(def one-ext 0x87de)
(def one-minus-constant-alpha 0x8004)
(def one-minus-constant-color 0x8002)
(def one-minus-dst-alpha 0x305)
(def one-minus-dst-color 0x307)
(def one-minus-src-alpha 0x303)
(def one-minus-src-color 0x301)
(def one-minus-src1-alpha 0x88fb)
(def one-minus-src1-color 0x88fa)
(def op-add-ext 0x8787)
(def op-clamp-ext 0x878e)
(def op-cross-product-ext 0x8797)
(def op-dot3-ext 0x8784)
(def op-dot4-ext 0x8785)
(def op-exp-base-2-ext 0x8791)
(def op-floor-ext 0x878f)
(def op-frac-ext 0x8789)
(def op-index-ext 0x8782)
(def op-log-base-2-ext 0x8792)
(def op-madd-ext 0x8788)
(def op-max-ext 0x878a)
(def op-min-ext 0x878b)
(def op-mov-ext 0x8799)
(def op-mul-ext 0x8786)
(def op-multiply-matrix-ext 0x8798)
(def op-negate-ext 0x8783)
(def op-power-ext 0x8793)
(def op-recip-ext 0x8794)
(def op-recip-sqrt-ext 0x8795)
(def op-round-ext 0x8790)
(def op-set-ge-ext 0x878c)
(def op-set-lt-ext 0x878d)
(def op-sub-ext 0x8796)
(def operand0-alpha 0x8598)
(def operand0-rgb 0x8590)
(def operand1-alpha 0x8599)
(def operand1-rgb 0x8591)
(def operand2-alpha 0x859a)
(def operand2-rgb 0x8592)
(def or 0x1507)
(def or-inverted 0x150)
(def or-reverse 0x150)
(def order 0xa01)
(def out-of-memory 0x505)
(def output-color0-ext 0x879b)
(def output-color1-ext 0x879c)
(def output-fog-ext 0x87bd)
(def output-texture-coord0-ext 0x879d)
(def output-texture-coord1-ext 0x879e)
(def output-texture-coord10-ext 0x87a7)
(def output-texture-coord11-ext 0x87a8)
(def output-texture-coord12-ext 0x87a9)
(def output-texture-coord13-ext 0x87aa)
(def output-texture-coord14-ext 0x87ab)
(def output-texture-coord15-ext 0x87ac)
(def output-texture-coord16-ext 0x87ad)
(def output-texture-coord17-ext 0x87ae)
(def output-texture-coord18-ext 0x87af)
(def output-texture-coord19-ext 0x87b0)
(def output-texture-coord2-ext 0x879f)
(def output-texture-coord20-ext 0x87b1)
(def output-texture-coord21-ext 0x87b2)
(def output-texture-coord22-ext 0x87b3)
(def output-texture-coord23-ext 0x87b4)
(def output-texture-coord24-ext 0x87b5)
(def output-texture-coord25-ext 0x87b6)
(def output-texture-coord26-ext 0x87b7)
(def output-texture-coord27-ext 0x87b8)
(def output-texture-coord28-ext 0x87b9)
(def output-texture-coord29-ext 0x87ba)
(def output-texture-coord3-ext 0x87a0)
(def output-texture-coord30-ext 0x87bb)
(def output-texture-coord31-ext 0x87bc)
(def output-texture-coord4-ext 0x87a1)
(def output-texture-coord5-ext 0x87a2)
(def output-texture-coord6-ext 0x87a3)
(def output-texture-coord7-ext 0x87a4)
(def output-texture-coord8-ext 0x87a5)
(def output-texture-coord9-ext 0x87a6)
(def output-vertex-ext 0x879a)
(def overlay 0x9296)
(def pack-alignment 0xd05)
(def pack-cmyk-hint-ext 0x800e)
(def pack-compressed-block-depth 0x912d)
(def pack-compressed-block-height 0x912c)
(def pack-compressed-block-size 0x912e)
(def pack-compressed-block-width 0x912b)
(def pack-image-height 0x806c)
(def pack-lsb-first 0xd01)
(def pack-resample-oml 0x8984)
(def pack-row-bytes-apple 0x8a15)
(def pack-row-length 0xd02)
(def pack-skip-images 0x806b)
(def pack-skip-pixels 0xd04)
(def pack-skip-rows 0xd03)
(def pack-swap-bytes 0xd00)
(def parameter-buffer-arb 0x80ee)
(def parameter-buffer-binding-arb 0x80ef)
(def partial-success-nv 0x902e)
(def pass-through-nv 0x86e6)
(def pass-through-token 0x700)
(def patch-default-inner-level 0x8e73)
(def patch-default-outer-level 0x8e74)
(def patch-vertices 0x8e72)
(def patches 0xe)
(def percentage-amd 0x8bc3)
(def perfmon-result-amd 0x8bc6)
(def perfmon-result-available-amd 0x8bc4)
(def perfmon-result-size-amd 0x8bc5)
(def performance-monitor-amd 0x9152)
(def perfquery-counter-data-bool32-intel 0x94fc)
(def perfquery-counter-data-double-intel 0x94fb)
(def perfquery-counter-data-float-intel 0x94fa)
(def perfquery-counter-data-uint32-intel 0x94f8)
(def perfquery-counter-data-uint64-intel 0x94f9)
(def perfquery-counter-desc-length-max-intel 0x94ff)
(def perfquery-counter-duration-norm-intel 0x94f1)
(def perfquery-counter-duration-raw-intel 0x94f2)
(def perfquery-counter-event-intel 0x94f0)
(def perfquery-counter-name-length-max-intel 0x94fe)
(def perfquery-counter-raw-intel 0x94f4)
(def perfquery-counter-throughput-intel 0x94f3)
(def perfquery-counter-timestamp-intel 0x94f5)
(def perfquery-donot-flush-intel 0x83f9)
(def perfquery-flush-intel 0x83fa)
(def perfquery-global-context-intel 0x1)
(def perfquery-gpa-extended-counters-intel 0x9500)
(def perfquery-query-name-length-max-intel 0x94fd)
(def perfquery-single-context-intel 0x0)
(def perfquery-wait-intel 0x83fb)
(def perspective-correction-hint 0xc50)
(def perturb-ext 0x85ae)
(def pixel-buffer-barrier-bit 0x80)
(def pixel-count-available-nv 0x8867)
(def pixel-count-nv 0x8866)
(def pixel-counter-bits-nv 0x8864)
(def pixel-cubic-weight-ext 0x8333)
(def pixel-mag-filter-ext 0x8331)
(def pixel-map-a-to-a 0xc79)
(def pixel-map-a-to-a-size 0xcb9)
(def pixel-map-b-to-b 0xc78)
(def pixel-map-b-to-b-size 0xcb8)
(def pixel-map-g-to-g 0xc77)
(def pixel-map-g-to-g-size 0xcb7)
(def pixel-map-i-to-a 0xc75)
(def pixel-map-i-to-a-size 0xcb5)
(def pixel-map-i-to-b 0xc74)
(def pixel-map-i-to-b-size 0xcb4)
(def pixel-map-i-to-g 0xc73)
(def pixel-map-i-to-g-size 0xcb3)
(def pixel-map-i-to-i 0xc70)
(def pixel-map-i-to-i-size 0xcb0)
(def pixel-map-i-to-r 0xc72)
(def pixel-map-i-to-r-size 0xcb2)
(def pixel-map-r-to-r 0xc76)
(def pixel-map-r-to-r-size 0xcb6)
(def pixel-map-s-to-s 0xc71)
(def pixel-map-s-to-s-size 0xcb1)
(def pixel-min-filter-ext 0x8332)
(def pixel-mode-bit 0x20)
(def pixel-pack-buffer 0x88eb)
(def pixel-pack-buffer-binding 0x88ed)
(def pixel-pack-buffer-binding-ext 0x88ed)
(def pixel-pack-buffer-ext 0x88eb)
(def pixel-transform-2d-ext 0x8330)
(def pixel-transform-2d-matrix-ext 0x8338)
(def pixel-transform-2d-stack-depth-ext 0x8336)
(def pixel-unpack-buffer 0x88ec)
(def pixel-unpack-buffer-binding 0x88ef)
(def pixel-unpack-buffer-binding-ext 0x88ef)
(def pixel-unpack-buffer-ext 0x88ec)
(def pn-triangles-ati 0x87f0)
(def pn-triangles-normal-mode-ati 0x87f3)
(def pn-triangles-normal-mode-linear-ati 0x87f7)
(def pn-triangles-normal-mode-quadratic-ati 0x87f8)
(def pn-triangles-point-mode-ati 0x87f2)
(def pn-triangles-point-mode-cubic-ati 0x87f6)
(def pn-triangles-point-mode-linear-ati 0x87f5)
(def pn-triangles-tesselation-level-ati 0x87f4)
(def point 0x1b00)
(def point-bit 0x2)
(def point-distance-attenuation 0x8129)
(def point-fade-threshold-size 0x8128)
(def point-size 0x0)
(def point-size-granularity 0xb13)
(def point-size-max 0x8127)
(def point-size-min 0x8126)
(def point-size-range 0xb12)
(def point-smooth 0xb10)
(def point-smooth-hint 0xc51)
(def point-sprite 0x8861)
(def point-sprite-coord-origin 0x8ca0)
(def point-token 0x701)
(def points 0x0)
(def polygon 0x9)
(def polygon-bit 0x8)
(def polygon-mode 0xb40)
(def polygon-offset-clamp-ext 0x8e1b)
(def polygon-offset-command-nv 0xe)
(def polygon-offset-factor 0x8038)
(def polygon-offset-fill 0x8037)
(def polygon-offset-line 0x2a02)
(def polygon-offset-point 0x2a01)
(def polygon-offset-units 0x2a00)
(def polygon-smooth 0xb41)
(def polygon-smooth-hint 0xc53)
(def polygon-stipple 0xb42)
(def polygon-stipple-bit 0x10)
(def polygon-token 0x703)
(def post-color-matrix-alpha-bias 0x80bb)
(def post-color-matrix-alpha-scale 0x80b7)
(def post-color-matrix-blue-bias 0x80ba)
(def post-color-matrix-blue-scale 0x80b6)
(def post-color-matrix-color-table 0x80d2)
(def post-color-matrix-green-bias 0x80b9)
(def post-color-matrix-green-scale 0x80b5)
(def post-color-matrix-red-bias 0x80b8)
(def post-color-matrix-red-scale 0x80b4)
(def post-convolution-alpha-bias 0x8023)
(def post-convolution-alpha-scale 0x801f)
(def post-convolution-blue-bias 0x8022)
(def post-convolution-blue-scale 0x801e)
(def post-convolution-color-table 0x80d1)
(def post-convolution-green-bias 0x8021)
(def post-convolution-green-scale 0x801d)
(def post-convolution-red-bias 0x8020)
(def post-convolution-red-scale 0x801c)
(def prefer-doublebuffer-hint-pgi 0x1a1f8)
(def previous 0x8578)
(def previous-texture-input-nv 0x86e4)
(def primary-color 0x8577)
(def primitive-bounding-box 0x92be)
(def primitive-restart 0x8f9d)
(def primitive-restart-fixed-index 0x8d69)
(def primitive-restart-for-patches-supported 0x8221)
(def primitive-restart-index 0x8f9e)
(def primitive-restart-index-nv 0x8559)
(def primitive-restart-nv 0x8558)
(def primitives-generated 0x8c87)
(def primitives-submitted-arb 0x82ef)
(def program 0x82e2)
(def program-address-registers-arb 0x88b0)
(def program-alu-instructions-arb 0x8805)
(def program-attrib-components-nv 0x8906)
(def program-attribs-arb 0x88ac)
(def program-binary-formats 0x87ff)
(def program-binary-length 0x8741)
(def program-binary-retrievable-hint 0x8257)
(def program-binding-arb 0x8677)
(def program-error-position-arb 0x864b)
(def program-error-string-arb 0x8874)
(def program-format-arb 0x8876)
(def program-format-ascii-arb 0x8875)
(def program-input 0x92e3)
(def program-instructions-arb 0x88a0)
(def program-length-arb 0x8627)
(def program-matrix-ext 0x8e2d)
(def program-matrix-stack-depth-ext 0x8e2f)
(def program-native-address-registers-arb 0x88b2)
(def program-native-alu-instructions-arb 0x8808)
(def program-native-attribs-arb 0x88ae)
(def program-native-instructions-arb 0x88a2)
(def program-native-parameters-arb 0x88aa)
(def program-native-temporaries-arb 0x88a6)
(def program-native-tex-indirections-arb 0x880a)
(def program-native-tex-instructions-arb 0x8809)
(def program-object-arb 0x8b40)
(def program-output 0x92e4)
(def program-parameters-arb 0x88a8)
(def program-pipeline 0x82e4)
(def program-pipeline-binding 0x825a)
(def program-point-size 0x8642)
(def program-point-size-arb 0x8642)
(def program-point-size-ext 0x8642)
(def program-result-components-nv 0x8907)
(def program-separable 0x8258)
(def program-string-arb 0x8628)
(def program-temporaries-arb 0x88a4)
(def program-tex-indirections-arb 0x8807)
(def program-tex-instructions-arb 0x8806)
(def program-under-native-limits-arb 0x88b6)
(def programmable-sample-location-arb 0x9341)
(def programmable-sample-location-nv 0x9341)
(def programmable-sample-location-table-size-arb 0x9340)
(def programmable-sample-location-table-size-nv 0x9340)
(def projection-stack-depth 0xba4)
(def provoking-vertex 0x8e4f)
(def provoking-vertex-ext 0x8e4f)
(def proxy-color-table 0x80d3)
(def proxy-histogram 0x8025)
(def proxy-post-color-matrix-color-table 0x80d5)
(def proxy-post-convolution-color-table 0x80d4)
(def proxy-texture-1d 0x8063)
(def proxy-texture-1d-array 0x8c19)
(def proxy-texture-2d 0x8064)
(def proxy-texture-2d-array 0x8c1b)
(def proxy-texture-2d-multisample 0x9101)
(def proxy-texture-2d-multisample-array 0x9103)
(def proxy-texture-3d 0x8070)
(def proxy-texture-cube-map 0x851b)
(def proxy-texture-cube-map-array 0x900b)
(def proxy-texture-rectangle 0x84f7)
(def proxy-texture-rectangle-arb 0x84f7)
(def purgeable-apple 0x8a1d)
(def q 0x2003)
(def quad-strip 0x8)
(def quads 0x7)
(def quads-follow-provoking-vertex-convention 0x8e4c)
(def quads-follow-provoking-vertex-convention-ext 0x8e4c)
(def query 0x82e3)
(def query-all-event-bits-amd 0xffffffff)
(def query-buffer 0x9192)
(def query-buffer-amd 0x9192)
(def query-buffer-barrier-bit 0x8000)
(def query-buffer-binding 0x9193)
(def query-buffer-binding-amd 0x9193)
(def query-by-region-no-wait 0x8e16)
(def query-by-region-no-wait-inverted 0x8e1a)
(def query-by-region-wait 0x8e15)
(def query-by-region-wait-inverted 0x8e19)
(def query-counter-bits 0x8864)
(def query-depth-bounds-fail-event-bit-amd 0x8)
(def query-depth-fail-event-bit-amd 0x2)
(def query-depth-pass-event-bit-amd 0x1)
(def query-no-wait 0x8e14)
(def query-no-wait-inverted 0x8e18)
(def query-object-amd 0x9153)
(def query-result 0x8866)
(def query-result-available 0x8867)
(def query-result-no-wait 0x9194)
(def query-result-no-wait-amd 0x9194)
(def query-stencil-fail-event-bit-amd 0x4)
(def query-target 0x82ea)
(def query-wait 0x8e13)
(def query-wait-inverted 0x8e17)
(def r 0x2002)
(def r11f-g11f-b10f 0x8c3a)
(def r16 0x822a)
(def r16-snorm 0x8f98)
(def r16f 0x822d)
(def r16i 0x8233)
(def r16ui 0x8234)
(def r3-g3-b2 0x2a10)
(def r32f 0x822e)
(def r32i 0x8235)
(def r32ui 0x8236)
(def r8 0x8229)
(def r8-snorm 0x8f94)
(def r8i 0x8231)
(def r8ui 0x8232)
(def raster-fixed-sample-locations-ext 0x932a)
(def raster-multisample-ext 0x9327)
(def raster-samples-ext 0x9328)
(def rasterizer-discard 0x8c89)
(def read-buffer 0xc02)
(def read-framebuffer 0x8ca8)
(def read-framebuffer-binding 0x8caa)
(def read-only 0x88b8)
(def read-pixel-data-range-length-nv 0x887b)
(def read-pixel-data-range-nv 0x8879)
(def read-pixel-data-range-pointer-nv 0x887d)
(def read-pixels 0x828c)
(def read-pixels-format 0x828d)
(def read-pixels-type 0x828e)
(def read-write 0x88ba)
(def reclaim-memory-hint-pgi 0x1a1fe)
(def red 0x1903)
(def red-bias 0xd15)
(def red-bits 0xd52)
(def red-integer 0x8d94)
(def red-max-clamp-ingr 0x8564)
(def red-min-clamp-ingr 0x8560)
(def red-scale 0xd14)
(def red-snorm 0x8f90)
(def reduce 0x8016)
(def referenced-by-compute-shader 0x930b)
(def referenced-by-fragment-shader 0x930a)
(def referenced-by-geometry-shader 0x9309)
(def referenced-by-tess-control-shader 0x9307)
(def referenced-by-tess-evaluation-shader 0x9308)
(def referenced-by-vertex-shader 0x9306)
(def reflection-map 0x8512)
(def released-apple 0x8a19)
(def render 0x1c00)
(def render-mode 0xc40)
(def renderbuffer 0x8d41)
(def renderbuffer-alpha-size 0x8d53)
(def renderbuffer-binding 0x8ca7)
(def renderbuffer-blue-size 0x8d52)
(def renderbuffer-color-samples-nv 0x8e10)
(def renderbuffer-coverage-samples-nv 0x8cab)
(def renderbuffer-depth-size 0x8d54)
(def renderbuffer-free-memory-ati 0x87fd)
(def renderbuffer-green-size 0x8d51)
(def renderbuffer-height 0x8d43)
(def renderbuffer-internal-format 0x8d44)
(def renderbuffer-red-size 0x8d50)
(def renderbuffer-samples 0x8cab)
(def renderbuffer-stencil-size 0x8d55)
(def renderbuffer-width 0x8d42)
(def renderer 0x1f01)
(def repeat 0x2901)
(def replace 0x1e01)
(def replace-value-amd 0x874b)
(def replicate-border 0x8153)
(def resample-average-oml 0x8988)
(def resample-decimate-oml 0x8989)
(def resample-replicate-oml 0x8986)
(def resample-zero-fill-oml 0x8987)
(def rescale-normal 0x803a)
(def reset-notification-strategy 0x8256)
(def retained-apple 0x8a1b)
(def return 0x102)
(def rg 0x8227)
(def rg-integer 0x8228)
(def rg-snorm 0x8f91)
(def rg16 0x822c)
(def rg16-snorm 0x8f99)
(def rg16f 0x822f)
(def rg16i 0x8239)
(def rg16ui 0x823a)
(def rg32f 0x8230)
(def rg32i 0x823b)
(def rg32ui 0x823c)
(def rg8 0x822b)
(def rg8-snorm 0x8f95)
(def rg8i 0x8237)
(def rg8ui 0x8238)
(def rgb 0x1907)
(def rgb-float16-apple 0x881b)
(def rgb-float16-ati 0x881b)
(def rgb-float32-apple 0x8815)
(def rgb-float32-ati 0x8815)
(def rgb-integer 0x8d98)
(def rgb-s3tc 0x83a0)
(def rgb-scale 0x8573)
(def rgb-snorm 0x8f92)
(def rgb10 0x8052)
(def rgb10-a2 0x8059)
(def rgb10-a2ui 0x906f)
(def rgb12 0x8053)
(def rgb16 0x8054)
(def rgb16-snorm 0x8f9a)
(def rgb16f 0x881b)
(def rgb16i 0x8d89)
(def rgb16ui 0x8d77)
(def rgb32f 0x8815)
(def rgb32i 0x8d83)
(def rgb32ui 0x8d71)
(def rgb4 0x804f)
(def rgb4-s3tc 0x83a1)
(def rgb5 0x8050)
(def rgb5-a1 0x8057)
(def rgb565 0x8d62)
(def rgb8 0x8051)
(def rgb8-snorm 0x8f96)
(def rgb8i 0x8d8f)
(def rgb8ui 0x8d7d)
(def rgb9-e5 0x8c3d)
(def rgba 0x1908)
(def rgba-dxt5-s3tc 0x83a4)
(def rgba-float-mode 0x8820)
(def rgba-float-mode-ati 0x8820)
(def rgba-float16-apple 0x881a)
(def rgba-float16-ati 0x881a)
(def rgba-float32-apple 0x8814)
(def rgba-float32-ati 0x8814)
(def rgba-integer 0x8d99)
(def rgba-integer-mode 0x8d9e)
(def rgba-mode 0xc31)
(def rgba-s3tc 0x83a2)
(def rgba-signed-components 0x8c3c)
(def rgba-snorm 0x8f93)
(def rgba-unsigned-dot-product-mapping-nv 0x86d9)
(def rgba12 0x805a)
(def rgba16 0x805b)
(def rgba16-snorm 0x8f9b)
(def rgba16f 0x881a)
(def rgba16i 0x8d88)
(def rgba16ui 0x8d76)
(def rgba2 0x8055)
(def rgba32f 0x8814)
(def rgba32i 0x8d82)
(def rgba32ui 0x8d70)
(def rgba4 0x8056)
(def rgba4-dxt5-s3tc 0x83a5)
(def rgba4-s3tc 0x83a3)
(def rgba8 0x8058)
(def rgba8-snorm 0x8f97)
(def rgba8i 0x8d8e)
(def rgba8ui 0x8d7c)
(def right 0x407)
(def s 0x2000)
(def sample-alpha-to-coverage 0x809e)
(def sample-alpha-to-one 0x809)
(def sample-buffers 0x80a8)
(def sample-coverage 0x80a0)
(def sample-coverage-invert 0x80ab)
(def sample-coverage-value 0x80aa)
(def sample-location-arb 0x8e50)
(def sample-location-nv 0x8e50)
(def sample-location-pixel-grid-height-arb 0x933f)
(def sample-location-pixel-grid-height-nv 0x933f)
(def sample-location-pixel-grid-width-arb 0x933e)
(def sample-location-pixel-grid-width-nv 0x933e)
(def sample-location-subpixel-bits-arb 0x933d)
(def sample-location-subpixel-bits-nv 0x933d)
(def sample-mask 0x8e51)
(def sample-mask-nv 0x8e51)
(def sample-mask-value 0x8e52)
(def sample-mask-value-nv 0x8e52)
(def sample-position 0x8e50)
(def sample-position-nv 0x8e50)
(def sample-shading 0x8c36)
(def sampler 0x82e6)
(def sampler-1d 0x8b5d)
(def sampler-1d-arb 0x8b5d)
(def sampler-1d-array 0x8dc0)
(def sampler-1d-array-shadow 0x8dc3)
(def sampler-1d-shadow 0x8b61)
(def sampler-1d-shadow-arb 0x8b61)
(def sampler-2d 0x8b5e)
(def sampler-2d-arb 0x8b5e)
(def sampler-2d-array 0x8dc1)
(def sampler-2d-array-shadow 0x8dc4)
(def sampler-2d-multisample 0x9108)
(def sampler-2d-multisample-array 0x910b)
(def sampler-2d-rect 0x8b63)
(def sampler-2d-rect-arb 0x8b63)
(def sampler-2d-rect-shadow 0x8b64)
(def sampler-2d-rect-shadow-arb 0x8b64)
(def sampler-2d-shadow 0x8b62)
(def sampler-2d-shadow-arb 0x8b62)
(def sampler-3d 0x8b5f)
(def sampler-3d-arb 0x8b5f)
(def sampler-binding 0x8919)
(def sampler-buffer 0x8dc2)
(def sampler-buffer-amd 0x9001)
(def sampler-cube 0x8b60)
(def sampler-cube-arb 0x8b60)
(def sampler-cube-map-array 0x900c)
(def sampler-cube-map-array-shadow 0x900d)
(def sampler-cube-shadow 0x8dc5)
(def sampler-object-amd 0x9155)
(def sampler-renderbuffer-nv 0x8e56)
(def samples 0x80a9)
(def samples-passed 0x8914)
(def scalar-ext 0x87be)
(def scaled-resolve-fastest-ext 0x90ba)
(def scaled-resolve-nicest-ext 0x90bb)
(def scissor-bit 0x80000)
(def scissor-box 0xc10)
(def scissor-command-nv 0x11)
(def scissor-test 0xc11)
(def screen 0x9295)
(def screen-coordinates-rend 0x8490)
(def secondary-color-array 0x845e)
(def secondary-color-array-address-nv 0x8f27)
(def secondary-color-array-buffer-binding 0x889c)
(def secondary-color-array-length-nv 0x8f31)
(def secondary-color-array-pointer 0x845d)
(def secondary-color-array-size 0x845a)
(def secondary-color-array-stride 0x845c)
(def secondary-color-array-type 0x845b)
(def select 0x1c02)
(def selection-buffer-pointer 0xdf3)
(def selection-buffer-size 0xdf4)
(def separable-2d 0x8012)
(def separate-attribs 0x8c8d)
(def separate-specular-color 0x81fa)
(def set 0x150)
(def set-amd 0x874a)
(def shade-model 0xb54)
(def shader 0x82e1)
(def shader-binary-dmp 0x9250)
(def shader-binary-formats 0x8df8)
(def shader-compiler 0x8dfa)
(def shader-consistent-nv 0x86dd)
(def shader-global-access-barrier-bit-nv 0x10)
(def shader-image-access-barrier-bit 0x20)
(def shader-image-atomic 0x82a6)
(def shader-image-load 0x82a4)
(def shader-image-store 0x82a5)
(def shader-include-arb 0x8dae)
(def shader-object-arb 0x8b48)
(def shader-operation-nv 0x86df)
(def shader-source-length 0x8b88)
(def shader-storage-barrier-bit 0x2000)
(def shader-storage-block 0x92e6)
(def shader-storage-buffer 0x90d2)
(def shader-storage-buffer-binding 0x90d3)
(def shader-storage-buffer-offset-alignment 0x90df)
(def shader-storage-buffer-size 0x90d5)
(def shader-storage-buffer-start 0x90d4)
(def shader-type 0x8b4f)
(def shading-language-version 0x8b8c)
(def shading-language-version-arb 0x8b8c)
(def shadow-attenuation-ext 0x834e)
(def shared-edge-nv 0xc0)
(def shared-texture-palette-ext 0x81fb)
(def short 0x1402)
(def signaled 0x9119)
(def signed-alpha-nv 0x8705)
(def signed-alpha8-nv 0x8706)
(def signed-hilo-nv 0x86f9)
(def signed-hilo16-nv 0x86fa)
(def signed-hilo8-nv 0x885f)
(def signed-intensity-nv 0x8707)
(def signed-intensity8-nv 0x8708)
(def signed-luminance-alpha-nv 0x8703)
(def signed-luminance-nv 0x8701)
(def signed-luminance8-alpha8-nv 0x8704)
(def signed-luminance8-nv 0x8702)
(def signed-normalized 0x8f9c)
(def signed-rgb-nv 0x86fe)
(def signed-rgb-unsigned-alpha-nv 0x870c)
(def signed-rgb8-nv 0x86ff)
(def signed-rgb8-unsigned-alpha8-nv 0x870d)
(def signed-rgba-nv 0x86fb)
(def signed-rgba8-nv 0x86fc)
(def simultaneous-texture-and-depth-test 0x82ac)
(def simultaneous-texture-and-depth-write 0x82ae)
(def simultaneous-texture-and-stencil-test 0x82ad)
(def simultaneous-texture-and-stencil-write 0x82af)
(def single-color 0x81f9)
(def skip-decode-ext 0x8a4a)
(def sluminance 0x8c46)
(def sluminance-alpha 0x8c44)
(def sluminance8 0x8c47)
(def sluminance8-alpha8 0x8c45)
(def sm-count-nv 0x933b)
(def smooth-line-width-granularity 0xb23)
(def smooth-line-width-range 0x0)
(def smooth-point-size-granularity 0xb13)
(def smooth-point-size-range 0x0)
(def softlight 0x929c)
(def source0-alpha 0x8588)
(def source0-rgb 0x8580)
(def source1-alpha 0x8589)
(def source1-rgb 0x8581)
(def source2-alpha 0x858a)
(def source2-rgb 0x8582)
(def sparse-buffer-page-size-arb 0x82f8)
(def sparse-storage-bit-arb 0x400)
(def sparse-texture-full-array-cube-mipmaps-arb 0x91a9)
(def sphere-map 0x2402)
(def src-alpha 0x302)
(def src-alpha-saturate 0x308)
(def src-color 0x300)
(def src0-alpha 0x8588)
(def src0-rgb 0x8580)
(def src1-alpha 0x8589)
(def src1-color 0x88f9)
(def src1-rgb 0x8581)
(def src2-alpha 0x858a)
(def src2-rgb 0x8582)
(def srgb 0x8c40)
(def srgb-alpha 0x8c42)
(def srgb-decode 0x8299)
(def srgb-read 0x8297)
(def srgb-write 0x8298)
(def srgb8 0x8c41)
(def srgb8-alpha8 0x8c43)
(def stack-overflow 0x503)
(def stack-underflow 0x504)
(def static-copy 0x88e6)
(def static-draw 0x88e4)
(def static-read 0x88e5)
(def stencil 0x1802)
(def stencil-attachment 0x8d20)
(def stencil-back-fail 0x8801)
(def stencil-back-func 0x8800)
(def stencil-back-op-value-amd 0x874d)
(def stencil-back-pass-depth-fail 0x8802)
(def stencil-back-pass-depth-pass 0x8803)
(def stencil-back-ref 0x8ca3)
(def stencil-back-value-mask 0x8ca4)
(def stencil-back-writemask 0x8ca5)
(def stencil-bits 0xd57)
(def stencil-buffer-bit 0x400)
(def stencil-clear-tag-value-ext 0x88f3)
(def stencil-clear-value 0xb91)
(def stencil-components 0x8285)
(def stencil-fail 0xb94)
(def stencil-func 0xb92)
(def stencil-index 0x1901)
(def stencil-index1 0x8d46)
(def stencil-index16 0x8d49)
(def stencil-index4 0x8d47)
(def stencil-index8 0x8d48)
(def stencil-op-value-amd 0x874c)
(def stencil-pass-depth-fail 0xb95)
(def stencil-pass-depth-pass 0xb96)
(def stencil-ref 0xb97)
(def stencil-ref-command-nv 0xc)
(def stencil-renderable 0x8288)
(def stencil-samples-nv 0x932e)
(def stencil-tag-bits-ext 0x88f2)
(def stencil-test 0xb90)
(def stencil-test-two-side-ext 0x8910)
(def stencil-value-mask 0xb93)
(def stencil-writemask 0xb98)
(def stereo 0xc33)
(def storage-cached-apple 0x85be)
(def storage-client-apple 0x85b4)
(def storage-private-apple 0x85bd)
(def storage-shared-apple 0x85bf)
(def stream-copy 0x88e2)
(def stream-draw 0x88e0)
(def stream-rasterization-amd 0x91a0)
(def stream-read 0x88e1)
(def strict-depthfunc-hint-pgi 0x1a216)
(def strict-lighting-hint-pgi 0x1a217)
(def strict-scissor-hint-pgi 0x1a218)
(def subpixel-bits 0xd50)
(def subpixel-precision-bias-x-bits-nv 0x9347)
(def subpixel-precision-bias-y-bits-nv 0x9348)
(def subsample-distance-amd 0x883f)
(def subtract 0x84e7)
(def success-nv 0x902f)
(def surface-mapped-nv 0x8700)
(def surface-registered-nv 0x86fd)
(def surface-state-nv 0x86eb)
(def sync-cl-event-arb 0x8240)
(def sync-cl-event-complete-arb 0x8241)
(def sync-condition 0x9113)
(def sync-fence 0x9116)
(def sync-flags 0x9115)
(def sync-flush-commands-bit 0x1)
(def sync-gpu-commands-complete 0x9117)
(def sync-status 0x9114)
(def sync-x11-fence-ext 0x90e1)
(def t 0x2001)
(def t2f-c3f-v3f 0x2a2a)
(def t2f-c4f-n3f-v3f 0x2a2c)
(def t2f-c4ub-v3f 0x2a29)
(def t2f-iui-n3f-v2f-ext 0x81b3)
(def t2f-iui-n3f-v3f-ext 0x81b4)
(def t2f-iui-v2f-ext 0x81b1)
(def t2f-iui-v3f-ext 0x81b2)
(def t2f-n3f-v3f 0x2a2b)
(def t2f-v3f 0x2a27)
(def t4f-c4f-n3f-v4f 0x2a2d)
(def t4f-v4f 0x2a28)
(def table-too-large 0x8031)
(def terminate-sequence-command-nv 0x0)
(def tess-control-output-vertices 0x8e75)
(def tess-control-program-nv 0x891e)
(def tess-control-program-parameter-buffer-nv 0x8c74)
(def tess-control-shader 0x8e88)
(def tess-control-shader-bit 0x8)
(def tess-control-shader-patches-arb 0x82f1)
(def tess-control-subroutine 0x92e9)
(def tess-control-subroutine-uniform 0x92ef)
(def tess-control-texture 0x829c)
(def tess-evaluation-program-nv 0x891f)
(def tess-evaluation-program-parameter-buffer-nv 0x8c75)
(def tess-evaluation-shader 0x8e87)
(def tess-evaluation-shader-bit 0x10)
(def tess-evaluation-shader-invocations-arb 0x82f2)
(def tess-evaluation-subroutine 0x92ea)
(def tess-evaluation-subroutine-uniform 0x92f0)
(def tess-evaluation-texture 0x829d)
(def tess-gen-mode 0x8e76)
(def tess-gen-point-mode 0x8e79)
(def tess-gen-spacing 0x8e77)
(def tess-gen-vertex-order 0x8e78)
(def tessellation-factor-amd 0x9005)
(def tessellation-mode-amd 0x9004)
(def texcoord1-bit-pgi 0x10000000)
(def texcoord2-bit-pgi 0x20000000)
(def texcoord3-bit-pgi 0x40000000)
(def texcoord4-bit-pgi 0x80000000)
(def texture 0x1702)
(def texture-1d 0xde0)
(def texture-1d-array 0x8c18)
(def texture-2d 0xde1)
(def texture-2d-array 0x8c1a)
(def texture-2d-multisample 0x9100)
(def texture-2d-multisample-array 0x9102)
(def texture-3d 0x806f)
(def texture-alpha-size 0x805f)
(def texture-alpha-type 0x8c13)
(def texture-application-mode-ext 0x834f)
(def texture-base-level 0x813c)
(def texture-binding-1d 0x8068)
(def texture-binding-1d-array 0x8c1c)
(def texture-binding-2d 0x8069)
(def texture-binding-2d-array 0x8c1d)
(def texture-binding-2d-multisample 0x9104)
(def texture-binding-2d-multisample-array 0x9105)
(def texture-binding-3d 0x806a)
(def texture-binding-buffer 0x8c2c)
(def texture-binding-cube-map 0x8514)
(def texture-binding-cube-map-array 0x900a)
(def texture-binding-rectangle 0x84f6)
(def texture-binding-rectangle-arb 0x84f6)
(def texture-binding-renderbuffer-nv 0x8e53)
(def texture-bit 0x40000)
(def texture-blue-size 0x805e)
(def texture-blue-type 0x8c12)
(def texture-border 0x1005)
(def texture-border-color 0x1004)
(def texture-border-values-nv 0x871a)
(def texture-buffer 0x8c2a)
(def texture-buffer-binding 0x8c2a)
(def texture-buffer-data-store-binding 0x8c2d)
(def texture-buffer-format 0x8c2e)
(def texture-buffer-offset 0x919d)
(def texture-buffer-offset-alignment 0x919f)
(def texture-buffer-size 0x919e)
(def texture-color-samples-nv 0x9046)
(def texture-compare-fail-value-arb 0x80bf)
(def texture-compare-func 0x884d)
(def texture-compare-mode 0x884c)
(def texture-components 0x1003)
(def texture-compressed 0x86a1)
(def texture-compressed-block-height 0x82b2)
(def texture-compressed-block-size 0x82b3)
(def texture-compressed-block-width 0x82b1)
(def texture-compressed-image-size 0x86a0)
(def texture-compression-hint 0x84ef)
(def texture-constant-data-sunx 0x81d6)
(def texture-coord-array-address-nv 0x8f25)
(def texture-coord-array-buffer-binding 0x889a)
(def texture-coord-array-length-nv 0x8f2f)
(def texture-coord-array-pointer 0x8092)
(def texture-coord-array-size 0x8088)
(def texture-coord-array-stride 0x808a)
(def texture-coord-array-type 0x8089)
(def texture-coverage-samples-nv 0x9045)
(def texture-cube-map 0x8513)
(def texture-cube-map-array 0x9009)
(def texture-cube-map-negative-x 0x8516)
(def texture-cube-map-negative-y 0x8518)
(def texture-cube-map-negative-z 0x851a)
(def texture-cube-map-positive-x 0x8515)
(def texture-cube-map-positive-y 0x8517)
(def texture-cube-map-positive-z 0x8519)
(def texture-cube-map-seamless 0x884f)
(def texture-depth 0x8071)
(def texture-depth-size 0x884a)
(def texture-depth-type 0x8c16)
(def texture-ds-size-nv 0x871d)
(def texture-dt-size-nv 0x871e)
(def texture-env 0x2300)
(def texture-env-color 0x2201)
(def texture-env-mode 0x2200)
(def texture-fetch-barrier-bit 0x8)
(def texture-filter-control 0x8500)
(def texture-fixed-sample-locations 0x9107)
(def texture-float-components-nv 0x888c)
(def texture-free-memory-ati 0x87fc)
(def texture-gather 0x82a2)
(def texture-gather-shadow 0x82a3)
(def texture-gen-mode 0x2500)
(def texture-gen-q 0xc63)
(def texture-gen-r 0xc62)
(def texture-gen-s 0xc60)
(def texture-gen-t 0xc61)
(def texture-green-size 0x805d)
(def texture-green-type 0x8c11)
(def texture-height 0x1001)
(def texture-hi-size-nv 0x871b)
(def texture-image-format 0x828f)
(def texture-image-type 0x8290)
(def texture-immutable-format 0x912f)
(def texture-immutable-levels 0x82df)
(def texture-intensity-size 0x8061)
(def texture-intensity-type 0x8c15)
(def texture-internal-format 0x1003)
(def texture-light-ext 0x8350)
(def texture-lo-size-nv 0x871c)
(def texture-lod-bias 0x8501)
(def texture-luminance-size 0x8060)
(def texture-luminance-type 0x8c14)
(def texture-mag-filter 0x2800)
(def texture-mag-size-nv 0x871f)
(def texture-material-face-ext 0x8351)
(def texture-material-parameter-ext 0x8352)
(def texture-max-anisotropy-ext 0x84fe)
(def texture-max-level 0x813d)
(def texture-max-lod 0x813b)
(def texture-memory-layout-intel 0x83ff)
(def texture-min-filter 0x2801)
(def texture-min-lod 0x813a)
(def texture-normal-ext 0x85af)
(def texture-priority 0x8066)
(def texture-range-length-apple 0x85b7)
(def texture-range-pointer-apple 0x85b8)
(def texture-rectangle 0x84f5)
(def texture-rectangle-arb 0x84f5)
(def texture-red-size 0x805c)
(def texture-red-type 0x8c10)
(def texture-reduction-mode-arb 0x9366)
(def texture-renderbuffer-data-store-binding-nv 0x8e54)
(def texture-renderbuffer-nv 0x8e55)
(def texture-resident 0x8067)
(def texture-samples 0x9106)
(def texture-shader-nv 0x86de)
(def texture-shadow 0x82a1)
(def texture-shared-size 0x8c3f)
(def texture-sparse-arb 0x91a6)
(def texture-srgb-decode-ext 0x8a48)
(def texture-stack-depth 0xba5)
(def texture-stencil-size 0x88f1)
(def texture-storage-hint-apple 0x85bc)
(def texture-storage-sparse-bit-amd 0x1)
(def texture-swizzle-a 0x8e45)
(def texture-swizzle-a-ext 0x8e45)
(def texture-swizzle-b 0x8e44)
(def texture-swizzle-b-ext 0x8e44)
(def texture-swizzle-g 0x8e43)
(def texture-swizzle-g-ext 0x8e43)
(def texture-swizzle-r 0x8e42)
(def texture-swizzle-r-ext 0x8e42)
(def texture-swizzle-rgba 0x8e46)
(def texture-swizzle-rgba-ext 0x8e46)
(def texture-target 0x1006)
(def texture-unsigned-remap-mode-nv 0x888f)
(def texture-update-barrier-bit 0x100)
(def texture-view 0x82b5)
(def texture-view-min-layer 0x82dd)
(def texture-view-min-level 0x82db)
(def texture-view-num-layers 0x82de)
(def texture-view-num-levels 0x82dc)
(def texture-width 0x1000)
(def texture-wrap-r 0x8072)
(def texture-wrap-s 0x2802)
(def texture-wrap-t 0x2803)
(def texture0 0x84c0)
(def texture1 0x84c1)
(def texture10 0x84ca)
(def texture11 0x84cb)
(def texture12 0x84cc)
(def texture13 0x84cd)
(def texture14 0x84ce)
(def texture15 0x84cf)
(def texture16 0x84d0)
(def texture17 0x84d1)
(def texture18 0x84d2)
(def texture19 0x84d3)
(def texture2 0x84c2)
(def texture20 0x84d4)
(def texture21 0x84d5)
(def texture22 0x84d6)
(def texture23 0x84d7)
(def texture24 0x84d8)
(def texture25 0x84d9)
(def texture26 0x84da)
(def texture27 0x84db)
(def texture28 0x84dc)
(def texture29 0x84dd)
(def texture3 0x84c3)
(def texture30 0x84de)
(def texture31 0x84df)
(def texture4 0x84c4)
(def texture5 0x84c5)
(def texture6 0x84c6)
(def texture7 0x84c7)
(def texture8 0x84c8)
(def texture9 0x84c9)
(def time-elapsed 0x88bf)
(def time-elapsed-ext 0x88bf)
(def timeout-expired 0x911b)
(def timestamp 0x8e28)
(def top-level-array-size 0x930c)
(def top-level-array-stride 0x930d)
(def transform-bit 0x1000)
(def transform-feedback 0x8e22)
(def transform-feedback-active 0x8e24)
(def transform-feedback-barrier-bit 0x800)
(def transform-feedback-binding 0x8e25)
(def transform-feedback-binding-nv 0x8e25)
(def transform-feedback-buffer 0x8c8e)
(def transform-feedback-buffer-active 0x8e24)
(def transform-feedback-buffer-active-nv 0x8e24)
(def transform-feedback-buffer-binding 0x8c8f)
(def transform-feedback-buffer-index 0x934b)
(def transform-feedback-buffer-mode 0x8c7f)
(def transform-feedback-buffer-paused 0x8e23)
(def transform-feedback-buffer-paused-nv 0x8e23)
(def transform-feedback-buffer-size 0x8c85)
(def transform-feedback-buffer-start 0x8c84)
(def transform-feedback-buffer-stride 0x934c)
(def transform-feedback-nv 0x8e22)
(def transform-feedback-overflow-arb 0x82ec)
(def transform-feedback-paused 0x8e23)
(def transform-feedback-primitives-written 0x8c88)
(def transform-feedback-stream-overflow-arb 0x82ed)
(def transform-feedback-varying 0x92f4)
(def transform-feedback-varying-max-length 0x8c76)
(def transform-feedback-varyings 0x8c83)
(def transform-hint-apple 0x85b1)
(def transpose-color-matrix 0x84e6)
(def transpose-current-matrix-arb 0x88b7)
(def transpose-modelview-matrix 0x84e3)
(def transpose-program-matrix-ext 0x8e2e)
(def transpose-projection-matrix 0x84e4)
(def transpose-texture-matrix 0x84e5)
(def triangle-fan 0x6)
(def triangle-strip 0x5)
(def triangle-strip-adjacency 0xd)
(def triangle-strip-adjacency-arb 0xd)
(def triangle-strip-adjacency-ext 0xd)
(def triangles 0x4)
(def triangles-adjacency 0xc)
(def triangles-adjacency-arb 0xc)
(def triangles-adjacency-ext 0xc)
(def gl-true 0x1)
(def type 0x92fa)
(def undefined-apple 0x8a1c)
(def undefined-vertex 0x8260)
(def uniform 0x92e1)
(def uniform-address-command-nv 0xa)
(def uniform-array-stride 0x8a3c)
(def uniform-atomic-counter-buffer-index 0x92da)
(def uniform-barrier-bit 0x4)
(def uniform-block 0x92e2)
(def uniform-block-active-uniform-indices 0x8a43)
(def uniform-block-active-uniforms 0x8a42)
(def uniform-block-binding 0x8a3f)
(def uniform-block-data-size 0x8a40)
(def uniform-block-index 0x8a3a)
(def uniform-block-name-length 0x8a41)
(def uniform-block-referenced-by-compute-shader 0x90ec)
(def uniform-block-referenced-by-fragment-shader 0x8a46)
(def uniform-block-referenced-by-geometry-shader 0x8a45)
(def uniform-block-referenced-by-tess-control-shader 0x84f0)
(def uniform-block-referenced-by-tess-evaluation-shader 0x84f1)
(def uniform-block-referenced-by-vertex-shader 0x8a44)
(def uniform-buffer 0x8a11)
(def uniform-buffer-address-nv 0x936f)
(def uniform-buffer-binding 0x8a28)
(def uniform-buffer-binding-ext 0x8def)
(def uniform-buffer-ext 0x8dee)
(def uniform-buffer-length-nv 0x9370)
(def uniform-buffer-offset-alignment 0x8a34)
(def uniform-buffer-size 0x8a2a)
(def uniform-buffer-start 0x8a29)
(def uniform-buffer-unified-nv 0x936e)
(def uniform-is-row-major 0x8a3e)
(def uniform-matrix-stride 0x8a3d)
(def uniform-name-length 0x8a39)
(def uniform-offset 0x8a3b)
(def uniform-size 0x8a38)
(def uniform-type 0x8a37)
(def unknown-context-reset 0x8255)
(def unpack-alignment 0xcf5)
(def unpack-client-storage-apple 0x85b2)
(def unpack-cmyk-hint-ext 0x800f)
(def unpack-compressed-block-depth 0x9129)
(def unpack-compressed-block-height 0x9128)
(def unpack-compressed-block-size 0x912a)
(def unpack-compressed-block-width 0x9127)
(def unpack-constant-data-sunx 0x81d5)
(def unpack-image-height 0x806e)
(def unpack-lsb-first 0xcf1)
(def unpack-resample-oml 0x8985)
(def unpack-row-bytes-apple 0x8a16)
(def unpack-row-length 0xcf2)
(def unpack-skip-images 0x806d)
(def unpack-skip-pixels 0xcf4)
(def unpack-skip-rows 0xcf3)
(def unpack-swap-bytes 0xcf0)
(def unsignaled 0x9118)
(def unsigned-byte 0x1401)
(def unsigned-byte-2-3-3-rev 0x8362)
(def unsigned-byte-3-3-2 0x8032)
(def unsigned-int 0x1405)
(def unsigned-int-10-10-10-2 0x8df6)
(def unsigned-int-10f-11f-11f-rev 0x8)
(def unsigned-int-2-10-10-10-rev 0x8368)
(def unsigned-int-24-8 0x84fa)
(def unsigned-int-5-9-9-9-rev 0x8c3e)
(def unsigned-int-8-8-8-8 0x8035)
(def unsigned-int-8-8-8-8-rev 0x8367)
(def unsigned-int-8-8-s8-s8-rev-nv 0x86db)
(def unsigned-int-atomic-counter 0x92db)
(def unsigned-int-image-1d 0x9062)
(def unsigned-int-image-1d-array 0x9068)
(def unsigned-int-image-2d 0x9063)
(def unsigned-int-image-2d-array 0x9069)
(def unsigned-int-image-2d-multisample 0x906b)
(def unsigned-int-image-2d-multisample-array 0x906c)
(def unsigned-int-image-2d-rect 0x9065)
(def unsigned-int-image-3d 0x9064)
(def unsigned-int-image-buffer 0x9067)
(def unsigned-int-image-cube 0x9066)
(def unsigned-int-image-cube-map-array 0x906a)
(def unsigned-int-s8-s8-8-8-nv 0x86da)
(def unsigned-int-sampler-1d 0x8dd1)
(def unsigned-int-sampler-1d-array 0x8dd6)
(def unsigned-int-sampler-2d 0x8dd2)
(def unsigned-int-sampler-2d-array 0x8dd7)
(def unsigned-int-sampler-2d-multisample 0x910a)
(def unsigned-int-sampler-2d-multisample-array 0x910d)
(def unsigned-int-sampler-2d-rect 0x8dd5)
(def unsigned-int-sampler-3d 0x8dd3)
(def unsigned-int-sampler-buffer 0x8dd8)
(def unsigned-int-sampler-buffer-amd 0x9003)
(def unsigned-int-sampler-cube 0x8dd4)
(def unsigned-int-sampler-cube-map-array 0x900f)
(def unsigned-int-sampler-renderbuffer-nv 0x8e58)
(def unsigned-int-vec2 0x8dc6)
(def unsigned-int-vec3 0x8dc7)
(def unsigned-int-vec4 0x8dc8)
(def unsigned-int16-nv 0x8ff0)
(def unsigned-int16-vec2-nv 0x8ff1)
(def unsigned-int16-vec3-nv 0x8ff2)
(def unsigned-int16-vec4-nv 0x8ff3)
(def unsigned-int64-amd 0x8bc2)
(def unsigned-int64-arb 0x140f)
(def unsigned-int64-nv 0x140f)
(def unsigned-int64-vec2-arb 0x8ff5)
(def unsigned-int64-vec2-nv 0x8ff5)
(def unsigned-int64-vec3-arb 0x8ff6)
(def unsigned-int64-vec3-nv 0x8ff6)
(def unsigned-int64-vec4-arb 0x8ff7)
(def unsigned-int64-vec4-nv 0x8ff7)
(def unsigned-int8-nv 0x8fec)
(def unsigned-int8-vec2-nv 0x8fed)
(def unsigned-int8-vec3-nv 0x8fee)
(def unsigned-int8-vec4-nv 0x8fef)
(def unsigned-normalized 0x8c17)
(def unsigned-short 0x1403)
(def unsigned-short-1-5-5-5-rev 0x8366)
(def unsigned-short-1-5-5-5-rev-ext 0x8366)
(def unsigned-short-4-4-4-4 0x8033)
(def unsigned-short-4-4-4-4-rev 0x8365)
(def unsigned-short-4-4-4-4-rev-ext 0x8365)
(def unsigned-short-5-5-5-1 0x8034)
(def unsigned-short-5-6-5 0x8363)
(def unsigned-short-5-6-5-rev 0x8364)
(def unsigned-short-8-8-apple 0x85ba)
(def unsigned-short-8-8-rev-apple 0x85bb)
(def upper-left 0x8ca2)
(def v2f 0x2a20)
(def v3f 0x2a21)
(def validate-status 0x8b83)
(def variant-array-ext 0x87e8)
(def variant-array-pointer-ext 0x87e9)
(def variant-array-stride-ext 0x87e6)
(def variant-array-type-ext 0x87e7)
(def variant-datatype-ext 0x87e5)
(def variant-ext 0x87c1)
(def variant-value-ext 0x87e4)
(def vbo-free-memory-ati 0x87fb)
(def vector-ext 0x87bf)
(def vendor 0x1f00)
(def version 0x1f02)
(def vertex-array-address-nv 0x8f21)
(def vertex-array-binding 0x85b5)
(def vertex-array-buffer-binding 0x8896)
(def vertex-array-length-nv 0x8f2b)
(def vertex-array-object-amd 0x9154)
(def vertex-array-pointer 0x808e)
(def vertex-array-range-apple 0x851d)
(def vertex-array-range-length-apple 0x851e)
(def vertex-array-range-pointer-apple 0x8521)
(def vertex-array-size 0x807a)
(def vertex-array-storage-hint-apple 0x851f)
(def vertex-array-stride 0x807c)
(def vertex-array-type 0x807b)
(def vertex-attrib-array-address-nv 0x8f20)
(def vertex-attrib-array-barrier-bit 0x1)
(def vertex-attrib-array-buffer-binding 0x889f)
(def vertex-attrib-array-divisor 0x88fe)
(def vertex-attrib-array-enabled 0x8622)
(def vertex-attrib-array-enabled-arb 0x8622)
(def vertex-attrib-array-integer 0x88fd)
(def vertex-attrib-array-integer-nv 0x88fd)
(def vertex-attrib-array-length-nv 0x8f2a)
(def vertex-attrib-array-long 0x874e)
(def vertex-attrib-array-normalized 0x886a)
(def vertex-attrib-array-normalized-arb 0x886a)
(def vertex-attrib-array-pointer 0x8645)
(def vertex-attrib-array-pointer-arb 0x8645)
(def vertex-attrib-array-size 0x8623)
(def vertex-attrib-array-size-arb 0x8623)
(def vertex-attrib-array-stride 0x8624)
(def vertex-attrib-array-stride-arb 0x8624)
(def vertex-attrib-array-type 0x8625)
(def vertex-attrib-array-type-arb 0x8625)
(def vertex-attrib-array-unified-nv 0x8f1e)
(def vertex-attrib-binding 0x82d4)
(def vertex-attrib-map1-apple 0x8a00)
(def vertex-attrib-map1-coeff-apple 0x8a03)
(def vertex-attrib-map1-domain-apple 0x8a05)
(def vertex-attrib-map1-order-apple 0x8a04)
(def vertex-attrib-map1-size-apple 0x8a02)
(def vertex-attrib-map2-apple 0x8a01)
(def vertex-attrib-map2-coeff-apple 0x8a07)
(def vertex-attrib-map2-domain-apple 0x8a09)
(def vertex-attrib-map2-order-apple 0x8a08)
(def vertex-attrib-map2-size-apple 0x8a06)
(def vertex-attrib-relative-offset 0x82d5)
(def vertex-binding-buffer 0x8f4f)
(def vertex-binding-divisor 0x82d6)
(def vertex-binding-offset 0x82d7)
(def vertex-binding-stride 0x82d8)
(def vertex-blend-arb 0x86a7)
(def vertex-consistent-hint-pgi 0x1a22b)
(def vertex-data-hint-pgi 0x1a22a)
(def vertex-element-swizzle-amd 0x91a4)
(def vertex-id-swizzle-amd 0x91a5)
(def vertex-program-arb 0x8620)
(def vertex-program-parameter-buffer-nv 0x8da2)
(def vertex-program-point-size 0x8642)
(def vertex-program-point-size-arb 0x8642)
(def vertex-program-two-side 0x8643)
(def vertex-program-two-side-arb 0x8643)
(def vertex-shader 0x8b31)
(def vertex-shader-binding-ext 0x8781)
(def vertex-shader-bit 0x1)
(def vertex-shader-ext 0x8780)
(def vertex-shader-instructions-ext 0x87cf)
(def vertex-shader-invariants-ext 0x87d1)
(def vertex-shader-invocations-arb 0x82f0)
(def vertex-shader-local-constants-ext 0x87d2)
(def vertex-shader-locals-ext 0x87d3)
(def vertex-shader-optimized-ext 0x87d4)
(def vertex-shader-variants-ext 0x87d0)
(def vertex-subroutine 0x92e8)
(def vertex-subroutine-uniform 0x92ee)
(def vertex-texture 0x829b)
(def vertex-weight-array-ext 0x850c)
(def vertex-weight-array-pointer-ext 0x8510)
(def vertex-weight-array-size-ext 0x850d)
(def vertex-weight-array-stride-ext 0x850f)
(def vertex-weight-array-type-ext 0x850e)
(def vertex-weighting-ext 0x8509)
(def vertex23-bit-pgi 0x4)
(def vertex4-bit-pgi 0x8)
(def vertices-submitted-arb 0x82ee)
(def vibrance-bias-nv 0x8719)
(def vibrance-scale-nv 0x8713)
(def video-buffer-binding-nv 0x9021)
(def video-buffer-internal-format-nv 0x902d)
(def video-buffer-nv 0x9020)
(def video-buffer-pitch-nv 0x9028)
(def video-capture-field-lower-height-nv 0x903b)
(def video-capture-field-upper-height-nv 0x903a)
(def video-capture-frame-height-nv 0x9039)
(def video-capture-frame-width-nv 0x9038)
(def video-capture-surface-origin-nv 0x903c)
(def video-capture-to-422-supported-nv 0x9026)
(def video-color-conversion-matrix-nv 0x9029)
(def video-color-conversion-max-nv 0x902a)
(def video-color-conversion-min-nv 0x902b)
(def video-color-conversion-offset-nv 0x902c)
(def view-class-128-bits 0x82c4)
(def view-class-16-bits 0x82ca)
(def view-class-24-bits 0x82c9)
(def view-class-32-bits 0x82c8)
(def view-class-48-bits 0x82c7)
(def view-class-64-bits 0x82c6)
(def view-class-8-bits 0x82cb)
(def view-class-96-bits 0x82c5)
(def view-class-bptc-float 0x82d3)
(def view-class-bptc-unorm 0x82d2)
(def view-class-rgtc1-red 0x82d0)
(def view-class-rgtc2-rg 0x82d1)
(def view-class-s3tc-dxt1-rgb 0x82cc)
(def view-class-s3tc-dxt1-rgba 0x82cd)
(def view-class-s3tc-dxt3-rgba 0x82ce)
(def view-class-s3tc-dxt5-rgba 0x82cf)
(def view-compatibility-class 0x82b6)
(def viewport 0xba2)
(def viewport-bit 0x800)
(def viewport-bounds-range 0x825d)
(def viewport-command-nv 0x10)
(def viewport-index-provoking-vertex 0x825f)
(def viewport-subpixel-bits 0x825c)
(def virtual-page-size-index-arb 0x91a7)
(def virtual-page-size-x-amd 0x9195)
(def virtual-page-size-x-arb 0x9195)
(def virtual-page-size-y-amd 0x9196)
(def virtual-page-size-y-arb 0x9196)
(def virtual-page-size-z-amd 0x9197)
(def virtual-page-size-z-arb 0x9197)
(def volatile-apple 0x8a1a)
(def w-ext 0x87d8)
(def wait-failed 0x911d)
(def warp-size-nv 0x9339)
(def warps-per-sm-nv 0x933a)
(def weight-array-arb 0x86ad)
(def weight-array-buffer-binding 0x889e)
(def weight-array-pointer-arb 0x86ac)
(def weight-array-size-arb 0x86ab)
(def weight-array-stride-arb 0x86aa)
(def weight-array-type-arb 0x86a9)
(def weight-sum-unity-arb 0x86a6)
(def weighted-average-arb 0x9367)
(def wide-line-hint-pgi 0x1a222)
(def write-discard-nv 0x88be)
(def write-only 0x88b9)
(def write-pixel-data-range-length-nv 0x887a)
(def write-pixel-data-range-nv 0x8878)
(def write-pixel-data-range-pointer-nv 0x887c)
(def x-ext 0x87d5)
(def xor 0x1506)
(def y-ext 0x87d6)
(def ycbaycr8a-4224-nv 0x9032)
(def ycbcr-422-apple 0x85b9)
(def ycbycr8-422-nv 0x9031)
(def z-ext 0x87d7)
(def z4y12z4cb12z4a12z4y12z4cr12z4a12-4224-nv 0x9036)
(def z4y12z4cb12z4cr12-444-nv 0x9037)
(def z4y12z4cb12z4y12z4cr12-422-nv 0x9035)
(def z6y10z6cb10z6a10z6y10z6cr10z6a10-4224-nv 0x9034)
(def z6y10z6cb10z6y10z6cr10-422-nv 0x9033)
(def zero 0x0)
(def zero-ext 0x87dd)
(def zero-to-one 0x935f)
(def zoom-x 0xd16)
(def zoom-y 0xd17)
