(ns thi.ng.morphogen.core
  (:require
   [thi.ng.geom.core :as g :refer [vec3]]
   [thi.ng.geom.circle :as c]
   [thi.ng.geom.polygon :as p]
   [thi.ng.geom.aabb :as a]
   [thi.ng.geom.gmesh :as gm]
   [thi.ng.geom.csg :as csg]
   [thi.ng.geom.meshio :as mio]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.math.core :as m :refer [*eps*]]
   [clojure.java.io :as io]
   [clojure.pprint :refer [pprint]]))

(declare operator)

(defn unmap-uvw
  "Takes a seq of 8 points defining a box and vector of normalized
  u,v,w coordinates. Applies trilinear interpolation to compute point
  within the box:
  u is coord along AD/BC edge, v along AE/BF, w along AB/DC."
  [[a b c d e f g h] [u v w]]
  (g/mix (g/mix a d e h u v) (g/mix b c f g u v) w))

(defn child-path
  "Takes a seq of child indices and constructs a lookup path/vector
  for them by interleaving `:out` in the seq:
      (child-path [1 0 2]) => [:out 1 :out 0 :out 2]"
  [path] (vec (interleave (repeat :out) path)))

(defn apply-recursively
  "Recursively injects tree into itself `n` times, starting at given
  child path. At each subsequent level, the original tree given is
  injected at index `id` of the `:out` child node vector. The initial
  path is simply a seq of indices and will be translated into an
  actual path using the `child-path` fn."
  [tree n path id]
  (loop [t' tree, path (child-path path), n (dec n)]
    (if (pos? n)
      (recur (assoc-in t' path tree) (into path [:out id]) (dec n))
      t')))

(defn map-leaves
  "Takes a fn and operator tree, applies f to all leaf nodes. The fn
  must accept 3 args: the leaf's parent node, the child index of the
  leaf in the parent and the tree depth. The leaf will be replaced
  with the fn's return value."
  ([f tree] (map-leaves f tree 0))
  ([f tree depth]
     (->> (:out tree)
          (interleave (range))
          (partition 2)
          (reduce
           (fn [acc [i c]]
             (cond
              (seq (:out c)) (assoc-in acc [:out i] (map-leaves f c (inc depth)))
              (map? c) (assoc-in acc [:out i] (f acc i depth))
              :default acc))
           tree))))

(defprotocol PNode
  (parent [_] "Returns the node's parent or `nil` if node is the root")
  (depth [_]  "Returns the node's tree depth")
  (faces [_]  "Returns a seq of the node's mesh faces (each face a vec of points)")
  (face-topology [_] "Returns number of vertices used for each face in the node"))

(defrecord BoxNode [points parent depth]
PNode
(parent [_] parent)
(depth [_] depth)
(faces [{[a b c d e f g h] :points}]
  [[b c g f]  ;; front
   [a e h d]  ;; back
   [a b f e]  ;; left
   [c d h g]  ;; right
   [f g h e]  ;; top
   [a d c b]] ;; bottom
  )
(face-topology [_] 4)
  g/PSubdivide
  (subdivide
    [_ {:keys [cols rows slices] :or {cols 1 rows 1 slices 1}}]
    (let [ru (d/successive-nth 2 (m/norm-range cols))
    rv (d/successive-nth 2 (m/norm-range rows))
    rw (d/successive-nth 2 (m/norm-range slices))
    map-p (fn [p] (->> p (unmap-uvw points) (map #(m/roundto % *eps*)) vec3))]
(for [[w1 w2] rw, [v1 v2] rv, [u1 u2] ru]
  (mapv map-p [[u1 v1 w1] [u1 v1 w2] [u2 v1 w2] [u2 v1 w1]
         [u1 v2 w1] [u1 v2 w2] [u2 v2 w2] [u2 v2 w1]])))))
(defn seed
  [x]
  (let [points (cond
                (number? x) (g/vertices (a/aabb x))
                (sequential? x) (vec x)
                (satisfies? g/PGraph x) (g/vertices x))]
    (BoxNode. points nil 0)))

(defmulti operator
  (fn [g-node {:keys [op]}]
   (if op [(type g-node) op])))

;; leaf node operator (no-op)
(defmethod operator nil [_ _] nil)

(defmethod operator [BoxNode :sd]
  [node {:keys [args]}]
  (let [depth (inc (depth node))]
    (->> (g/subdivide node args)
   (map #(BoxNode. % node depth)))))

(defn subdivide-inset
  [[a b c d e f g h :as points]
   {i :inset dir :dir :or {i 0.1 dir :y}}]
  (let [ii (- 1.0 i)
        map-points (fn [base uv]
                     (mapcat
                      (fn [[u v]]
                        [(unmap-uvw points (assoc (vec3) uv [u v]))
                         (unmap-uvw points (assoc base uv [u v]))])
                      [[i i] [i ii] [ii ii] [ii i]]))]
    (condp = dir
      :x (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points g/V3X :yz)]
           [[b c d a b1 b2 a2 a1]
            [c1 c2 d2 d1 f g h e]
            [b c b2 b1 f g c2 c1]
            [a1 a2 d a d1 d2 h e]
            [b1 b2 a2 a1 c1 c2 d2 d1]])
      :y (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points g/V3Y :xz)]
           [[b1 b c c1 b2 f g c2]
            [a a1 d1 d e a2 d2 h]
            [a b b1 a1 e f b2 a2]
            [d1 c1 c d d2 c2 g h]
            [a1 b1 c1 d1 a2 b2 c2 d2]])
      :z (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points g/V3Z :xy)]
           [[a b c d a1 a2 d2 d1]
            [b1 b2 c2 c1 e f g h]
            [a b a2 a1 e f b2 b1]
            [d1 d2 c d c1 c2 g h]
            [a1 a2 d2 d1 b1 b2 c2 c1]]))))

(defmethod operator [BoxNode :sd-inset]
  [node {:keys [args]}]
  (let [depth (inc (depth node))]
    (->> (subdivide-inset (:points node) args)
         (map #(BoxNode. % node depth)))))

(defmethod operator [BoxNode :extrude]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir len] :or {dir :n len 1.0}} :args}]
  [(BoxNode.
    (condp = dir
      :e (let [o (g/* (g/normal3* c d h) len)]
           [a b (g/+ c o) (g/+ d o) e f (g/+ g o) (g/+ h o)])
      :w (let [o (g/* (g/normal3* a b f) len)]
           [(g/+ a o) (g/+ b o) c d (g/+ e o) (g/+ f o) g h])
      :n (let [o (g/* (g/normal3* e f g) len)]
           [a b c d (g/+ e o) (g/+ f o) (g/+ g o) (g/+ h o)])
      :s (let [o (g/* (g/normal3* d a c) len)]
           [(g/- a o) (g/- b o) (g/- c o) (g/- d o) e f g h])
      :f (let [o (g/* (g/normal3* b c g) len)]
           [a (g/+ b o) (g/+ c o) d e (g/+ f o) (g/+ g o) h])
      :b (let [o (g/* (g/normal3* d a e) len)]
           [(g/+ a o) b c (g/+ d o) (g/+ e o) f g (g/+ h o)]))
    node (inc (depth node)))])

(defn reflect-on-plane
  [p q n]
  (g/+ q (g/reflect (g/- q p) (g/normalize n))))

(defmethod operator [BoxNode :reflect]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir] :or {dir :n}} :args}]
  [node
   (BoxNode.
    (condp = dir
      :e (let [n (g/normal3* c d g)]
           [d c (reflect-on-plane b c n) (reflect-on-plane a d n)
            h g (reflect-on-plane f g n) (reflect-on-plane e h n)])
      :w (let [n (g/normal3* a b f)]
           [(reflect-on-plane d a n) (reflect-on-plane c b n) b a
            (reflect-on-plane h e n) (reflect-on-plane g f n) f e])
      :s (let [n (g/normal3* a c b)]
           [(reflect-on-plane e a n) (reflect-on-plane f b n)
            (reflect-on-plane g c n) (reflect-on-plane h d n)
            a b c d])
      :n (let [n (g/normal3* e f g)]
           [e f g h
            (reflect-on-plane a e n) (reflect-on-plane b f n)
            (reflect-on-plane c g n) (reflect-on-plane d h n)])
      :f (let [n (g/normal3* b c g)]
           [b (reflect-on-plane a b n) (reflect-on-plane d c n) c
            f (reflect-on-plane e f n) (reflect-on-plane h g n) g])
      :b (let [n (g/normal3* a e h)]
           [(reflect-on-plane b a n) a d (reflect-on-plane c d n)
            (reflect-on-plane f e n) e h (reflect-on-plane g h n)]))
    node (inc (depth node)))])

(defmethod operator [BoxNode :scale-edge]
  [{[a b c d e f g h] :points :as node}
   {{:keys [edge sym scale] :or {scale 0.5}} :args}]
  (let [s (* scale 0.5)
        scale-if (fn [sid p q s]
                   (if (= sid sym)
                     [(g/mix p q s) (g/mix q p s)] [p q]))
        scale (fn [p q s1 i j s2 k l]
                (let [p' (g/mix p q s), q' (g/mix q p s)
                      dpq (g/dist p' q')
                      [i j] (scale-if s1 i j (/ dpq (g/dist i j) 2))
                      [k l] (scale-if s2 k l (/ dpq (g/dist k l) 2))]
                  [p' q' i j k l]))]
    [(BoxNode.
      (condp = edge
        ;; bottom
        :ab (let [[a b c d e f] (scale a b :x c d :y e f)]
              [a b c d e f g h])
        :bc (let [[b c a d f g] (scale b c :z a d :y f g)]
              [a b c d e f g h])
        :cd (let [[c d a b g h] (scale c d :x a b :y g h)]
              [a b c d e f g h])
        :ad (let [[a d b c e h] (scale a d :z b c :y e h)]
              [a b c d e f g h])
        ;; top
        :ef (let [[e f g h a b] (scale e f :x g h :y a b)]
              [a b c d e f g h])
        :fg (let [[f g e h b c] (scale f g :z e h :y b c)]
              [a b c d e f g h])
        :gh (let [[g h e f c d] (scale g h :x e f :y c d)]
              [a b c d e f g h])
        :eh (let [[e h f g a d] (scale e h :z f g :y a d)]
              [a b c d e f g h])
        ;; left
        :ae (let [[a e d h b f] (scale a e :x d h :z b f)]
              [a b c d e f g h])
        :bf (let [[b f c g a e] (scale b f :x c g :z a e)]
              [a b c d e f g h])
        ;; right
        :cg (let [[c g b f d h] (scale c g :x b f :z d h)]
              [a b c d e f g h])
        :dh (let [[d h a e c g] (scale d h :x a e :z c g)]
              [a b c d e f g h]))
      node (inc (depth node)))]))

(defn operator-output
  [n out empty?]
  (let [default (vec (repeat n (if empty? nil {})))]
    (cond
     (map? out) (reduce-kv assoc default out)
     (sequential? out) (vec out)
     :default default)))

(defn subdiv
  [& {:keys [cols rows slices out empty?] :or {cols 1 rows 1 slices 1}}]
  {:op :sd
   :args {:cols cols :rows rows :slices slices}
   :out (operator-output (* cols rows slices) out empty?)})

(defn subdiv-inset
  [& {:keys [dir inset out empty?] :or {dir :y inset 0.25}}]
  {:op :sd-inset
   :args {:dir dir :inset inset}
   :out (operator-output 5 out empty?)})

(defn reflect
  [& {:keys [dir out empty?] :or {dir :n}}]
  {:op :reflect
   :args {:dir dir}
   :out (operator-output 2 out empty?)})

(defn extrude
  [& {:keys [dir len out empty?] :or {dir :n len 1.0}}]
  {:op :extrude
   :args {:dir dir :len len}
   :out (operator-output 1 out empty?)})

(defn scale-edge
  [edge sym & {:keys [scale out] :or {scale 0.5}}]
  {:op :scale-edge
   :args {:edge edge :sym sym :scale scale}
   :out (operator-output 1 out false)})

(defn walk
  ([seed tree] (walk seed tree [] 1e6))
  ([seed tree max-depth] (walk seed tree max-depth []))
  ([node tree max-depth acc]
     ;;(prn :d (depth node) (:points node) tree)
     (if (< (depth node) max-depth)
       (let [children (operator node tree)]
         (if children
           (->> children
                (interleave (:out tree))
                (partition 2)
                (reduce
                 (fn [acc [ctree c]]
                   ;;(prn :c (:points c) :ctree ctree)
                   (if ctree
                     (walk c ctree max-depth acc)
                     acc))
                 acc))
           (conj acc (gm/into-mesh (faces node)))))
       (conj acc (gm/into-mesh (faces node))))))

(defn circle-lattice-seg
  [n h wall]
  (let [theta (/ m/PI n)
        off (vec3 0 0 h)
        poly (g/rotate (g/as-polygon (c/circle) n) (- (- m/HALF_PI) theta))
        [b c] (map vec3 (g/vertices poly))
        [a d] (map vec3 (g/vertices (p/inset-polygon poly (- wall))))
        [f g] (map #(g/+ off %) [b c])
        [e h] (map #(g/+ off %) [a d])]
    [b f g c a e h d]))

(defn sphere-lattice-seg
  [n h inset wall]
  (let [theta (/ m/PI n)
        off (vec3 0 0 h)
        poly (g/rotate (g/as-polygon (c/circle) n) (- (- m/HALF_PI) theta))
        [b c] (map vec3 (g/vertices poly))
        [a d] (map vec3 (g/vertices (p/inset-polygon poly (- wall))))
        [f g] (map #(g/+ off %) (g/vertices (p/inset-polygon poly (- inset))))
        [e h] (map #(g/+ off %) (g/vertices (p/inset-polygon poly (- (- inset) wall))))]
    [b f g c a e h d]))

(defn csg-union-mesh
  [meshes]
  (->> meshes
       (map csg/mesh->csg)
       (reduce csg/union)
       (csg/csg->mesh)))

(defn union-mesh
  [meshes]
  (reduce g/into-mesh meshes))

(defn save-mesh
  ([seed tree] (save-mesh seed tree "p.ply" 1e6))
  ([seed tree path] (save-mesh seed tree path 1e6))
  ([seed tree path max-depth]
     (with-open [o (io/output-stream path)]
       (->> (walk seed tree max-depth)
            (union-mesh)
            (mio/write-ply o)))))
