(ns thi.ng.sema.core.api
  (:require
   [thi.ng.sema.common.util :as util]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.error :as err]
   [clj-time.format :as tf]
   [clj-time.coerce :as tc]
   [clojure.string :as str]))

(defprotocol PModel
  (add-statement [_ s] [_ g s])
  (add-many [_ statements] [_ g statements])
  (remove-statement [_ s] [_ g s])
  (remove-many [_ statements] [_ g statements])
  (update-statement [_ s1 s2] [_ g s1 s2])
  (select [_] [_ s p o] [_ g s p o])
  (subjects [_])
  (predicates [_])
  (objects [_])
  (subject? [_ x])
  (predicate? [_ x])
  (object? [_ x])
  (indexed? [_ x])
  (remove-subject [_ s] [_ g s]))

(defprotocol PPrefixMap
  (add-prefix [_ prefix uri] [_ prefix-map])
  (prefix-map [_]))

(defprotocol PBaseURI
  (base-uri [_])
  (set-base-uri [_ uri]))

(defprotocol PSetOps
  (union [_ others])
  (intersection [_ others])
  (difference [_ others]))

(defprotocol PDataset
  (remove-model [_ id])
  (update-model [_ id f])
  (get-model [_ id]))

(defprotocol PIndexable
  (index-value [_]))

(defprotocol PNode
  (label [_]))

(defprotocol PLiteral
  (plain-literal? [_])
  (language [_])
  (datatype [_]))

(defprotocol PResource
  (ns-uri [_])
  (local-uri [_]))

(defrecord NodeURI [uri]
  PNode
  (label [_] (.toString uri))
  PIndexable
  (index-value [_] (.toString uri)))

(defrecord NodeLiteral [label ^String lang type]
  PNode
  (label [_] label)
  PLiteral
  (plain-literal? [_]
    (and (nil? type) (nil? lang)))
  (language [_] lang)
  (datatype [_] type)
  PIndexable
  (index-value [_]
    (if lang
      (str label "@" lang)
      (if type
        (str label "^^<" type ">")
        label))))

(defrecord NodeBlank [id]
  PNode
  (label [_] id)
  PIndexable
  (index-value [_] id))

(extend-protocol PIndexable
  String
  (index-value [_] _)
  nil
  (index-value [_] _))

(defn uri?
  "Returns true if x is a NodeURI instance"
  [x] (= NodeURI (type x)))

(defn maybe-uri?
  [x] (not (nil? (re-find #"[\w+]+:[/\w\d]" x))))

(defn blank?
  "Returns true if x is a NodeBlank instance"
  [x] (= NodeBlank (type x)))

(defn literal?
  "Returns true if x is a NodeLiteral instance"
  [x] (= NodeLiteral (type x)))

(defn ensure-triple
  "Throws IllegalArgumentException if the given triple is invalid,
  i.e. subject must be either NodeURI or NodeBlank, pred must be
  NodeURI, object must be any PNode implementation."
  ([[s p o]] (ensure-triple s p o))
  ([s p o]
     (when-not (and (or (uri? s) (blank? s))
                  (uri? p)
                  (satisfies? PNode o))
       (throw (IllegalArgumentException. (str "Invalid triple: " s p o))))))

(declare resource literal-value)

(defn resolve-iri
  [^String base ^String iri] (if (neg? (.indexOf iri ":")) (str base iri) iri))

(defn resolve-pname
  [prefixes ^String pname]
  (when pname
    (if (string? pname)
      (let [idx (.indexOf pname ":")]
        (when (>= idx 0)
          (when-let [prefix (get prefixes (subs pname 0 idx))]
            (str prefix (subs pname (inc idx))))))
      pname)))

(defn resolve-pname-as-node
  [prefixes ^String pname]
  (resource (resolve-pname prefixes pname)))

(defn resolve-literal
  [prefixes ^String x]
  (let [[[_ val _ lang _ type :as re]] (re-seq #"'(.*)'(?:(@([\w\-]+))?)(?:(\^\^([^\x00-\x20<>\"\{\}\|\^\`]+$))?)" x)]
    (when val
      (if lang
        (NodeLiteral. val lang nil)
        (if-let [uri (resolve-pname prefixes type)]
          (NodeLiteral. val nil uri)
          (NodeLiteral. val nil type))))))

(defn iri-as-pname
  [prefixes ^String iri]
  (loop [[[pre piri] & more] (seq prefixes)]
    (when pre
      (if (.startsWith iri ^String piri)
        (str pre ":" (subs iri (count piri)))
        (recur more)))))

(defn iri-as-pname-kw
  [prefixes iri]
  (keyword (iri-as-pname prefixes iri)))

(defn pname-iri
  [prefixes]
  #(if-let [pname (iri-as-pname prefixes (label %))]
     pname (label %)))

(defn pname-iri-kw
  [prefixes]
  #(if-let [pname (iri-as-pname-kw prefixes (label %))]
     pname (label %)))

(defn pname-iri-or-value
  [prefixes]
  #(if (satisfies? PNode %)
     (if (uri? %)
       (if-let [pname (iri-as-pname prefixes (label %))]
         pname (label %))
       (if (literal? %)
         (literal-value %)
         (label %)))
     %))

(defn resource
  ([uri] (NodeURI. uri))
  ([ds pname-or-iri]
     (if (= \< (first pname-or-iri))
       (resolve-iri (base-uri ds) (subs pname-or-iri 1 (dec (count pname-or-iri))))
       (resolve-pname-as-node (prefix-map ds) pname-or-iri))))

(defn blank-node
  ([] (NodeBlank. (util/uuid)))
  ([id] (NodeBlank. (str id))))

(defn literal
  ([val] (NodeLiteral. (str val) nil nil))
  ([val lang] (NodeLiteral. (str val) lang nil))
  ([val lang type]
     (NodeLiteral.
      (str val) lang
      (cond
       (nil? type)             nil
       (satisfies? PNode type) (label type)
       (string? type)          type
       :default                (err/illegal-arg! type)))))

(declare XSD)

(defn as-resource
  [ds x blank-allowed?]
  (cond
   (satisfies? PNode x) x
   (map? x) (cond
             (:uri x) (map->NodeURI x)
             (and blank-allowed? (:id x)) (map->NodeBlank x)
             :default nil)
   (string? x) (resource ds x)
   :default nil))

(defn as-node
  "Attempts to cast x into a PNode implementation, including typed
  literals for numbers, dates & booleans. If given a string, first
  calls maybe-uri? and constructs a NodeURI or NodeLiteral. If given a
  map, resulting node type will depend on presence of these keys:
  :uri (NodeURI), :id (NodeBlank), :label (NodeLiteral)."
  [x]
  (cond
   (satisfies? PNode x)            x
   (map? x)    (cond
                (:uri x)           (map->NodeURI x)
                (:id x)            (map->NodeBlank x)
                (:label x)         (map->NodeLiteral x)
                :default           nil)
   (string? x) (if (maybe-uri? x)  (resource x) (literal x))
   (number? x) (condp = (type x)
                 java.lang.Long    (literal x nil (:long XSD))
                 java.lang.Integer (literal x nil (:int XSD))
                 java.lang.Double  (literal x nil (:double XSD))
                 java.lang.Float   (literal x nil (:float XSD))
                 java.lang.Byte    (literal x nil (:byte XSD))
                 java.lang.Short   (literal x nil (:short XSD))
                 nil)
   (= java.util.Date (type x))     (literal
                                    (tf/unparse (:basic-date-time tf/formatters)
                                                (tc/from-date x))
                                    nil (:date-time XSD))
   (= java.lang.Boolean (type x))  (literal x nil (:boolean XSD))
   :default nil))

(def ^:const default-prefixes
  {"rdf" "http://www.w3.org/1999/02/22-rdf-syntax-ns#"
   "rdfs" "http://www.w3.org/2000/01/rdf-schema#"
   "owl" "http://www.w3.org/2002/07/owl#"
   "foaf" "http://xmlns.com/foaf/0.1/"
   "xsd" "http://www.w3.org/2001/XMLSchema#"
   "inf" "http://owl.thi.ng/inference#"
   })

(defn resolve-pname-map
  ([m] (resolve-pname-map default-prefixes m true))
  ([m as-node?] (resolve-pname-map default-prefixes m as-node?))
  ([prefixes m as-node?]
     (let [rfn (if as-node? resolve-pname-as-node resolve-pname)]
       (reduce-kv (fn [m k v] (assoc m k (rfn prefixes v))) {} m))))

(def ^:const RDF
  (resolve-pname-map
   {:type "rdf:type"
    :statement "rdf:Statement"
    :subject "rdf:subject"
    :predicate "rdf:predicate"
    :object "rdf:object"
    :li "rdf:li"
    :alt "rdf:Alt"
    :bag "rdf:Bag"
    :list "rdf:List"
    :seq "rdf:Seq"
    :first "rdf:first"
    :rest "rdf:rest"
    :nil "rdf:nil"}))

(def ^:const RDFS
  (resolve-pname-map
   {:sub-property "rdfs:subPropertyOf"
    :sub-class "rdfs:subClassOf"
    :range "rdfs:range"
    :domain "rdfs:domain"}))

(def ^:const OWL
  (resolve-pname-map
   {:class "owl:Class"
    :thing "owl:Thing"
    :sym-property "owl:SymmetricProperty"
    :trans-property "owl:TransitiveProperty"
    :inverse-of "owl:inverseOf"}))

(def ^:const INF
  (resolve-pname-map
   {:ruleset "inf:RuleSet"
    :rules "inf:rules"
    :name "inf:name"
    :match "inf:match"
    :result "inf:result"
    :subject "inf:subject"
    :predicate "inf:predicate"
    :object "inf:object"}))

(def ^:const XSD
  (resolve-pname-map
   {:boolean "xsd:boolean"
    :byte "xsd:byte"
    :short "xsd:short"
    :integer "xsd:integer"
    :int "xsd:int"
    :long "xsd:long"
    :non-positive-integer "xsd:nonPositiveInteger"
    :non-negative-integer "xsd:nonNegativeInteger"
    :positive-integer "xsd:positiveInteger"
    :negative-integer "xsd:negativeInteger"
    :unsigned-byte "xsd:unsignedByte"
    :unsigned-short "xsd:unsignedShort"
    :unsigned-int "xsd:unsignedInt"
    :unsigned-long "xsd:unsignedLong"
    :decimal "xsd:decimal"
    :double "xsd:double"
    :float "xsd:float"
    :string "xsd:string"
    :date-time "xsd:dateTime"} false))

(def ^:const NUMERIC-XSD-TYPES
  (->> [:integer
        :decimal
        :float
        :double
        :non-positive-integer
        :non-negative-integer
        :positive-integer
        :negative-integer
        :long
        :int
        :short
        :byte
        :unsigned-long
        :unsigned-int
        :unsigned-short
        :unsigned-byte]
       (map XSD)
       (set)))

                                                                                

(defmulti literal-value #(datatype %))

(defmethod literal-value (:integer XSD)
  [x] (try
                                           
               (js/parseInt (label x))
        (catch Exception e)))

(defmethod literal-value (:float XSD)
  [x]
                                                               
         (js/parseFloat (label x)))

(defmethod literal-value (:double XSD)
  [x]
                                                               
         (js/parseFloat (label x)))

(defmethod literal-value (:boolean XSD)
  [x]
                                                                   
         (let [l (label x)]
           (cond
            (or (= "true" l) (= "TRUE" l)) true
            (or (= "false" l) (= "FALSE" l)) false
            :default nil)))

(defmethod literal-value (:string XSD)
  [x] (label x))

(defmethod literal-value nil
  [x] (label x))

(defmethod literal-value (:date-time XSD)
  [x]
                                                                                          )

     
              
     
       
                                     
                                 
                                    
                                   
                                 
                                   
                                   
                                       
                                                  
                 

;; TODO fix
      
(defn xsd-type
  [x]
  (cond
   (goog/isString x) (:string XSD)
   (instance? Long x) (:long XSD)
   (instance? Double x) (:float XSD)
   (instance? Integer x) (:int XSD)
   (instance? Byte x) (:byte XSD)
   (instance? Short x) (:short XSD)
   (instance? Float x) (:float XSD)
   (or (x == true) (x == false)) (:boolean XSD)
   :default nil))

(defn resolve-item
  [prefixes ^String base x]
  (cond
   (symbol? x) x
   (string? x) (condp = (first x)
                 \' (resolve-literal prefixes x)
                 \< (resolve-iri base (subs x 1 (dec (count x))))
                 (resolve-pname-as-node prefixes x))
   :default x))

(defn resolve-simple-pattern
  [prefixes base [s p o :as t]]
  (let [ss (resolve-item prefixes base s)
        pp (if (= "a" p)
             (:type RDF)
             (resolve-item prefixes base p))
        oo (resolve-item prefixes base o)]
    (if (and ss pp oo)
      [[ss pp oo]]
      (err/illegal-arg! "couldn't resolve pattern: " t))))

(defn resolve-prop-path-pattern
  [prefixes base [s p o :as t]]
  (let [path (str/split p #"/")
        vars (concat [s] (repeatedly (dec (count path)) #(symbol (str "?" (gensym "___q")))))]
    (->> (concat (interleave vars path) [o])
         (d/successive-nth 3 2)
         (mapcat #(resolve-simple-pattern prefixes base %)))))

(defn resolve-patterns
  [prefixes base patterns]
  (mapcat
   (fn [[s p o :as t]]
     (if (and (string? p) (pos? (.indexOf ^String p "/")))
       (resolve-prop-path-pattern prefixes base t)
       (resolve-simple-pattern prefixes base t)))
   patterns))

(defn apply-template
  [tpl data]
  (let [vars (set (d/filter-tree util/qvar? tpl))
        [blanks vars] (d/bisect util/blank-var? vars)
        blanks (zipmap blanks (map (fn [_] (blank-node)) blanks))
        vars (zipmap vars (map #(get data %) vars))
        subst (merge vars blanks)]
    (map #(replace subst %) tpl)))

(defn remove-context
  [t] (if (= 4 (count t)) (rest t) t))

(defn rdf-container-triples
  ([c-type coll] (rdf-container-triples c-type (blank-node) coll))
  ([c-type node coll]
     (let [node-base (str (default-prefixes "rdf") \_)]
       (->> coll
            (map-indexed
             (fn [i v]
               [node (resource (str node-base (inc i))) v]))
            (cons [node (:type RDF) (c-type RDF)])))))

(defn rdf-list-triples
  ([coll] (rdf-list-triples (blank-node) coll))
  ([node coll]
     (loop [triples [[node (:type RDF) (:list RDF)]] n node [i & more] coll]
       (let [stm [[n (:first RDF) (or i (:nil RDF))]]]
         (if (seq more)
           (let [nxt (blank-node)]
             (recur
              (concat triples stm [[n (:rest RDF) nxt]])
              nxt more))
           (concat triples stm [[n (:rest RDF) (:nil RDF)]]))))))

(defn add-container
  ([store c-type coll]
     (add-container store c-type (blank-node) coll))
  ([store c-type node coll]
     (add-many store (rdf-container-triples c-type node coll))))

(defn add-bag
  ([store coll] (add-container store :bag coll))
  ([store node coll] (add-container store :bag node coll)))

(defn add-alt
  ([store coll] (add-container store :alt (set coll)))
  ([store node coll] (add-container store :alt node (set coll))))

(defn add-seq
  ([store coll] (add-container store :seq coll))
  ([store node coll] (add-container store :seq node coll)))

(defn add-list
  ([store coll] (add-list store (blank-node) coll))
  ([store node coll]
     (add-many store (rdf-list-triples node coll))))

(defn rdf-list-seq
  [store node]
  (lazy-seq
   (let [triples (select store node nil nil)
         f (some (fn [[_ p o]] (when (= p (:first RDF)) o)) triples)
         r (some (fn [[_ p o]] (when (= p (:rest RDF)) o)) triples)]
     (when f (if r (cons f (rdf-list-seq store r)) [f])))))

(defn add-reified-statement
  ([store triple] (add-reified-statement store (blank-node) triple))
  ([store node [s p o] & extra]
     (let [store (add-many
                  store (map #(cons node %)
                             [[(:type RDF) (:statement RDF)]
                              [(:subject RDF) s]
                              [(:predicate RDF) p]
                              [(:object RDF) o]]))]
       (if (seq extra)
         (add-many store (map #(cons node %) extra))
         store))))

(defn add-reified-group
  ([store triples extra] (add-reified-group store (blank-node) triples extra))
  ([store node triples extra]
     (let [[store coll] (reduce
                         (fn [[ds nodes] t]
                           (let [n (blank-node)]
                             [(add-reified-statement ds n t) (conj nodes n)]))
                         [store []] triples)
           store (add-bag store node coll)]
       (if (seq extra)
         (add-many store (map #(cons node %) extra))
         store))))

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/sema/core/api.cljx
