(ns thi.ng.trio.core
  (:refer-clojure :exclude [object? indexed?])
  (:require
   [thi.ng.common.data.core :as d]
   [thi.ng.common.data.unionfind :as u]
   [thi.ng.common.error :as err]
   [clojure.set :as set]
         [clojure.pprint]
         [clojure.core.protocols :as cp]))

(defprotocol PTripleSeq
 (triple-seq [_]))
(defprotocol PModelConvert
 (as-model [_]))
(defprotocol PModel
  (subjects [_])
  (predicates [_])
  (objects [_])
  (subject? [_ x])
  (predicate? [_ x])
  (object? [_ x])
  (indexed? [_ x])
  (model-size [_]))
(defprotocol PModelSelect
  (select [_] [_ s p o] [_ g s p o])
  (select-with-alts [_ s p o] [_ g s p o]))
(defprotocol PDataset
  (remove-model [_ id])
  (update-model [_ id m])
  (get-model [_ id]))
(defprotocol PTripleAccess
  (add-triple [_ s] [_ g s])
  (add-triples [_ triples] [_ g triples])
  (remove-triple [_ s] [_ g s])
  (remove-triples [_ triples] [_ g triples])
  (update-triple [_ s s'] [_ g s s'])
  (remove-subject [_ s] [_ g s]))
(defprotocol PAliasModelSupport
  (rewrite-alias [_ a b]))
(defprotocol PModelUpdateHooks
  (add-pre-commit-hook [_ [add remove]])
  (add-post-commit-hook [_ [add remove]])
  (remove-pre-commit-hook [_ [add remove]])
  (remove-post-commit-hook [_ [add remove]]))

(defn- remove-from-index
  [idx i1 i2 i3]
  (let [kv (idx i1)
        v (disj (kv i2) i3)
        kv (if (seq v) kv (dissoc kv i2))]
    (if (seq kv)
      (if (seq v)
        (assoc-in idx [i1 i2] v)
        (assoc idx i1 kv))
      (dissoc idx i1))))

(defn- rewrite-alias*
  [store pred id p q]
  (if (pred store p)
    (let [xs (apply select store (assoc [nil nil nil] id p))
          store (remove-triples store xs)]
      (add-triples store (map #(assoc % id q) xs)))
    store))

(defn rewrite-alias-naive
  [store p q]
  (-> store
      (rewrite-alias* subject?   0 p q)
      (rewrite-alias* predicate? 1 p q)
      (rewrite-alias* object?    2 p q)))

(defn trace [prefix x] (prn prefix x) x)

(declare swizzle swizzle-assoc)

(deftype Triple
         [s p o ^:unsynchronized-mutable __hash]
                                 
      clojure.lang.ILookup
      (valAt
       [_ k] (swizzle _ k nil))
      (valAt
       [_ k nf] (swizzle _ k nf))

      java.util.concurrent.Callable
      (call
       [_] (.invoke ^clojure.lang.IFn _))
      java.lang.Runnable
      (run
        [_] (.invoke ^clojure.lang.IFn _))

      clojure.lang.IFn
      (invoke
       [_ k] (swizzle _ k nil))
      (invoke
       [_ k nf] (swizzle _ k nf))
      (applyTo
       [_ args]
       (condp = (count args)
         1 (swizzle _ (first args) nil)
         2 (swizzle _ (first args) (second args))
         (err/arity-error! (count args))))

      clojure.lang.IPersistentVector
      clojure.lang.Associative
      (count
       [_] 3)
      (length
       [_] 3)
      (containsKey
       [_ k] (not (nil? (#{0 1 2 :s :p :o} k))))
      (entryAt
       [_ k] (clojure.lang.MapEntry. k (case (int k) 0 s, 1 p, 2 o, (err/key-error! k))))
      (assoc
          [_ k v] (swizzle-assoc _ k v))
      (assocN
       [_ k v]
       (case (int k)
         0 (Triple. v p o nil)
         1 (Triple. s v o nil)
         2 (Triple. s p v nil)
         (err/key-error! k)))

      java.util.Collection
      (isEmpty
       [_] false)
      (iterator
       [_] (.iterator ^java.util.Collection (list s p o)))
      (toArray
       [_] (object-array _))
      (size
       [_] 3)

      clojure.lang.Sequential
      clojure.lang.ISeq
      clojure.lang.Seqable
      (first
       [_] s)
      (next
       [_] (cons p (cons o nil)))
      (more
       [_] (cons p (cons o nil)))
      (cons
       [_ x] [s p o x])
      (peek
       [_] o)
      (pop
       [_] [s p])
      (rseq
       [_] (Triple. o p s nil))
      (seq
       [_] _)
      (nth
       [_ k] (case (int k) 0 s, 1 p, 2 o, (err/key-error! k)))
      (nth
       [_ k nf] (case (int k) 0 s, 1 p, 2 o, nf))
      (equiv
       [_ x]
       (if (instance? Triple x)
           (and (clojure.lang.Util/equiv s (.-s ^Triple x))
                (clojure.lang.Util/equiv p (.-p ^Triple x))
                (clojure.lang.Util/equiv o (.-o ^Triple x)))
           (and (instance? java.util.Collection x)
                (== 3 (count x))
                (clojure.lang.Util/equiv s (nth x 0))
                (clojure.lang.Util/equiv p (nth x 1))
                (clojure.lang.Util/equiv o (nth x 2)))))
      (equals
       [_ x]
       (if (instance? Triple x)
           (and (clojure.lang.Util/equals s (.-s ^Triple x))
                (clojure.lang.Util/equals p (.-p ^Triple x))
                (clojure.lang.Util/equals o (.-o ^Triple x)))
           (and (instance? java.util.Collection x)
                (== 3 (count x))
                (clojure.lang.Util/equals s (nth x 0))
                (clojure.lang.Util/equals p (nth x 1))
                (clojure.lang.Util/equals o (nth x 2)))))
      (hashCode
       [_]
       (-> 31
           (unchecked-add-int (hash s))
           (unchecked-multiply-int 31)
           (unchecked-add-int (hash p))
           (unchecked-multiply-int 31)
           (unchecked-add-int (hash o))))

      clojure.lang.IHashEq
      (hasheq
       [_]
       (or __hash (set! __hash
                        (mix-collection-hash
                         (-> 31
                             (unchecked-add-int (hash s))
                             (unchecked-multiply-int 31)
                             (unchecked-add-int (hash p))
                             (unchecked-multiply-int 31)
                             (unchecked-add-int (hash o)))
                         3))))

      Comparable
      (compareTo
       [_ x]
       (if (instance? Triple x)
         (let [c (compare s (.-s ^Triple x))]
           (if (== 0 c)
             (let [c (compare p (.-p ^Triple x))]
               (if (== 0 c)
                 (compare o (.-o ^Triple x))
                 c))
             c))
         (let [c (count x)]
           (if (== 3 c) (compare x _) (- 3 c)))))

      cp/InternalReduce
      (internal-reduce
       [_ f start]
       (let [acc (f start s)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc p)]
             (if (reduced? acc)
               @acc
               (let [acc (f acc o)]
                 (if (reduced? acc)
                   @acc
                   acc)))))))

      cp/CollReduce
      (coll-reduce
       [_ f]
       (let [acc (f s p)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc o)]
             (if (reduced? acc)
               @acc
               acc)))))
      (coll-reduce
       [_ f start]
       (let [acc (f start s)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc p)]
             (if (reduced? acc)
               @acc
               (let [acc (f acc o)]
                 (if (reduced? acc)
                   @acc
                   acc)))))))

      Object
      (toString
       [_]
       (.toString
        (doto (StringBuilder. "[")
          (.append (pr-str s))
          (.append " ")
          (.append (pr-str p))
          (.append " ")
          (.append (pr-str o))
          (.append "]"))))
                 
              
                                   

              
               
                                
               
                                  

          
               
                                
               
                                  

               
              
              

                   
                      
                                                 
              
                                      

              
                
               
                     
                               
                               
                               
                              

                  
           
              
              
             
                                  

            
             
                                  

               
            
              

                  
             
                                

               
            
                                                               
            
                                                  

                  
             
                        

             
             
              
            
                  

                  
                
             
                                
                                              
                        
                                                  
                            
                                             
                     
                 
                            
                                                  

            
             
           
                  
                              
                                      
                                      
                                 
                                             
                                 
                                             
                                  
                        

             
              
             
                                
                                                                                 
                                               
                                                                 

              
               
                
                          
                            
                
                                
                                
                    
                        
               
                      
                              
                            
                
                                
                                
                    
                                    
                                    
                        
                              

             
                
                                        
)
      (defmethod clojure.pprint/simple-dispatch Triple
  [^Triple o] ((get-method clojure.pprint/simple-dispatch clojure.lang.IPersistentVector) o))
      (defmethod print-method Triple [^Triple o ^java.io.Writer w] (.write w (.toString o)))
(defn- lookup3
  [^Triple _ k nf]
  (case k
    \s (.-s _)
    \p (.-p _)
    \o (.-o _)
    (or nf (err/key-error! k))))

(defn- swizzle
  [^Triple _ k default]
  (if (number? k)
    (case (int k)
      0 (.-s _)
      1 (.-p _)
      2 (.-o _)
      (or default (err/key-error! k)))
    (case k
      :s (.-s _)
      :p (.-p _)
      :o (.-o _)
      (let [n (name k) c (count n)]
        (case c
          2 [(lookup3 _ (nth n 0) default)
             (lookup3 _ (nth n 1) default)]
          3 (Triple.
             (lookup3 _ (nth n 0) default)
             (lookup3 _ (nth n 1) default)
             (lookup3 _ (nth n 2) default)
             nil)
          (or default (err/key-error! k)))))))

(defn- swizzle-assoc*
  [_ keymap k v]
  (let [n (name k)
        c (count n)]
    (if (and (<= c (count keymap)) (== c (count v) (count (into #{} n))))
      (loop [acc (vec _), i 0, n n]
        (if n
          (recur (assoc acc (keymap (first n)) (v i)) (inc i) (next n))
          (Triple. (acc 0) (acc 1) (acc 2) nil)))
      (err/key-error! k))))

(defn- swizzle-assoc
  [^Triple _ k v]
  (case k
    :s (Triple. v (.-p _) (.-o _) nil)
    :p (Triple. (.-s _) v (.-o _) nil)
    :o (Triple. (.-s _) (.-p _) v nil)
    0 (Triple. v (.-p _) (.-o _) nil)
    1 (Triple. (.-s _) v (.-o _) nil)
    2 (Triple. (.-s _) (.-p _) v nil)
    (swizzle-assoc* _ {\s 0 \p 1 \o 2} k v)))
(defn triple
  ([s p o] (Triple. s p o nil))
  ([t] (if (instance? Triple t) t (let [[s p o] t] (Triple. s p o nil)))))

(defn- select-with-alts-1
  [coll idx]
  (->> (set/intersection coll (set (keys idx)))
       (mapcat #(->> % idx vals (apply concat)))))

(defn- select-with-alts-2
  [outer inner idx]
  (mapcat
   (fn [o]
     (let [out (idx o)]
       (->> (set (keys out))
            (set/intersection inner)
            (mapcat out))))
   outer))

(defn- select-with-alts-3
  [outer inner preds idx lookup]
  (reduce
   (fn [acc o]
     (let [out (idx o)]
       (->> (set (keys out))
            (set/intersection preds)
            (reduce
             (fn [acc p]
               (if-let [t (some #(if (inner (lookup %)) %) (out p))]
                 (conj acc t)
                 acc))
             acc))))
   [] outer))

(defrecord PlainMemoryStore [spo pos osp size]
  PTripleAccess
  (add-triple
    [_ [s p o :as t]]
    (if (-> spo (get s nil) (get p nil) (get t nil)) _
        (let [s (get (find spo s) 0 s)
              p (get (find pos p) 0 p)
              o (get (find osp o) 0 o)
              t (Triple. s p o nil)]
          (PlainMemoryStore.
           (update-in spo [s p] d/set-conj t)
           (update-in pos [p o] d/set-conj t)
           (update-in osp [o s] d/set-conj t)
           (inc size)))))
  (add-triples [_ triples]
    (loop [changed? false, spo spo, pos pos, osp osp, size size, xs triples]
      (if xs
        (let [[s p o :as t] (first xs)]
          (if (-> spo (get s nil) (get p nil) (get t nil))
            (recur changed? spo pos osp size (next xs))
            (let [s (get (find spo s) 0 s)
                  p (get (find pos p) 0 p)
                  o (get (find osp o) 0 o)
                  t (Triple. s p o nil)]
              (recur
               true
               (update-in spo [s p] d/set-conj t)
               (update-in pos [p o] d/set-conj t)
               (update-in osp [o s] d/set-conj t)
               (inc size)
               (next xs)))))
        (if changed?
          (PlainMemoryStore. spo pos osp size)
          _))))
  (remove-triple [_ [s p o :as t]]
    (if (-> spo (get s nil) (get p nil) (get t nil))
      (PlainMemoryStore.
       (remove-from-index spo s p t)
       (remove-from-index pos p o t)
       (remove-from-index osp o s t)
       (dec size))
      _))
  (remove-triples [_ triples]
    (loop [changed? false, spo spo, pos pos, osp osp, size size, xs triples]
      (if xs
        (let [[s p o :as t] (first xs)]
          (if (-> spo (get s nil) (get p nil) (get t nil))
            (recur
             true
             (remove-from-index spo s p t)
             (remove-from-index pos p o t)
             (remove-from-index osp o s t)
             (dec size)
             (next xs))
            (recur changed? spo pos osp size (next xs))))
        (if changed?
          (PlainMemoryStore. spo pos osp size)
          _))))
  (update-triple [_ s1 s2]
    (add-triple (remove-triple _ s1) s2))
  (remove-subject [_ s]
    (remove-triples _ (select _ s nil nil)))

  PModel
  (subject? [_ x]
    (if (spo x) x))
  (predicate? [_ x]
    (if (pos x) x))
  (object? [_ x]
    (if (osp x) x))
  (indexed? [_ x]
    (if (or (spo x) (pos x) (osp x)) x))
  (subjects [_] (keys spo))
  (predicates [_] (keys pos))
  (objects [_] (keys osp))
  (model-size [_] size)

  PModelSelect
  (select [_]
    (select _ nil nil nil))
  (select
    [_ s p o]
    (if s
      (if p
        (if o
          ;; s p o
          (let [t (triple s p o)]
            (if (-> spo (get s nil) (get p nil) (get t nil)) [t]))
          ;; s p nil
          (-> spo (get s nil) (get p nil)))
        ;; s nil o / s nil nil
        (if o
          (-> osp (get o nil) (get s nil))
          (->> (spo s) vals (apply concat))))
      (if p
        (if o
          ;; nil p o
          (-> pos (get p nil) (get o nil))
          ;; nil p nil
          (->> (pos p) vals (apply concat)))
        (if o
          ;; nil nil o
          (->> (osp o) vals (apply concat))
          ;; nil nil nil
          (->> spo vals (mapcat vals) (apply concat))))))

  (select-with-alts
    [_ s p o]
    (let [s (if (set? s) (if-not (empty? s) s) (if s #{s}))
          p (if (set? p) (if-not (empty? p) p) (if p #{p}))
          o (if (set? o) (if-not (empty? o) o) (if o #{o}))]
      (if s
        (if p
          (if o
            (select-with-alts-3 s o p spo peek)
            (select-with-alts-2 s p spo))
          (if o
            (select-with-alts-2 o s osp)
            (select-with-alts-1 s spo)))
        (if p
          (if o
            (select-with-alts-2 p o pos)
            (select-with-alts-1 p pos))
          (if o
            (select-with-alts-1 o osp)
            (->> spo vals (mapcat vals) (apply concat)))))))

  PAliasModelSupport
  (rewrite-alias
    [_ a b] (rewrite-alias-naive _ a b)))
(defprotocol PNode
  (get-children [_])
  (add-child [_ c])
  (remove-child [_ c]))

(defprotocol PGraph
  (add-node [_ n] [_ n parents])
  (get-node [_ t])
  (get-ids [_])
  (get-nodes [_]))

(deftype TripleNode [triple ^:volatile-mutable children]
  PNode
  (get-children [_] children)
  (add-child [_ c] (set! children (conj (or children #{}) c)) _)
  (remove-child [_ c] (set! children (disj children c)) _)

  Object
  (toString [_] (str (pr-str triple) " " (pr-str children))))

(declare select-tg*)

(deftype TripleGraph [nodes ids next-id]
  Object
  (toString
    [_] (str ":nodes " (pr-str nodes)
             " :ids " (pr-str ids)
             " :next " next-id))

  PGraph
  (add-node
    [_ n] (add-node _ n nil))
  (add-node
    [_ n parents]
    (let [g (TripleGraph.
             (assoc nodes next-id n)
             (assoc ids (.-triple  ^TripleNode n) next-id)
             (inc next-id))]
      (when (seq parents)
        (doseq [^TripleNode p parents]
          (add-child p next-id)))
      g))
  (get-node
    [_ t] (nodes (ids t)))

  PTripleAccess
  (add-triple
    [_ [s p o :as t]]
    (if-not (ids t)
      (let [id (.-next-id _)
            ^TripleGraph g (add-node _ (TripleNode. t nil) nil)
            index-branch (fn [g patterns]
                           (loop [^TripleGraph g g, id id, ps patterns]
                             (if ps
                               (let [id' ((.-ids g) (first ps))]
                                 (if id'
                                   (do
                                     (add-child ^TripleNode ((.-nodes g) id') id)
                                     g)
                                   (recur
                                    (add-node g (TripleNode. (first ps) #{id}) nil)
                                    (.-next-id g)
                                    (next ps))))
                               g)))]
        (-> (index-branch g [[s p nil] [s nil nil] [nil nil nil]])
            (index-branch [[nil p o] [nil p nil]])
            (index-branch [[s nil o] [nil nil o]])
            ))
      _))

  PModelSelect
  (select
    [_ s p o]
    (persistent! (select-tg* nodes (ids [s p o]) (transient [])))))

(defn- select-tg*
  [nodes id acc]
  (let [^TripleNode n (nodes id)]
    (if n
      (let [c (get-children n)]
        (if c
          (reduce #(select-tg* nodes %2 %) acc c)
          (conj! acc (.-triple n))))
      acc)))

(defn triple-graph
  []
  (let [root (TripleNode. [nil nil nil] nil)]
    (add-node (TripleGraph. {} {} 0) root)))

(defrecord AliasMemoryStore [store aliases]
  PTripleAccess
  (add-triple
    [_ [s p o]]
    (let [t [(or (u/canonical aliases s) s)
             (or (u/canonical aliases p) p)
             (or (u/canonical aliases o) o)]]
      (AliasMemoryStore. (add-triple store t) aliases)))
  (add-triples
    [_ triples]
    (loop [store store, xs triples]
      (if xs
        (let [[s p o] (first xs)
              t [(or (u/canonical aliases s) s)
                 (or (u/canonical aliases p) p)
                 (or (u/canonical aliases o) o)]]
          (recur (add-triple store t) (next xs)))
        (AliasMemoryStore. store aliases))))

  PModel
  (subject? [_ x]
    (subject? store (or (u/canonical aliases x) x)))
  (predicate? [_ x]
    (predicate? store (or (u/canonical aliases x) x)))
  (object? [_ x]
    (object? store (or (u/canonical aliases x) x)))
  (indexed? [_ x]
    (indexed? store (or (u/canonical aliases x) x)))
  (subjects [_] (subjects store))
  (predicates [_] (predicates store))
  (objects [_] (objects store))
  (model-size [_] (model-size store))

  PModelSelect
  (select
    [_] (select _ nil nil nil))
  (select
    [_ s p o]
    (let [s (or (u/canonical aliases s) s)
          p (or (u/canonical aliases p) p)
          o (or (u/canonical aliases o) o)]
      (select store s p o)))

  u/PUnionFind
  (canonical [_ p] (u/canonical aliases p))
  (canonical? [_ p] (u/canonical? aliases p))
  (component [_ p] (u/component aliases p))
  (disjoint-components [_] (u/disjoint-components aliases))
  (register [_ p] (AliasMemoryStore. store (u/register aliases p)))
  (unregister
    [_ p]
    (if (u/canonical? aliases p)
      (let [q (first (disj (u/component aliases p) p))
            aliases (u/unregister aliases p)
            store (if q (rewrite-alias store p q) store)]
        (AliasMemoryStore. store aliases))
      (AliasMemoryStore. store (u/unregister aliases p))))
  (unified? [_ p q] (u/unified? aliases p q))
  (union
    [_ p q]
    (if (and p q)
      (let [aliases (u/union aliases p q)
            canon (u/canonical aliases p)
            store (if (= p canon)
                    (rewrite-alias store q canon)
                    (rewrite-alias store p canon))]
        (AliasMemoryStore. store aliases))
      (err/illegal-arg! (str "aliases must be both non-nil values: " [p q])))))

(defrecord PlainDataset [models]
  PTripleAccess
  (add-triple [_ s]
    (add-triple _ :default s))
  (add-triple [_ g s]
    (update-in _ [:models g] add-triple s))
  (add-triples [_ triples]
    (add-triples _ :default triples))
  (add-triples [_ g triples]
    (update-in _ [:models g] add-triples triples))
  (remove-triple [_ s]
    (remove-triple _ :default s))
  (remove-triple [_ g s]
    (update-in _ [:models g] remove-triple s))
  (remove-triples [_ triples]
    (remove-triples _ :default triples))
  (remove-triples [_ g triples]
    (update-in _ [:models g] remove-triples triples))
  (remove-subject [_ s]
    (remove-subject _ :default s))
  (remove-subject [_ g s]
    (update-in _ [:models g] remove-subject s))

  PModel
  (subject? [_ x]
    (some #(subject? % x) (vals models)))
  (predicate? [_ x]
    (some #(predicate? % x) (vals models)))
  (object? [_ x]
    (some #(object? % x) (vals models)))
  (indexed? [_ x]
    (some #(indexed? % x) (vals models)))
  (subjects [_]
    (set (mapcat subjects (vals models))))
  (predicates [_]
    (set (mapcat predicates (vals models))))
  (objects [_]
    (set (mapcat objects (vals models))))
  (model-size [_]
    (reduce + (map model-size (vals models))))

  PModelSelect
  (select [_]
    (select _ nil nil nil))
  (select [_ s p o]
    (mapcat #(select % s p o) (vals models)))
  (select [_ g s p o]
    (if-let [g (models g)] (select g s p o)))

  PDataset
  (update-model [_ id m]
    (assoc-in _ [:models id] m))
  (remove-model [_ id]
    (update-in _ [:models] dissoc id))
  (get-model [_ id]
    (models id)))

(defn plain-store
  [& triples]
  (add-triples (PlainMemoryStore. (hash-map) (hash-map) (hash-map) 0) triples))

(defn alias-store
  [store aliases & triples]
  (add-triples (reduce (partial apply u/union) (AliasMemoryStore. store (u/disjoint-set)) aliases) triples))
(defn plain-dataset
  [& {:as models}]
  (PlainDataset. (assoc models :default (plain-store))))

(defn triple-seq-associative
  "Converts a single nested map into a seq of triples.
  Each key must have another map as value. Toplevel keys become
  subjects, value map keys predicates, inner map values objects. Each
  predicate key can define a seq of values to produce multiple
  triples."
  [coll]
  (mapcat
   (fn [[s v]]
     (mapcat
      (fn [[p o]]
        (if (sequential? o)
          (mapv (fn [o] [s p o]) o)
          [[s p o]]))
      v))
   coll))
(defn triple-seq-sequential
 [coll]
 (mapcat
  (fn [triple]
    (if (map? triple)
      (triple-seq triple)
      (->> triple
           (map #(if (sequential? %) % [%]))
           (apply d/cartesian-product))))
  coll))
(extend-protocol PTripleSeq
         clojure.lang.Sequential
                         
  (triple-seq [_] (triple-seq-sequential _))
             
                                                   
                
                                                   
                   
                                                   
         clojure.lang.IPersistentMap
                          
  (triple-seq [_] (triple-seq-associative _))
                           
                                                    )

(extend-protocol PModelConvert
         clojure.lang.Sequential
                         
  (as-model [_] (apply plain-store (triple-seq-sequential _)))
             
                                                                     
                
                                                                     
                   
                                                                     
         clojure.lang.IPersistentMap
                          
  (as-model [_] (apply plain-store (triple-seq-associative _)))
                           
                                                                      )

(defn select-from-seq
  [_ s p o]
  (-> (if s
        (if p
          (if o
            (fn [[s' p' o']] (and (= s s') (= p p') (= o o')))
            (fn [[s' p']] (if (= s s') (= p p'))))
          (if o
            (fn [[s' _ o']] (if (= s s') (= o o')))
            (fn [t] (= s (first t)))))
        (if p
          (if o
            (fn [[_ p' o']] (if (= p p') (= o o')))
            (fn [t] (= p (nth t 1))))
          (if o
            (fn [t] (= o (nth t 2)))
            identity)))
      (filter _)))

(extend-protocol PModelSelect
         clojure.lang.Sequential
                         
  (select [_] (select-from-seq _ nil nil nil))
  (select [_ s p o] (select-from-seq _ s p o))
             
                                                     
                                                     
                
                                                     
                                                     
                   
                                                     
                                                     )

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/trio/core.cljx
