(ns thi.ng.trio.query
  (:require
   [thi.ng.trio.core :as api]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.error :as err]
   [thi.ng.common.data.unionfind :as uf]
   [clojure.set :as set]
   [clojure.core.reducers :as r]))

(defn trace [id _] (dorun _) (prn id _) _)
(def ^:dynamic *auto-qvar-prefix* "?__q")

(defn qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (re-matches #"^\?.*" (name x))))

(defn auto-qvar?
  "Returns true, if x is an auto-generated qvar
  (a symbol prefixed with *auto-qvar-prefix*)"
  [x] (and (symbol? x) (zero? (.indexOf ^String (name x) *auto-qvar-prefix*))))

(defn make-qvar
  "Creates a new auto-named qvar (symbol)."
  [] (gensym *auto-qvar-prefix*))

(defn qvar-name
  [x] (-> x name (subs 1)))
(defn resolve-path-pattern
  "Takes a path triple pattern where the predicate is a seq of preds.
  Returns seq of query patterns with injected temp qvars for inbetween
  patterns. E.g.

      [?s [p1 p2 p3] ?o]
      => ([?s p1 ?__q0] [?__q0 p2 ?__q1] [?__q1 p3 ?o])"
  [[s p o]]
  (let [vars (->> make-qvar
                  (repeatedly (dec (count p)))
                  (cons s))]
    (->> (concat (interleave vars p) [o])
         (d/successive-nth 3 2))))

(defn resolve-patterns
  [patterns]
  (mapcat
   (fn [[_ p :as t]]
     (if (vector? p)
       (resolve-path-pattern t)
       [t]))
   patterns))

(defn produce-patterns-with-bound-vars
  "Takes a triple pattern (possibly with variables) and a map of
  possible value sets (each *must* be a set or single value) for each var.
  Produces lazy seq of resulting triple query patterns using cartesian
  product of all values.

      (produce-patterns-with-bound-vars
        [?a :type ?b]
        {?a #{\"me\" \"you\"} ?b #{\"foo\" \"bar\"})
      => ((\"me\" :type \"foo\") (\"me\" :type \"bar\")
          (\"you\" :type \"foo\") (\"you\" :type \"bar\"))"
  [[s p o] bindings]
  (let [s (or (bindings s) s)
        p (or (bindings p) p)
        o (or (bindings o) o)]
    (if (some set? [s p o])
      (d/cartesian-product
       (if (set? s) s #{s}) (if (set? p) p #{p}) (if (set? o) o #{o}))
      [[s p o]])))

(defn sort-patterns
  "Sorts a seq of triple patterns in dependency order using any
  re-occuring vars. Triples with least qvars will be in head
  position."
  [patterns]
  (let [q (map #(let [v (d/filter-tree qvar? %)] [(count v) v %]) patterns)
        singles (->> q (filter #(= 1 (first %))) (mapcat second) set)]
    (->> q
         (sort-by (fn [[c v]] (- (* c 4) (count (filter singles v)))))
         (map peek))))

(defn triple-verifier
  "Takes a triple pattern (potentially with vars) and 3 booleans to
  indicate which SPO is a var. Returns fn which accepts a result triple
  and returns false if any of the vars clash (e.g. a qvar is used multiple
  times but result has different values in each position or likewise, if
  different vars relate to same values)."
  [[ts tp to] vars varp varo]
  (cond
   (and vars varp varo) (cond
                         (= ts tp to) (fn [[rs rp ro]] (= rs rp ro))
                         (= ts tp) (fn [[rs rp ro]] (and (= rs rp) (not= rs ro)))
                         (= ts to) (fn [[rs rp ro]] (and (= rs ro) (not= rs rp)))
                         (= tp to) (fn [[rs rp ro]] (and (= rp ro) (not= rs rp)))
                         :default (constantly true))
   (and vars varp) (if (= ts tp)
                     (fn [[rs rp]] (= rs rp))
                     (fn [[rs rp]] (not= rs rp)))
   (and vars varo) (if (= ts to)
                     (fn [[rs _ ro]] (= rs ro))
                     (fn [[rs _ ro]] (not= rs ro)))
   (and varp varo) (if (= tp to)
                     (fn [[_ rp ro]] (= rp ro))
                     (fn [[_ rp ro]] (not= rp ro)))
   :default (constantly true)))
(defn unique-bindings?
  "Returns true if all values in the given map are unique, i.e.
  no two keys are mapped to the same value."
  [map] (let [u? (== (count (into #{} (vals map))) (count map))] (prn u? map) u?))

(defn accumulate-result-vars
  "Takes a query result map seq and returns map of all found qvars
  as keys and their value sets."
  ([res] (accumulate-result-vars {} nil res))
  ([acc vars res]
     (reduce
      (fn [acc b]
        (merge-with
         (fn [a b] (if (set? a) (conj a b) (hash-set a b)))
         acc (if vars (select-keys b vars) b)))
      acc res)))

(defn accumulate-var-triples
  "Takes a qvar, a value for that var and a query result map.
  Returns reducible of all triples for which the qvar has given value."
  [?v v res]
  (                      mapcat val
         (                      filter (fn [[k]] (= v (k ?v))) res)))

(defn replace-result-vars
  "Takes a result map, a map of qvars to rename and a seq of qvars to remove.
  Returns updated result map."
  [res kmap rkeys]
  (reduce-kv
   (fn [res k v]
     (let [k (reduce dissoc k rkeys)]
       (assoc res (set/rename-keys k kmap) v)))
   {} res))

(defn select-renamed-keys
  "Similar to clojure.core/select-keys, but instead of key seq takes a
  map of keys to be renamed (keys in that map are the original keys to
  be selected, their values the renamed keys in returned map)."
  [map aliases]
  (loop [ret {}, keys aliases]
    (if keys
      (let [kv (first keys)
            entry (find map (key kv))]
        (recur
         (if entry
           (conj ret [(val kv) (val entry)])
           ret)
         (next keys)))
      ret)))

(defn order-asc
  [vars res]
  (if (coll? vars)
    (sort-by (fn [r] (reduce #(conj % (r %2)) [] vars)) res)
    (sort-by (fn [r] (get r vars)) res)))

(defn order-desc
  [vars res]
  (if (coll? vars)
    (sort-by
     (fn [r] (reduce #(conj % (r %2)) [] vars))
     #(- (compare % %2))
     res)
    (sort-by #(get % vars) #(- (compare % %2)) res)))

(defn distinct-result-set
  [res]
  (->> res
       (reduce
        (fn [acc r]
          (let [vs (set (vals r))]
            (if (acc vs) acc (assoc acc vs r))))
        {})
       (vals)))

(defn keywordize-result-vars
  [res]
  (map
   (fn [r] (into {} (map (fn [[k v]] [(-> k qvar-name keyword) v]) r)))
   res))

(defn compile-filter-fn
 [filters]
 (fn [res] (every? (fn [[k f]] (if-let [rv (res k)] (f rv) true)) filters)))

(defn compile-bind-fn
 [bindings]
 (fn [res] (reduce-kv (fn [acc k f] (assoc acc k (f res (res k)))) res bindings)))
(defn triples->dot
  "Takes a seq of triples and returns them as digraph spec in
  Graphviz .dot format."
  [triples]
  (apply
   str
   (concat
    "digraph G {\n"
    "node[color=\"black\",style=\"filled\",fontname=\"Inconsolata\",fontcolor=\"white\"];\n"
    "edge[fontname=\"Inconsolata\",fontsize=\"9\"];\n"
    (map
     (fn [t]
       (let [[s p o] (map #(if (string? %) % (pr-str %)) t)]
         (str "\"" s "\" -> \"" o "\" [label=\"" p "\"];\n")))
     triples)
    "}")))

(defn join
  [a b]
  (->> (set/join a b)
       (mapcat
        (fn [k]
          (if (unique-bindings? k)
            [k])))))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb' [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j (first (set/join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))

(defn select-with-bindings
  ([ds t]
     (select-with-bindings ds {} {} t))
  ([ds bindings opts [s p o :as t]]
     (let [vs? (qvar? s), vp? (qvar? p), vo? (qvar? o), vm {}
           [qs vm] (if vs? [(bindings s) (assoc vm 0 s)] [s vm])
           [qp vm] (if vp? [(bindings p) (assoc vm 1 p)] [p vm])
           [qo vm] (if vo? [(bindings o) (assoc vm 2 o)] [o vm])
           res     (api/select ds qs qp qo)]
       (if (seq res)
         (let [verify (triple-verifier t vs? vp? vo?)
               flt (opts :filter)
               res-fn (if flt
                        (let [flt (if (map? flt) (compile-filter-fn flt) flt)]
                          #(if (verify %)
                             (let [vbinds (select-renamed-keys % vm)]
                               (if (flt vbinds) vbinds))))
                        #(if (verify %) (select-renamed-keys % vm)))]
           (->> res
                (map res-fn)
                (filter identity)))))))

(defn select-join
  ([ds patterns]
     (select-join ds {} {} patterns))
  ([ds bindings {bind :bind flt :filter opt? :optional?} patterns]
     (let [[p & ps] (sort-patterns patterns)
           res      (select-with-bindings ds bindings {} p)
           join-fn  (if opt? join-optional join)
           flt      (if (map? flt) (compile-filter-fn flt) flt)
           bind     (if (map? bind) (compile-bind-fn bind) bind)]
       (if (seq res)
         (loop [res res, ps ps]
           (if ps
             (let [r' (select-with-bindings ds bindings {} (first ps))
                   res (if (seq r') (join-fn res r'))]
               (if (seq res)
                 (recur res (next ps))))
             (cond->> res
                      flt (filter flt)
                      bind (map bind))))))))

(defn subvec-slices
  "Takes a min & max count and returns function accepting a vector as
  single arg. When called, returns vector of subvec slices each starting
  at index 0 and with an increasing length from min to max."
  [n1 n2]
  (fn [path]
    (mapv #(subvec path 0 %) (range n1 (inc (min (count path) n2))))))

(defn dfs-forward*
  [ds s p sv acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc sv)
          o (make-qvar)
          r (select-with-bindings ds {s sv} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              ovals (filter (comp not visited) (set (map o r)))]
          (if (seq ovals)
            (                      mapcat
                   (fn [ov] (dfs-forward* ds o p ov acc min max))
                   ovals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn dfs-backward*
  [ds o p ov acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc ov)
          s (make-qvar)
          r (select-with-bindings ds {o ov} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              svals (filter (comp not visited) (set (map s r)))]
          (if (seq svals)
            (                      mapcat
                   (fn [sv] (dfs-backward* ds s p sv acc min max))
                   svals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn select-transitive
  ([ds [s p o :as t]]
     (if (vector? p)
       (select-transitive ds t (count p) (count p))
       (select-transitive ds t 1 1000000)))
  ([ds [s p o] mind maxd]
     (let [mind (max mind 1)
           maxd (max maxd 1)]
       (if (= mind maxd)
         (let [p (if (vector? p) p [p])
               p (take mind (cycle p))]
           (select-join ds {} {} (resolve-path-pattern [s p o])))
         (let [vs? (qvar? s)
               vo? (qvar? o)
               v (make-qvar)
               conf (cond
                     (and vs? vo?) {:s s ;; [?s p ?o]
                                    :o v
                                    :bmap {}
                                    :lookup (fn [b] [(b v) (b s)])
                                    :bind (fn [p] {s (first p) o (peek p)})
                                    :search dfs-forward*}
                     vo?           {:s v ;; [x p ?o]
                                    :o o
                                    :bmap {v s}
                                    :lookup (fn [b] [(b o) (b v)])
                                    :bind (fn [p] {o (peek p)})
                                    :search dfs-forward*}
                     vs?           {:s s ;; [?s p x]
                                    :o v
                                    :bmap {v o}
                                    :lookup (fn [b] [(b s) (b v)])
                                    :bind (fn [p] {s (peek p)})
                                    :search dfs-backward*})
               {:keys [bind search]} conf
               slices (subvec-slices (inc mind) (inc maxd))]
           (->> (select-with-bindings ds (:bmap conf) {} [(:s conf) p (:o conf)])
                (r/map    (:lookup conf))
                (r/mapcat #(search ds v p (% 0) [(% 1)] mind maxd))
                (r/mapcat slices)
                (r/map    bind)
                (into #{})))))))

(defn query-opts
  [filter bind]
  {:filter (if filter
             (if (fn? filter)
               filter
               (compile-filter-fn filter)))
   :bind   (if bind
             (if (fn? bind)
               bind
               (compile-bind-fn bind)))})

(defmulti compile-query-step
  (fn [qfn q type] type))

(defmethod compile-query-step :join
  [qfn {:keys [from join filter bind]} _]
  (prn :join join)
  (let [opts (query-opts filter bind)]
    (fn [res]
      (let [a (qfn res)]
        (if (or (= ::empty a) (seq a))
          (let [b (select-join from {} opts join)
                res' (if (= ::empty a) b (thi.ng.trio.query2/join a b))]
            res')
          a)))))

(defmethod compile-query-step :optional
  [qfn {:keys [from optional filter bind]} _]
  (prn :join-opt optional)
  (let [opts (query-opts filter bind)]
    (fn [res]
      (let [a (qfn res)
            b (select-join from {} opts optional)
            res' (if (= ::empty a) b (join-optional a b))]
        res'))))

(defmethod compile-query-step :multi-query
  [qfn {:keys [from query]} _]
  (prn :multi query)
  (loop [qfn qfn, query query]
    (if query
      (let [q (first query)
            q (if (:from q) q (assoc q :from from))
            qfn (cond
                 (:join q)     (compile-query-step qfn q :join)
                 (:optional q) (compile-query-step qfn q :optional))]
        (recur qfn (next query)))
      qfn)))

(defmethod compile-query-step :bind
  [qfn bind _]
  (prn :bindings bind)
  (let [bind (if (fn? bind)
               bind
               (compile-bind-fn bind))]
    (fn [res] (map bind (qfn res)))))

(defmethod compile-query-step :order-asc
  [qfn order _]
  (prn :order-asc order)
  (fn [res] (order-asc order (qfn res))))

(defmethod compile-query-step :order-desc
  [qfn order _]
  (prn :order-desc order)
  (fn [res] (order-desc order (qfn res))))

(defmethod compile-query-step :group
  [qfn group _]
  (prn :group group)
  (fn [res] (group-by group (qfn res))))

(defmethod compile-query-step :filter-vars
  [qfn vars _]
  (prn :select-vars vars)
  (cond
   (= :* vars)        qfn
   (sequential? vars) (fn [res] (map #(select-keys % vars) (qfn res)))
   :else              (compile-query-step qfn [vars] :filter-vars)))

(defmethod compile-query-step :select
  [qfn {:keys [select from query bind order-asc order-desc group] :as q} _]
  (prn :select select)
  (cond->
   (compile-query-step qfn q :multi-query)
   bind       (compile-query-step bind :bind)
   order-asc  (compile-query-step order-asc :order-asc)
   order-desc (compile-query-step order-desc :order-desc)
   true       (compile-query-step select :filter-vars)
   group      (compile-query-step group :group)))

(defmethod compile-query-step :construct-triples
  [qfn {:keys [construct] :as q} _]
  (fn [res]
    (->> res
         qfn
         (mapcat
          (fn [r]
            (map
             (fn [[s p o]]
               (let [s (if (qvar? s) (r s) s)
                     p (if (qvar? p) (r p) p)
                     o (if (qvar? o) (r o) o)]
                 (if (and s p o) (api/triple s p o))))
             construct)))
         (filter identity)
         (set))))

(defmethod compile-query-step :construct
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :group)
              (assoc :select :*))]
    (-> qfn
        (compile-query-step q :select)
        (compile-query-step q :construct-triples))))

(defmethod compile-query-step :ask
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :bind :group)
              (assoc :select :*))
        qfn (compile-query-step qfn q :select)]
    (fn [res] (if (seq (qfn res)) true false))))

(defmethod compile-query-step :describe
  [qfn {:keys [from describe] :as q} _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :group)
              (assoc :select :*))
        qfn (compile-query-step qfn q :select)
        describe (if (sequential? describe) describe [describe])]
    (fn [res]
      (let [res (qfn res)
            vars (select-keys (accumulate-result-vars res) describe)]
        (if (seq vars)
          (mapcat
           (fn [v]
             (let [vals (vars v)
                   vals (if (coll? vals) vals [vals])]
               (mapcat #(api/select from % nil nil) vals)))
           (keys vars)))))))

(defn compile-query
  [q]
  (if-let [type (some #{:select :ask :construct :describe} (keys q))]
    (compile-query-step identity q type)
    (err/unsupported! "Unsupported query type")))

(defn query
  [q] ((if (fn? q) q (compile-query q)) ::empty))

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/trio/query2.cljx
