(ns thi.ng.trio.entities
  (:require
   [thi.ng.trio.core :as api]
   [thi.ng.trio.query :as q]
   [thi.ng.trio.utils :as utils]
   [thi.ng.trio.entities.utils :as eu]
   [thi.ng.trio.vocabs.rdf :refer [rdf]]
   [thi.ng.validate.core :as v]
   [thi.ng.common.error :as err]))

(defmacro defentity
  [name type props]
  (let [->sym      (comp symbol clojure.core/name)
        props      (merge {:type {:prop (:type rdf)}} props)
        fields     (cons 'id (map ->sym (keys props)))
        ctor-name  (utils/->kebab-case name)
        ctor       (symbol (str 'make- ctor-name))
        dctor      (symbol (str 'describe- ctor-name))
        dctor-as   (symbol (str 'describe-as- ctor-name))
        mctor      (symbol (str 'map-> name))
        wrap-props (symbol (str ctor-name '-props))]
    `(let [props#      ~props
           type#       ~type
           type-prop#  (-> props# :type :prop)
           validators# (eu/build-validators props#)
           inits#      (eu/build-initializers props#)
           defaults#   (eu/build-defaults props#)]
       ;;(prn "-------- " ~type)
       ;;(prn :props ~props)
       ;;(prn :fields ~fields)
       ;;(prn :validators validators#)
       ;;(prn :inits inits#)
       ;;(prn :defaults defaults#)

       (defn ~wrap-props [] props#)

       (defrecord ~name [~@fields]
         api/PTripleSeq
         (~'triple-seq
           [_#] (eu/filtered-triple-seq {~'id (eu/triple-map (~wrap-props) _#)})))

       (defn ~ctor
         {:doc ~(str "Constructs a new `" (namespace-munge *ns*) "." `~name "` entity from given map.\n"
                     "  Applies entity's property intializers, defaults & validation.\n"
                     "  In case of validation errors, throws map of errors using `slingshot/throw+`.")
          :arglists '([~'props])}
         [opts#]
         (let [[opts# err#]
               (-> opts#
                   (assoc :id (or (:id opts#) (utils/new-uuid)))
                   (assoc :type (or (:type opts#) type#))
                   (eu/apply-initializers inits#)
                   (eu/inject-defaults defaults#)
                   (v/validate validators#))]
           (if (nil? err#)
             (~mctor opts#)
             (err/illegal-arg! (pr-str err#)))))

       (defn ~dctor
         {:doc ~(str "Executes a :describe query for given entity ID in graph.\n"
                     "  Returns seq of triples. ")
          :arglists '([~'graph ~'id])}
         [g# id#]
         (q/query
          {:describe '~'?id
           :from g#
           :query [{:where [['~'?id type-prop# type#]]}]
           :values {'~'?id #{id#}}}))

       (defn ~dctor-as
         {:doc ~(str "Constructs a new `" (namespace-munge *ns*) "." `~name "` entity based on a graph query\n"
                     "  for given entity ID and its specified props/rels (any additional rels will be included\n"
                     "  too, but are not validated during construction). If returned query results conflict\n"
                     "  with entity validation, throws map of errors using `slingshot/throw+`.")
          :arglists '([~'graph ~'id])}
         [g# id#]
         (-> (~dctor g# id#)
             (eu/entity-map-from-triples props# id#)
             (~ctor))))))
