(ns thi.ng.trio.query
  (:require
   [thi.ng.trio.core :as api]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.error :as err]
   [thi.ng.common.data.unionfind :as uf]
   [thi.ng.validate.core :as v]
   [clojure.set :as set]
   [clojure.core.reducers :as r]
                                                         ))

(defn compile-filter-fn
  "Takes a map of qvar bindings as keys and filter fns as values.
  Returns fn which accepts a single result map and returns true if all
  given filter fns succeed. Filter fns must accept a single arg, i.e.
  the qvar's value in the result map. Filter fns are only called if
  the qvar is bound. If a filter is defined for a qvar which does not
  ever appear in a solution, the related query will produce no
  results."
  [filters]
  (if filters
    (if (fn? filters)
      filters
      (fn [res]
        (every?
         (fn [[k f]] (let [rv (res k)] (if-not (nil? rv) (f rv) false)))
         filters)))))
(defn compile-bind-fn
  "Takes a map of qvar bindings as keys and bind fns as values.
  Returns a fn which accepts a single result map and returns updated
  map with injected/updated qvars. Bind fns must accept two args: the
  full result map and the value of the fn's associated qvars. The fn's
  return value is used as new binding for the qvar unless nil. Bind
  fns are called regardless if qvar is currently bound (non-nil). This
  is in order to allow for the injection of new vars into the
  solution."
  [bindings]
  (if bindings
    (if (fn? bindings)
      bindings
      (fn [res]
        (reduce-kv
         (fn [acc k f]
           (let [rv (f res (res k))]
             (if (nil? rv) acc (assoc acc k rv))))
         res bindings)))))
(defn query-opts
  "Takes a query options map and injects compiled versions for
  :filter & :bind keys."
  [opts]
  (-> opts
      (update-in [:filter] compile-filter-fn)
      (update-in [:bind] compile-bind-fn)))
(def ^:dynamic *auto-qvar-prefix* "?__q")

(def qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  (fn [x] (and (symbol? x) (= \? (.charAt ^String (name x) 0)))))

(defn auto-qvar?
  "Returns true, if x is an auto-generated qvar
  (a symbol prefixed with *auto-qvar-prefix*)"
  [x] (and (symbol? x) (zero? (.indexOf ^String (name x) ^String *auto-qvar-prefix*))))

(defn auto-qvar
  "Creates a new auto-named qvar (symbol)."
  [] (gensym *auto-qvar-prefix*))

(defn qvar-name
  [x] (-> x name (subs 1)))
(defn collect-source-graphs
  [q] (->> q (map :from) (into #{(:from q)}) (filter identity)))
(defn resolve-path-pattern
  "Takes a path triple pattern where the predicate is a seq of preds.
  Returns seq of query patterns with injected temp qvars for inbetween
  patterns. E.g.

      [?s [p1 p2 p3] ?o]
      => ([?s p1 ?__q0] [?__q0 p2 ?__q1] [?__q1 p3 ?o])"
  [[s p o]]
  (let [vars (->> auto-qvar
                  (repeatedly (dec (count p)))
                  (cons s))]
    (->> (concat (interleave vars p) [o])
         (d/successive-nth 3 2))))

(defn resolve-patterns
  [patterns]
  (mapcat
   (fn [[_ p :as t]]
     (if (vector? p)
       (resolve-path-pattern t)
       [t]))
   patterns))
(defn produce-patterns-with-bound-vars
  "Takes a triple pattern (possibly with variables) and a map of
  possible value sets (each *must* be a set or single value) for each var.
  Produces lazy seq of resulting triple query patterns using cartesian
  product of all values.

      (produce-patterns-with-bound-vars
        [?a :type ?b]
        {?a #{\"me\" \"you\"} ?b #{\"foo\" \"bar\"})
      => ((\"me\" :type \"foo\") (\"me\" :type \"bar\")
          (\"you\" :type \"foo\") (\"you\" :type \"bar\"))"
  [[s p o] bindings]
  (let [s (or (bindings s) s)
        p (or (bindings p) p)
        o (or (bindings o) o)]
    (if (some set? [s p o])
      (d/cartesian-product
       (if (set? s) s #{s}) (if (set? p) p #{p}) (if (set? o) o #{o}))
      [[s p o]])))
(defn sort-patterns
  "Sorts a seq of triple patterns in dependency order using any
  re-occuring vars. Triples with least qvars will be in head
  position."
  [patterns]
  (let [q (map #(let [v (d/filter-tree qvar? %)] [(count v) v %]) patterns)
        singles (->> q (filter #(= 1 (first %))) (mapcat second) set)]
    (->> q
         (sort-by (fn [[c v]] (- (* c 4) (count (filter singles v)))))
         (map peek))))
(defn triple-verifier
  "Takes a triple pattern (potentially with vars) and 3 booleans to
  indicate which SPO is a var. Returns fn which accepts a result triple
  and returns false if any of the vars clash (e.g. a qvar is used multiple
  times but result has different values in each position or likewise, if
  different vars relate to same values)."
  [[ts tp to] vars varp varo]
  (cond
   (and vars varp varo) (cond
                         (= ts tp to) (fn [r] (= (r 0) (r 1) (r 2)))
                         (= ts tp) (fn [r] (and (= (r 0) (r 1)) (not= (r 0) (r 2))))
                         (= ts to) (fn [r] (and (= (r 0) (r 2)) (not= (r 0) (r 1))))
                         (= tp to) (fn [r] (and (= (r 1) (r 2)) (not= (r 0) (r 1))))
                         :else (constantly true))
   (and vars varp)      (if (= ts tp)
                          (fn [r] (= (r 0) (r 1)))
                          (fn [r] (not= (r 0) (r 1))))
   (and vars varo)      (if (= ts to)
                          (fn [r] (= (r 0) (r 2)))
                          (fn [r] (not= (r 0) (r 2))))
   (and varp varo)      (if (= tp to)
                          (fn [r] (= (r 1) (r 2)))
                          (fn [r] (not= (r 1) (r 2))))
   :else                (constantly true)))
(defn unique-bindings?
  "Returns true if all values in the given map are unique, i.e.
  no two keys are mapped to the same value."
  [map] (== (count (into #{} (vals map))) (count map)))

(defn accumulate-result-vars
  "Takes a seq of query result maps and returns a map of all found
  qvars as keys and their value sets."
  ([res] (accumulate-result-vars {} nil res))
  ([acc vars res]
     (let [acc (reduce
                (fn [acc b]
                  (merge-with
                   (fn [a b] (if (set? a) (conj a b) (hash-set a b)))
                   acc (if vars (select-keys b vars) b)))
                acc res)]
       (reduce-kv
        (fn [acc k v] (if (set? v) acc (assoc acc k #{v})))
        acc acc))))

(defn select-renamed-keys
  "Similar to clojure.core/select-keys, but instead of key seq takes a
  map of keys to be renamed (keys in that map are the original keys to
  be selected, their values the renamed keys in returned map)."
  [ret map aliases]
  (loop [ret ret, keys aliases]
    (if keys
      (let [kv (first keys)
            entry (find map (key kv))]
        (recur
         (if entry
           (assoc ret (val kv) (val entry))
           ret)
         (next keys)))
      ret)))

(defn order-asc
  [vars res]
  (if (coll? vars)
    (sort-by (fn [r] (reduce #(conj % (r %2)) [] vars)) res)
    (sort-by (fn [r] (get r vars)) res)))

(defn order-desc
  [vars res]
  (if (coll? vars)
    (sort-by
     (fn [r] (reduce #(conj % (r %2)) [] vars))
     #(- (compare % %2))
     res)
    (sort-by #(get % vars) #(- (compare % %2)) res)))

(defn distinct-result-set
  [res]
  (->> res
       (reduce
        (fn [acc r]
          (let [vs (set (vals r))]
            (if (acc vs) acc (assoc acc vs r))))
        {})
       (vals)))

(defn keywordize-result-vars
  [res]
  (map
   (fn [r] (into {} (map (fn [[k v]] [(-> k qvar-name keyword) v]) r)))
   res))
(defn triples->dot
  "Takes a seq of triples and returns them as digraph spec in
  Graphviz .dot format."
  [triples]
  (apply
   str
   (concat
    "digraph G {\n"
    "node[color=\"black\",style=\"filled\",fontname=\"Inconsolata\",fontcolor=\"white\"];\n"
    "edge[fontname=\"Inconsolata\",fontsize=\"9\"];\n"
    (map
     (fn [t]
       (let [[s p o] (map #(if (string? %) % (pr-str %)) t)]
         (str "\"" s "\" -> \"" o "\" [label=\"" p "\"];\n")))
     triples)
    "}")))

(defn join
  [a b]
  (->> (set/join a b)
       (mapcat
        (fn [k]
          (if (unique-bindings? k)
            [k])))))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb' [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j (first (set/join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))

(defn minus
  [a b]
  (let [vars (accumulate-result-vars b)]
    (reduce-kv
     (fn [res k v]
       (let [v (if (coll? v) v [v])]
         (reduce
          (fn [res v] (vec (remove (fn [r] (= (r k) v)) res))) res v)))
     a vars)))

(defn union
  [a b]
                             
  (concat a b))

(defn unbound-var?
  [bindings v? x] (and v? (not (bindings x))))

(defn bind-translator
  [vs? vp? vo? [s p o]]
  (if vs?
    (if vp?
      (if vo?
        (fn [r] {s (r 0) p (r 1) o (r 2)})
        (fn [r] {s (r 0) p (r 1)}))
      (if vo?
        (fn [r] {s (r 0) o (r 2)})
        (fn [r] {s (r 0)})))
    (if vp?
      (if vo?
        (fn [r] {p (r 1) o (r 2)})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defn unbound-vars-translator
  [bindings vs? vp? vo? [s p o]]
  (if (unbound-var? bindings vs? s)
    (if (unbound-var? bindings vp? p)
      (if (unbound-var? bindings vo? o)
        (fn [p] [nil nil nil])
        (fn [p] [nil nil (p 2)]))
      (if (unbound-var? bindings vo? o)
        (fn [p] [nil (p 1) nil])
        (fn [p] (assoc p 0 nil))))
    (if (unbound-var? bindings vp? p)
      (if (unbound-var? bindings vo? o)
        (fn [p] [(p 0) nil nil])
        (fn [p] (assoc p 1 nil)))
      (if (unbound-var? bindings vo? o)
        (fn [p] (assoc p 2 nil))
        identity))))

(defn select-with-bindings-alts
  [ds bindings opts [s p o :as t]]
  (let [vs?      (qvar? s), vp? (qvar? p), vo? (qvar? o)
        vmap     (bind-translator vs? vp? vo? t)
        verify   (triple-verifier t vs? vp? vo?)
        flt      (compile-filter-fn (opts :filter))
        res-fn   (if (:triples opts)
                   (if flt
                     #(if (verify %)
                        (let [vbinds (vmap %)]
                          (if (flt vbinds) [vbinds %])))
                     #(if (verify %) [(vmap %) %]))
                   (if flt
                     #(if (verify %)
                        (let [vbinds (vmap %)]
                          (if (flt vbinds) vbinds)))
                     #(if (verify %) (vmap %))))
        qs (if vs? (bindings s) s)
        qp (if vp? (bindings p) p)
        qo (if vo? (bindings o) o)
                                                  
        res (api/select-with-alts ds qs qp qo)]
    (if (seq res)
      (loop [acc (transient []), res res]
        (if res
          (let [r (res-fn (first res))]
            (if r
              (recur (conj! acc r) (next res))
              (recur acc (next res))))
          (persistent! acc))))))

(defn select-with-bindings
  [ds bindings opts [s p o :as t]]
  (let [patterns (produce-patterns-with-bound-vars t bindings)
        vs?      (qvar? s), vp? (qvar? p), vo? (qvar? o)
        vmap     (bind-translator vs? vp? vo? t)
        pmap     (unbound-vars-translator bindings vs? vp? vo? t)
        verify   (triple-verifier t vs? vp? vo?)
        flt      (compile-filter-fn (opts :filter))
        res-fn   (if (:triples opts)
                   (if flt
                     #(if (verify %)
                        (let [vbinds (vmap %)]
                          (if (flt vbinds) [vbinds %])))
                     #(if (verify %) [(vmap %) %]))
                   (if flt
                     #(if (verify %)
                        (let [vbinds (vmap %)]
                          (if (flt vbinds) vbinds)))
                     #(if (verify %) (vmap %))))]
                                    
    (loop [acc (transient []), ps patterns]
      (if ps
        (let [[qs qp qo] (pmap (vec (first ps)))
                                                    
              res (api/select ds qs qp qo)]
          (if (seq res)
            (recur
             (loop [acc acc, res res]
               (if res
                 (let [r (res-fn (first res))]
                   (if r
                     (recur (conj! acc r) (next res))
                     (recur acc (next res))))
                 acc))
             (next ps))
            (recur acc (next ps))))
        (persistent! acc)))))

(defn select-join
  ([ds patterns]
     (select-join ds {} {} patterns))
  ([ds bindings {bind :bind flt :filter opt? :optional?} patterns]
     (let [[p & ps] (sort-patterns patterns)
           res      (select-with-bindings ds bindings {} p)
           join-fn  (if opt? join-optional join)
           flt      (compile-filter-fn flt)
           bind     (compile-bind-fn bind)]
       (if (seq res)
         (loop [res res, ps ps]
           (if ps
             (let [binds (merge-with into (accumulate-result-vars res) bindings)
                   r' (select-with-bindings-alts ds binds {} (first ps))
                   res (if (seq r') (join-fn res r'))]
               (if (seq res)
                 (recur res (next ps))))
             (cond->> res
                      flt (filter flt)
                      bind (map bind))))))))

(defn subvec-slices
  "Takes a min & max count and returns function accepting a vector as
  single arg. When called, returns vector of subvec slices each starting
  at index 0 and with an increasing length from min to max."
  [n1 n2]
  (fn [path]
    (mapv #(subvec path 0 %) (range n1 (inc (min (count path) n2))))))

(defn dfs-forward*
  [ds s p sv acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc sv)
          o (auto-qvar)
          r (select-with-bindings ds {s sv} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              ovals (filter (comp not visited) (set (map o r)))]
          (if (seq ovals)
            (                      mapcat
                   (fn [ov] (dfs-forward* ds o p ov acc min max))
                   ovals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn dfs-backward*
  [ds o p ov acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc ov)
          s (auto-qvar)
          r (select-with-bindings ds {o ov} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              svals (filter (comp not visited) (set (map s r)))]
          (if (seq svals)
            (                      mapcat
                   (fn [sv] (dfs-backward* ds s p sv acc min max))
                   svals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn select-transitive
  ([ds [s p o :as t]]
     (if (vector? p)
       (select-transitive ds t (count p) (count p))
       (select-transitive ds t 1 1000000)))
  ([ds [s p o] mind maxd]
     (let [mind (max mind 1)
           maxd (max maxd 1)]
       (if (= mind maxd)
         (let [p (if (vector? p) p [p])
               p (take mind (cycle p))]
           (select-join ds {} {} (resolve-path-pattern [s p o])))
         (let [vs? (qvar? s)
               vo? (qvar? o)
               v (auto-qvar)
               conf (cond
                     (and vs? vo?) {:s s ;; [?s p ?o]
                                    :o v
                                    :bmap {}
                                    :lookup (fn [b] [(b v) (b s)])
                                    :bind (fn [p] {s (first p) o (peek p)})
                                    :search dfs-forward*}
                     vo?           {:s v ;; [x p ?o]
                                    :o o
                                    :bmap {v s}
                                    :lookup (fn [b] [(b o) (b v)])
                                    :bind (fn [p] {o (peek p)})
                                    :search dfs-forward*}
                     vs?           {:s s ;; [?s p x]
                                    :o v
                                    :bmap {v o}
                                    :lookup (fn [b] [(b s) (b v)])
                                    :bind (fn [p] {s (peek p)})
                                    :search dfs-backward*})
               {:keys [bind search]} conf
               slices (subvec-slices (inc mind) (inc maxd))]
           (->> (select-with-bindings ds (:bmap conf) {} [(:s conf) p (:o conf)])
                (r/map    (:lookup conf))
                (r/mapcat #(search ds v p (% 0) [(% 1)] mind maxd))
                (r/mapcat slices)
                (r/map    bind)
                (into #{})))))))

(def validator-select
  {:select    (v/alts
               [(v/symbol) (v/keyword) (v/sequential)]
               "must be a qvar, keyword or seq")
        
                               
                      
                                                                     
                                                                                
                                                                                                        
   :query     {:* (v/map)}
   :bind      (v/optional
               (v/alts
                [(v/map) (v/function)]
                "must be a map or fn"))
   :group     (v/optional
               (v/alts
                [(v/symbol) (v/sequential) (v/function)]
                "must be a qvar, qvar seq or fn"))
   :aggregate (v/optional (v/map))
   :filter    (v/optional
               (v/alts
                [(v/map) (v/function)]
                "must be a map or fn"))})
(defn validate
  [q spec]
  (try
    (let [[ret err] (v/validate q spec)]
      (if-not err
        ret
        (err/throw! (str "Error compiling query: " err))))
    (catch                        js/Error e
      (err/throw!                              e))))
  

(defmulti compile-query-step
  (fn [qfn q type] type))
(defmethod compile-query-step :select
  [qfn {:keys [select bind order-asc order-desc distinct] :as q} _]
                              
  (validate q validator-select)
  (validate {:from (collect-source-graphs q)} {:from (v/required)})
  (cond->
   (compile-query-step qfn q :query-dispatch)
   bind       (compile-query-step bind :bind)
   order-asc  (compile-query-step order-asc :order-asc)
   order-desc (compile-query-step order-desc :order-desc)
   true       (compile-query-step q :project-vars)
   distinct   (compile-query-step q :distinct-bindings)))
(defn construct-triples
  [patterns res]
  (->> res
       (mapcat
        (fn [r]
          (map
           (fn [[s p o]]
             (let [s (if (qvar? s) (r s) s)
                   p (if (qvar? p) (r p) p)
                   o (if (qvar? o) (r o) o)]
               (if (and s p o) (api/triple s p o))))
           patterns)))
       (filter identity)
       (set)))

(defmethod compile-query-step :construct-triples
  [qfn {:keys [construct] :as q} _]
  (fn [res] (->> res qfn (construct-triples construct))))

(defmethod compile-query-step :construct
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :group)
              (assoc :select :*))]
    (-> qfn
        (compile-query-step q :select)
        (compile-query-step q :construct-triples))))
(defmethod compile-query-step :ask
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :bind :group :aggregate)
              (assoc :select :*))
        qfn (compile-query-step qfn q :select)]
    (fn [res] (if (seq (qfn res)) true false))))
(defmethod compile-query-step :describe
  [qfn {:keys [describe] :as q} _]
  (let [q        (-> q
                     (dissoc :order-asc :order-desc :group :aggregate)
                     (assoc :select :*))
        qfn      (compile-query-step qfn q :select)
        describe (if (sequential? describe) describe [describe])
        graphs   (collect-source-graphs q)]
    (fn [res]
      (let [vars (select-keys (accumulate-result-vars (qfn res)) describe)]
        (if (seq vars)
          (reduce-kv
           (fn [acc _ vals]
             (reduce
              (fn [acc from]
                (-> acc
                    (into (mapcat #(api/select from % nil nil) vals))
                    (into (mapcat #(api/select from nil nil %) vals))))
              acc graphs))
           #{} vars))))))
(defn- maybe-deref
  [x] (try @x (catch                        js/Error e x)))

(defn- unpack-model
  [model]
  (let [model (maybe-deref model)]
    (if (satisfies? api/PModelSelect model)
      model
      (if (satisfies? api/PModelConvert model)
        (api/as-model model)
        (err/illegal-arg! "Referenced data model doesn't implement PModelSelect")))))

(defmethod compile-query-step :join
  [qfn {:keys [from values where] :as q} _]
                                  
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)]
        (if (or (= ::empty a) (seq a))
          (let [b (select-join (unpack-model from) values opts where)
                res' (if (= ::empty a) b (join a b))]
            res')
          a)))))

(defmethod compile-query-step :minus
  [qfn {:keys [from values minus] :as q} _]
                                   
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)]
        (if (seq a)
          (thi.ng.trio.query/minus a (select-join (unpack-model from) values opts minus))
          a)))))

(defmethod compile-query-step :optional
  [qfn {:keys [from values optional] :as q} _]
                                         
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)]
        (if (or (= ::empty a) (seq a))
          (let [b (select-join (unpack-model from) values opts optional)
                res' (if (= ::empty a) b (join-optional a b))]
            res')
          a)))))

(defmethod compile-query-step :union
  [qfn {:keys [from values union] :as q} _]
                                   
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)
            b (select-join (unpack-model from) values opts union)
            res' (if (= ::empty a) b (thi.ng.trio.query/union a b))]
        res'))))
(defmethod compile-query-step :query-dispatch
  [qfn {:keys [from values query]} _]
                                     
  (loop [qfn qfn, query query, first? true]
    (if query
      (let [q (first query)
            from (or (:from q) from)
            from' (maybe-deref from)
            q (if (or (satisfies? api/PModelSelect from')
                      (satisfies? api/PModelConvert from'))
                (assoc q :from from))
            q (if values
                (assoc q :values (merge-with (fnil into #{}) values (:values q)))
                (update-in q [:values] (fnil identity {})))
            _ (when (and first? (seq (select-keys q [:optional :minus :union])))
                (err/illegal-arg! "First sub-query type must be of type :where"))
            qfn (cond
                 (:where q)    (compile-query-step qfn q :join)
                 (:optional q) (compile-query-step qfn q :optional)
                 (:minus q)    (compile-query-step qfn q :minus)
                 (:union q)    (compile-query-step qfn q :union))]
        (recur qfn (next query) false))
      qfn)))
(defmethod compile-query-step :bind
  [qfn bind _]
                              
  (let [bind (compile-bind-fn bind)]
    (fn [res] (map bind (qfn res)))))

(defmethod compile-query-step :order-asc
  [qfn order _]
                                
  (fn [res] (order-asc order (qfn res))))

(defmethod compile-query-step :order-desc
  [qfn order _]
                                 
  (fn [res] (order-desc order (qfn res))))

(defmethod compile-query-step :distinct-bindings
  [qfn spec _]
  (if (:group spec)
    (fn [res]
      (reduce-kv (fn [acc k v] (assoc acc k (into #{} v))) {} (qfn res)))
    (fn [res]
      (into #{} (qfn res)))))
(defn renamed-vars
  [binding-map]
  (reduce-kv
   (fn [vars k v]
     (if (and (map? v) (:as v))
       (assoc vars (:as v) v)
       (assoc vars k v)))
   {} binding-map))

(defn selected-var-keys
  [qvars]
  (reduce
   (fn [vars v]
     (if (map? v)
       (merge vars v)
       (assoc vars v v)))
   {} qvars))

(defn parse-var-bind-spec
  [v spec] (if (map? spec) [(or (:use spec) v) (:fn spec)] [nil spec]))

(defn compute-select-bindings
  [res svars]
  (reduce-kv
   (fn [acc v spec]
     (let [[k f] (parse-var-bind-spec v spec)
           v' (if k
                (if (fn? f) (f (res k)) (res k))
                (f res))]
       (if v'
         (assoc acc v v')
         acc)))
   {} svars))

(defn validate-selected-vars
  [svars]
  (if (svars :*)
    (if (> (count svars) 1)
      (err/illegal-arg! (str "can't use :* selector with other vars: " (vals svars)))
      (dissoc svars :*))
    svars))

(defmethod compile-query-step :project-vars
  [qfn {:keys [select group aggregate filter] :as q} _]
                                    
  (let [select     (if (sequential? select) select [select])
        svars      (selected-var-keys select)
        catch-all? (svars :*)
        svars      (validate-selected-vars svars)
        agg-only?  (every? (set (keys aggregate)) (keys svars))
        filter     (compile-filter-fn filter)]
    (if group
      (compile-query-step qfn [group aggregate svars filter agg-only? catch-all?] :project-groups)
      (if (seq aggregate)
        (if (and agg-only? (not catch-all?))
          (compile-query-step qfn [aggregate svars filter] :project-vars-aggregates)
          (compile-query-step qfn [aggregate svars catch-all? filter] :project-vars-mixed))
        (if catch-all?
          (compile-query-step qfn filter :project-vars-all)
          (compile-query-step qfn [svars filter] :project-vars-simple))))))
(defn compile-group-fn
  [group]
  (if (fn? group)
    group
    (if (sequential? group)
      (fn [res] (reduce #(conj % (res %2)) [] group))
      (fn [res] (res group)))))

(defn compute-aggregates
  [vars vals]
  (reduce-kv
   (fn [inner v spec]
     (let [[k f] (parse-var-bind-spec v spec)]
       (if k
         (assoc inner v (f (map #(get % k) vals)))
         (assoc inner v (f vals)))))
   {} vars))

(defn project-aggregates-only
  [avars svars filter]
  (fn [vals]
    (let [agg (compute-aggregates avars vals)]
      (if (or (nil? filter) (filter agg))
        (compute-select-bindings agg svars)))))

(defn project-aggregates-mixed
  [avars svars all? flt]
  (fn [vals]
    (let [agg  (compute-aggregates avars vals)
          vals (map #(merge % agg) vals)
          vals (if flt (filter flt vals) vals)]
      (if (seq vals)
        (if all?
          vals
          (map #(compute-select-bindings % svars) vals))))))

(defn project-aggregates-simple
  [vars flt]
  (fn [vals]
    (let [vals (if flt (filter flt vals) vals)]
      (if (seq vals)
        (map #(compute-select-bindings % vars) vals)))))

(defn project-aggregates-all
  [flt]
  (fn [vals]
    (let [vals (if flt (filter flt vals) vals)]
      (if (seq vals) vals))))

(defmethod compile-query-step :project-vars-aggregates
  [qfn [avars svars filter] _]
                                             
  (let [project-fn (project-aggregates-only avars svars filter)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-mixed
  [qfn [avars svars all? flt] _]
                                                      
  (let [project-fn (project-aggregates-mixed avars svars all? flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-simple
  [qfn [vars flt] _]
                                           
  (let [project-fn (project-aggregates-simple vars flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-all
  [qfn flt _]
                                   
  (let [project-fn (project-aggregates-all flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defn project-groups-with
  [project-fn res]
  (reduce-kv
   (fn [acc k vals]
     (if-let [v (project-fn vals)]
       (assoc acc k v)
       acc))
   {} res))

(defmethod compile-query-step :group-aggregates-only
  [qfn [avars svars filter] _]
                                             
  (let [project-fn (project-aggregates-only avars svars filter)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-mixed
  [qfn [avars svars all? flt] _]
                                                      
  (let [project-fn (project-aggregates-mixed avars svars all? flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-simple
  [qfn [vars flt] _]
                                           
  (let [project-fn (project-aggregates-simple vars flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-all
  [qfn flt _]
                                   
  (let [project-fn (project-aggregates-all flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :project-groups
  [qfn [group aggregate svars filter agg-only? catch-all?] _]
                                               
  (let [group (compile-group-fn group)
        qfn   (fn [res] (group-by group (qfn res)))]
    (if (seq aggregate)
      (if (and agg-only? (not catch-all?))
        (compile-query-step qfn [aggregate svars filter] :group-aggregates-only)
        (compile-query-step qfn [aggregate svars catch-all? filter] :group-aggregates-mixed))
      (if catch-all?
        (compile-query-step qfn filter :group-aggregates-all)
        (compile-query-step qfn [svars filter] :group-aggregates-simple)))))

(defn compile-query
  "Pre-compiles a query spec hashmap into a query function to be
  executed with `query`. Throws an exception if compilation fails
  due to illegal args in spec."
  [q]
  (if-let [type (some #{:select :ask :construct :describe} (keys q))]
    (compile-query-step identity q type)
    (err/illegal-arg! "Unsupported query type")))

(defn query
  "Takes a single query spec (raw or compiled) and executes it.
  Returned result format depends on query type. A failed query
  either results in nil or an empty collection."
  [q]
  (let [res ((if (fn? q) q (compile-query q)) ::empty)]
    (if (seq res) res)))

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/trio/query.cljx
