(ns thi.ng.trio.vocabs.utils
  (:require
   [thi.ng.trio.core :as api]
   #+clj [clojure.java.io :as io]
   #+clj [clojure.edn :as edn]
   [clojure.string :as str]))

(defn expand-pname
  [prefixes x]
  (if (string? x)
    (let [[[_ p n]] (re-seq #"^([A-Za-z0-9\-_]+):([A-Za-z0-9\-_]*)$" x)]
      (if-let [p (prefixes p)]
        (str p n)))))

(defn expand-pname-maybe
  [prefixes x]
  (or (expand-pname prefixes x) x))

(defn expand-pnames-in-triple
  [prefixes [s p o]]
  (api/triple
   (expand-pname-maybe prefixes s)
   (expand-pname-maybe prefixes p)
   (expand-pname-maybe prefixes o)))

(defn find-prefix
  [prefixes ^String uri]
  (loop [[[pre puri] & more] (seq prefixes)]
    (when pre
      (if (== 0 (.indexOf uri ^String puri))
        [pre (subs uri (count puri))]
        (recur more)))))

(defn vocabs-from-model
  [prefixes graph]
  (->> graph
       (api/subjects)
       (reduce
        (fn [vocabs uri]
          (let [[p n] (find-prefix prefixes uri)]
            (if p
              (assoc-in vocabs [(keyword p) (keyword n)] uri)
              vocabs)))
        {})))

(defn triple-seq-with-prefixes
  [prefixes triples]
  (->> triples
       (api/triple-seq)
       (mapv #(expand-pnames-in-triple prefixes %))))

#+clj
(defn load-vocab-triples
  [src]
  (let [{:keys [prefixes triples]}
        (->> src
             (io/input-stream)
             (slurp)
             (edn/read-string))]
    {:prefixes prefixes
     :triples (triple-seq-with-prefixes prefixes triples)}))

#+clj
(defn load-vocabs-as-model
  [src]
  (let [{:keys [prefixes triples]} (load-vocab-triples src)]
    (vocabs-from-model prefixes (api/as-model triples))))

(defn make-vocab
  [uri xs]
  (->> xs (map (fn [x] [x (str uri (name x))])) (into {})))
