(ns ^{:mranderson/inlined true} mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.coercer
  "Notes:
   - Coercion of records handled specially due different behaviors on different JVMs.
     Must be in equality of specificity of in our protocol extensions.

     From https://clojure.org/reference/protocols:
     
       if one interface is derived from the other, the more derived is used, 
       else which one is used is unspecified" 
  (:require
   [clojure.string :as string]
   #?@(:clj
       [[mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.comment]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.fn]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.forms]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.integer]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.keyword :refer [keyword-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.meta :refer [meta-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.namespaced-map]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.protocols :as node :refer [NodeCoerceable coerce]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.quote]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.reader-macro :refer [reader-macro-node var-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.regex :refer [regex-node pattern-string-for-regex]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.seq :refer [vector-node list-node set-node map-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.stringz :refer [string-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.token :refer [token-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.uneval]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.whitespace :as ws]]
       :cljs
       [[mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.comment :refer [CommentNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.fn :refer [FnNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.forms :refer [FormsNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.integer :refer [IntNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.keyword :refer [KeywordNode keyword-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.meta :refer [MetaNode meta-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.namespaced-map :refer [NamespacedMapNode MapQualifierNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.protocols :as node :refer [NodeCoerceable coerce]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.quote :refer [QuoteNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.reader-macro :refer [ReaderNode ReaderMacroNode DerefNode reader-macro-node var-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.regex :refer [RegexNode regex-node pattern-string-for-regex]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.seq :refer [SeqNode vector-node list-node set-node map-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.stringz :refer [StringNode string-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.token :refer [TokenNode SymbolNode token-node]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.uneval :refer [UnevalNode]]
        [mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj.node.whitespace :refer [WhitespaceNode CommaNode NewlineNode] :as ws]]))
  #?(:clj
     (:import [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.comment CommentNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.fn FnNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.forms FormsNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.integer IntNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.keyword KeywordNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.meta MetaNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.namespaced_map NamespacedMapNode MapQualifierNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.quote QuoteNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.reader_macro ReaderNode ReaderMacroNode DerefNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.regex RegexNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.seq SeqNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.stringz StringNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.token TokenNode SymbolNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.uneval UnevalNode]
              [mranderson.inlined.rewrite_clj.v1v0v682_alpha.rewrite_clj.node.whitespace WhitespaceNode CommaNode NewlineNode])))

#?(:clj (set! *warn-on-reflection* true))


;; ## Rewrite-clj nodes coerce to themselves
;; It is important that all mranderson.inlined.rewrite-clj.v1v0v682-alpha.rewrite-clj nodes be specified, else we'll coerce them to records
(extend-protocol NodeCoerceable
  CommaNode          (coerce [v] v)
  CommentNode        (coerce [v] v)
  DerefNode          (coerce [v] v)
  FnNode             (coerce [v] v)
  FormsNode          (coerce [v] v)
  IntNode            (coerce [v] v)
  KeywordNode        (coerce [v] v)
  MapQualifierNode   (coerce [v] v)
  MetaNode           (coerce [v] v)
  NamespacedMapNode  (coerce [v] v)
  NewlineNode        (coerce [v] v)
  QuoteNode          (coerce [v] v)
  ReaderMacroNode    (coerce [v] v)
  ReaderNode         (coerce [v] v)
  RegexNode          (coerce [v] v)
  SeqNode            (coerce [v] v)
  StringNode         (coerce [v] v)
  SymbolNode         (coerce [v] v)
  TokenNode          (coerce [v] v)
  UnevalNode         (coerce [v] v)
  WhitespaceNode     (coerce [v] v))

;; ## Helpers

(defn- split-to-lines
  "Slightly different than string/split-lines in that:
   - includes all lines even if empty
   - behaves the same on clj and cljs"
  [s]
  (loop [s s
         lines []]
    (if-let [m (first (re-find #"(\r\n|\r|\n)" s))]
      (let [eol-ndx (string/index-of s m)]
        (recur (subs s (+ eol-ndx (count m)))
               (conj lines (subs s 0 eol-ndx))))
      (conj lines s))))

(defn node-with-meta
  [n value]
  (if #?(:clj (instance? clojure.lang.IMeta value)
         :cljs (satisfies? IWithMeta value))
    (let [mta (node/meta-elided value)]
      (if (empty? mta)
        n
        (meta-node (coerce mta) n)))
    n))

(let [comma (ws/whitespace-nodes ", ")
      space (ws/whitespace-node " ")]
  (defn- map->children
    [m]
    (->> (mapcat
          (fn [[k v]]
            (list* (coerce k) space (coerce v) comma))
          m)
         (drop-last (count comma))
         (vec))))

(defn- record-node
  [m]
  (reader-macro-node
   [(token-node #?(:clj (symbol (.getName ^Class (class m)))
                   :cljs ;; this is a bit hacky, but is one way of preserving original name
                   ;; under advanced cljs optimizations
                   (let [s (pr-str m)]
                     (symbol (subs s 1 (string/index-of s "{"))))))
    (map-node (map->children m))]))

(defn- create-map-node [children]
  (node-with-meta
   (map-node (map->children children))
   children))

;; ## Tokens (and special case for record for cljs)

(extend-protocol NodeCoerceable
  #?(:clj clojure.lang.Keyword :cljs Keyword)
  (coerce [v]
    (keyword-node v)))

(extend-protocol NodeCoerceable
  #?(:clj java.lang.String :cljs string)
  (coerce [v]
    (string-node (split-to-lines v))))

#?(:clj
   (extend-protocol NodeCoerceable
     Object 
     (coerce [v]
       (node-with-meta
        (token-node v)
        v)))
   :cljs
   (extend-protocol NodeCoerceable
     default
     (coerce [v]
       (node-with-meta
        ;; in cljs, this is where we check for a record, in clj it happens under map handling
        (if (record? v)
          (record-node v)
          (token-node v))
        v))))

(extend-protocol NodeCoerceable
  nil
  (coerce [v]
    (token-node nil)))

;; ## Regex

(extend-protocol NodeCoerceable
  #?(:clj java.util.regex.Pattern :cljs js/RegExp)
  (coerce [v]
    (regex-node (pattern-string-for-regex v))))

;; ## Seqs

(defn- seq-node
  [f sq]
  (node-with-meta
    (->> (map coerce sq)
         (ws/space-separated)
         (vec)
         (f))
    sq))

(extend-protocol NodeCoerceable
  #?(:clj clojure.lang.IPersistentVector :cljs PersistentVector)
  (coerce [sq]
    (seq-node vector-node sq))
  #?(:clj clojure.lang.IPersistentList :cljs List)
  (coerce [sq]
    (seq-node list-node sq))
  #?(:clj clojure.lang.Cons :cljs Cons)
  (coerce [sq]
    (seq-node list-node sq))
  #?(:clj clojure.lang.IPersistentSet :cljs PersistentHashSet)
  (coerce [sq]
    (seq-node set-node sq)))

#?(:cljs
   ;; cljs empty list is special
   (extend-protocol NodeCoerceable
     EmptyList
     (coerce [sq]
       (seq-node list-node sq))))

;; ## Maps

#?(:clj
   (extend-protocol NodeCoerceable
     clojure.lang.IPersistentMap
     (coerce [m]
       (if (record? m)
         (node-with-meta (record-node m) m)
         (create-map-node m))))
   :cljs
   (do
     (extend-protocol NodeCoerceable
       PersistentHashMap
       (coerce [m] (create-map-node m)))
     (extend-protocol NodeCoerceable
       PersistentArrayMap
       (coerce [m] (create-map-node m)))))

;; ## Vars

(extend-protocol NodeCoerceable
  #?(:clj clojure.lang.Var :cljs Var)
  (coerce [v]
    (-> (str v)
        (subs 2)
        (symbol)
        (token-node)
        (vector)
        (var-node))))
