(ns thosmos.util
  (:require [clojure.core.rrb-vector :as fv]
            [clojure.pprint :refer [pprint]]))

(defmacro functionize [macro]
  `(fn [& args#] (eval (cons '~macro args#))))

(defn merge-tree [a b]
  (if (and (map? a) (map? b))
    (merge-with #(merge-tree %1 %2) a b)
    b))

(defn sort-maps-by
  "Sort a sequence of maps (ms) on multiple keys (ks)"
  [ms ks]
  (sort-by #(vec (map % ks)) ms))

;; from https://clojurian.blogspot.com/2012/11/beware-of-mapcat.html
(defn eager-mapcat
  [f coll]
  (lazy-seq
    (if (not-empty coll)
      (concat
        (f (first coll))
        (eager-mapcat f (rest coll))))))

;; http://www.markhneedham.com/blog/2014/04/06/clojure-not-so-lazy-sequences-a-k-a-chunking-behaviour/
(defn unchunk [s]
  (when (seq s)
    (lazy-seq
      (cons (first s)
        (unchunk (next s))))))

(defn format-instant
  ([instant] (.format (java.text.SimpleDateFormat. "yyyy-MM-dd HH:mm:ss") instant))
  ([instant format] (.format (java.text.SimpleDateFormat. format) instant)))

(defn zoned-datetime-from-instant [ins]
  (-> ins
    (.getTime)
    (/ 1000)
    java.time.Instant/ofEpochSecond
    (java.time.ZonedDateTime/ofInstant
      (java.time.ZoneId/of "America/Los_Angeles"))))

(defn vec-remove [pos coll]
  (fv/catvec (fv/subvec coll 0 pos) (fv/subvec coll (inc pos) (count coll))))

(defn round2
  "Round a double to the given precision (number of significant digits)"
  [precision d]
  (let [factor (Math/pow 10 precision)]
    (/ (Math/round (* d factor)) factor)))

(defn square [n] (* n n))

(defn mean [a] (/ (reduce + a) (count a)))

(defn std-dev
  [a]
  (let [mn (mean a)]
    (Math/sqrt
      (/ (reduce #(+ %1 (square (- %2 mn))) 0 a)
        (dec (count a))))))

(defn percent-prec [avg stddev]
  (* (/ stddev avg) 100))

;; if it's nil, make it 0
(defn inc! [val]
  (inc (or val 0)))

;; if it's nil, make it a []
(defn conjv [coll val]
  (conj (or coll []) val))

(defn ppspit [f content]
  (with-open [^java.io.Writer w (apply clojure.java.io/writer f nil)]
    (pprint content w)))