(ns ctim.schemas.exploit-target
  (:require [ctim.schemas.common :as c]
            [ctim.schemas.relationships :as rel]
            [ctim.schemas.vocabularies :as v]
            [schema.core :as s]
            [ring.swagger.schema :refer [describe]]
            [schema-tools.core :as st]))

(s/defschema Vulnerability
  "See http://stixproject.github.io/data-model/1.2/et/VulnerabilityType/"
  (st/merge
   {:title (describe s/Str "title for this vulnerability")
    :description (describe s/Str "title for this vulnerability")}
   (st/optional-keys
    {:is_known (describe
                s/Bool
                (str "whether or not the vulnerability is known (i.e. not a 0-day)"
                     " at the time of characterization."))
     :is_public_acknowledged (describe
                              s/Bool
                              (str "whether or not the vulnerability is"
                                   " publicly acknowledged by the vendor"))
     :short_description (describe s/Str
                                  "short text description of this vulnerability")
     :cve_id (describe s/Str "CVE identifier")
     :osvdb_id (describe s/Int "OSVDB identifier")
     :source (describe ; source of CVE or OSVDB ref
              s/Str
              "the source of the CVE or OSVDB as a textual description or URL")
     :discovered_datetime (describe
                           c/Time
                           (str "date and time that this vulnerability"
                                " was first discovered")) ; Simplified
     :published_datetime (describe
                          c/Time
                          (str "date and time that this vulnerability"
                               " was first published")) ; Simplified
     ;; TODO - :affected_software below is greatly simplified, should it be expanded?
     :affected_software (describe
                         [s/Str]
                         (str "list of platforms and software that"
                              " are affected by this vulnerability"))
     :references (describe
                  [c/URI]
                  "list of external references describing this vulnerability")
     ;; Not provided: CVSS_Score ; Should it be?
     })))

(s/defschema Weakness
  "See http://stixproject.github.io/data-model/1.2/et/WeaknessType/"
  {:description (describe s/Str "text description of this Weakness")
   (s/optional-key :cwe_id)
   (describe s/Str "CWE identifier") ;; CWE identifier for a particular weakness
   })

(s/defschema Configuration
  "See http://stixproject.github.io/data-model/1.2/et/ConfigurationType/"
  {:description (describe s/Str "text description of this Configuration")
   (s/optional-key :short_description)
   (describe s/Str "short text description of this Configuration")
   (s/optional-key :cce_id)
   (describe s/Str "CCE identifier") ;; The CCE identifier for a configuration item
   })

(s/defschema ExploitTarget
  "See http://stixproject.github.io/data-model/1.2/et/ExploitTargetType/"
  (st/merge
   c/GenericStixIdentifiers
   {:valid_time c/ValidTime
    :tlp c/TLP}
   (st/optional-keys
    {:version (describe s/Str "schema version for this content")
     :vulnerability (describe
                     [Vulnerability]
                     (str "identifies and characterizes a Vulnerability"
                          " as a potential Exploit Target"))
     :weakness (describe
                [Weakness]
                (str "identifies and characterizes a Weakness as"
                     " a potential Exploit Target"))
     :configuration (describe
                     [Configuration]
                     (str "identifies and characterizes a Configuration as"
                          " a potential Exploit Target"))
     :potential_COAs (describe
                      rel/RelatedCOAs
                      (str "identifies and characterizes a Configuration as"
                           " a potential Exploit Target"))
     :source (describe s/Str "ExploitTarget source")
     :related_exploit_targets (describe
                               rel/RelatedExploitTargets
                               (str "identifies and characterizes a Configuration"
                                    " as a potential Exploit Target"))
     ;; Not provided: related_packages (deprecated)
     ;; Not provided: handling
     })))

(s/defschema Type
  (s/enum "exploit-target"))

(s/defschema NewExploitTarget
  "Schema for submitting ExploitTargets"
  (st/merge
   (st/dissoc ExploitTarget :version)
   (st/optional-keys
    {:id c/ID
     :valid_time c/ValidTime
     :type Type
     :tlp c/TLP})))

(s/defschema StoredExploitTarget
  "An exploit-target as stored in the data store"
  (c/stored-schema "exploit-target" ExploitTarget))

