(ns ctim.schemas.vulnerability
  (:require #?(:clj  [flanders.core :as f :refer [def-entity-type def-map-type def-eq]]
               :cljs [flanders.core :as f :refer-macros [def-entity-type def-map-type def-eq]])
            #?(:clj [ctim.lib.generators :as gen])
            [ctim.schemas.common :as c]
            [ctim.schemas.vocabularies :as v]))

(def type-identifier "vulnerability")

(def-eq VulnerabilityTypeIdentifier type-identifier
  :description "The fixed value \"vulnerability\"")

(def vulnerability-desc
  (str "a mistake in software that can be directly "
       "used by a hacker to gain access to a system or network"))

(def vulnerability-desc-link
  "[Vulnerability](http://docs.oasis-open.org/cti/stix/v2.0/cs01/part2-stix-objects/stix-v2.0-cs01-part2-stix-objects.html#_Toc496714334)")

(defn valid-score?
  "check that a score is above 0 and below 10"
  [score]
  (< 0 score 10))

(def Score
  (f/num
   :description "a Score number from 0 to 10"
   :spec valid-score?
   :float? true
   #?(:clj :gen)
   #?(:clj gen/score)))

(def vector-string-exp
  (re-pattern "^CVSS:3.0/((AV:[NALP]|AC:[LH]|PR:[UNLH]|UI:[NR]|S:[UC]|[CIA]:[NLH]|E:[XUPFH]|RL:[XOTWU]|RC:[XURC]|[CIA]R:[XLMH]|MAV:[XNALP]|MAC:[XLH]|MPR:[XUNLH]|MUI:[XNR]|MS:[XUC]|M[CIA]:[XNLH])/)*(AV:[NALP]|AC:[LH]|PR:[UNLH]|UI:[NR]|S:[UC]|[CIA]:[NLH]|E:[XUPFH]|RL:[XOTWU]|RC:[XURC]|[CIA]R:[XLMH]|MAV:[XNALP]|MAC:[XLH]|MPR:[XUNLH]|MUI:[XNR]|MS:[XUC]|M[CIA]:[XNLH])$"))

(defn vector-string?
  "validate a vector string using a regexp"
  [s]
  (re-matches vector-string-exp s))

(def VectorString
  (f/str
   :description
   (str "a text representation of a set of CVSS metrics."
        "It is commonly used to record or transfer "
        "CVSS metric information in a concise form")
   :spec vector-string?
   #?(:clj :gen)
   #?(:clj gen/vector-string)))

(def-map-type CVSSV3
  (concat
   (f/required-entries
    (f/entry :vector_string VectorString)
    (f/entry :base_score Score)
    (f/entry :base_severity v/VulnerabilitySeverity))
   (f/optional-entries
    (f/entry :attack_vector v/VulnerabilityAttackVector
             :description (str "Reflects the context by which "
                               "vulnerability exploitation is possible"))
    (f/entry :modified_attack_vector v/VulnerabilityModifiedAttackVector
             :description "modified attack vector")

    (f/entry :attack_complexity v/VulnerabilityAttackComplexity
             :description (str "describes the conditions beyond the attacker's "
                               "control that must exist in order to exploit the "
                               "vulnerability"))
    (f/entry :modified_attack_complexity v/VulnerabilityModifiedAttackComplexity
             :description "modified attack complexity")

    (f/entry :privileges_required v/VulnerabilityPrivilegesRequired
             :description (str "describes the level of privileges an attacker "
                               "must possess before successfully exploiting the vulnerability"))
    (f/entry :modified_privileges_required v/VulnerabilityModifiedPrivilegesRequired
             :description "modified privileges required")

    (f/entry :user_interaction v/VulnerabilityUserInteraction
             :description (str "captures the requirement for a user, "
                               "other than the attacker, to participate in "
                               "the successful compromise of the vulnerable component"))
    (f/entry :modified_user_interaction v/VulnerabilityModifiedUserInteraction
             :description "modified user interaction")

    (f/entry :scope v/VulnerabilityScope
             :description (str "the ability for a vulnerability in one "
                               "software component to impact resources beyond "
                               "its means, or privileges"))
    (f/entry :modified_scope v/VulnerabilityModifiedScope
             :description "modified scope")

    (f/entry :confidentiality_impact v/VulnerabilityConfidentialityImpact
             :description (str "measures the impact to the confidentiality of"
                               "the information resources managed by a software "
                               "component due to a successfully exploited vulnerability"))

    (f/entry :modified_confidentiality_impact v/VulnerabilityModifiedConfidentialityImpact
             :description "modified confidentiality impact")

    (f/entry :integrity_impact v/VulnerabilityIntegrityImpact
             :description (str "measures the impact to integrity "
                               "of a successfully exploited vulnerability"))
    (f/entry :modified_integrity_impact v/VulnerabilityModifiedIntegrityImpact
             :description (str "modified integrity impact"))

    (f/entry :availability_impact v/VulnerabilityAvailabilityImpact
             :description (str "measures the impact to the availability "
                               "of the impacted component resulting from a successfully"
                               "exploited vulnerability"))
    (f/entry :modified_availability_impact v/VulnerabilityModifiedAvailabilityImpact
             :description "modified availability impact")

    (f/entry :exploit_code_maturity v/VulnerabilityExploitCodeMaturity
             :description "measures the likelihood of the vulnerability being attacked")

    (f/entry :remediation_level v/VulnerabilityRemediationLevel
             :description (str "Remediation Level of a vulnerability "
                               "is an important factor for prioritization"))
    (f/entry :report_confidence v/VulnerabilityReportConfidence
             :description (str "measures the degree of confidence in the existence "
                               "of the vulnerability and the credibility of the known technical details"))
    (f/entry :temporal_score Score
             :description (str "Round up(VulnerabilityBaseScore × "
                               "VulnerabilityExploitCodeMaturity × "
                               "VulnerabilityRemediationLevel × "
                               "VulnerabilityReportConfidence)"))
    (f/entry :temporal_severity Score
             :description "temporal severity")

    ;; requirements
    (f/entry :availability_requirement v/VulnerabilitySecurityRequirements)
    (f/entry :confidentiality_requirement v/VulnerabilitySecurityRequirements)
    (f/entry :integrity_requirement v/VulnerabilitySecurityRequirements)

    ;; optional scores
    (f/entry :environmental_score Score)
    (f/entry :environmental_severity v/VulnerabilitySeverity))))

(def-map-type VulnerabilityImpact
  (f/optional-entries
   (f/entry :cvss_v3 CVSSV3)
   (f/entry :impact_score Score)
   (f/entry :exploitability_score Score)))

;; CVE MEtadata (ID Assigner), description, references, v3 impact data published_date, lastmodified_date

(def-map-type CVEDataMeta
  (f/optional-entries
   (f/entry :id c/ShortString)
   (f/entry :assigner c/ShortString)))

(def-map-type CVE
  (f/required-entries
   (f/entry :cve_data_meta CVEDataMeta)))

(def-entity-type Vulnerability
  {:description vulnerability-desc
   :reference vulnerability-desc-link}
  c/base-entity-entries
  c/describable-entity-entries
  c/sourcable-object-entries
  (f/required-entries
   (f/entry :description c/Markdown
            :description
            (str "A description that provides more details and context about the Vulnerability,"
                 "potentially including its purpose and its key characteristics."))
   (f/entry :type VulnerabilityTypeIdentifier
            :description (str "The fixed value " type-identifier)))
  (f/optional-entries
   (f/entry :cve CVE)
   (f/entry :impact VulnerabilityImpact)
   (f/entry :published_date c/Time)
   (f/entry :last_modified_date c/Time)))

(def-entity-type NewVulnerability
  "For submitting a new vulnerability"
  (:entries Vulnerability)
  c/base-new-entity-entries
  (f/optional-entries
   (f/entry :type VulnerabilityTypeIdentifier)))

(def-entity-type StoredVulnerability
  "A vulnerability as stored in the data store"
  (:entries Vulnerability)
  c/base-stored-entity-entries)

(def VulnerabilityRef
  (c/ref-for-type type-identifier))
