(ns cisco.tools.namespace.parallel-refresh
  (:refer-clojure :exclude [time])
  (:require
   [clojure.java.classpath :as classpath]
   [clojure.java.io :as io]
   [clojure.repl]
   [clojure.spec.alpha :as spec]
   [clojure.string :as string]
   [clojure.tools.namespace.dependency]
   [clojure.tools.namespace.dir :as dir]
   [clojure.tools.namespace.find :as find]
   [clojure.tools.namespace.parse :as parse]
   [clojure.tools.namespace.reload :as reload]
   [clojure.tools.namespace.repl :as repl]
   [clojure.tools.namespace.track :as track]
   [clojure.tools.reader.reader-types :refer [indexing-push-back-reader push-back-reader]]
   [loom.alg]
   [loom.graph]
   [nedap.speced.def :as speced]
   [nedap.utils.collections.eager :refer [partitioning-pmap]])
  (:import
   (java.io File StringWriter)
   (java.util.concurrent LinkedBlockingQueue ThreadPoolExecutor TimeUnit)
   (java.util.concurrent.locks ReentrantLock)
   (java.util.jar JarFile)))

;; XXX I think tool.n has a bug where it uses topo-sort, only to ruin it. investigate
;; while we're here, impl some ideas from https://github.com/reducecombine/rehardened
;; XXX rename to rehardened.tools.namespace (ns) + tools.namespace.rehardened (proj/repo)

;; XXX an idea that has worked get is caching file reads per .lastModified timestamp.
;; check if it can be applied in more places

;; XXX very occasionally, when one promise is delivered an error (e.g. 'unknown-ns),
;; another other might never be delivered anything, causing a hang.
;; possible cause: there's a dependency tree between promises

;; XXX after a failed refresh, a second (refresh) will return :ok.
;; example bad scenario:
;; #:clojure.tools.namespace.reload{:error-ns iroh-int.services.module-type-patch.web-service-test}
;; iroh-int.services.module-type-patch.web-service-test> (select-keys clojure.tools.namespace.repl/refresh-tracker [:clojure.tools.namespace.track/load ])
;; #:clojure.tools.namespace.track{:load ()}
;; i.e., load is empty, error-ns not.
;; so, either populate laod from each (parallel) error, or load error-ns's in addition to changed files
;; (check how t.n upstream does it)

(defn islands [graph]
  (loom.alg/connected-components (loom.graph/digraph graph)))

(speced/defn read-ns-decl
  "Reads ns declaration in file with line/column metadata"
  [^string? filename]
  (when-not (-> filename File. .isDirectory)
    (try
      (with-open [reader (-> filename io/reader push-back-reader indexing-push-back-reader)]
        (parse/read-ns-decl reader))
      (catch Exception e
        (if (-> e ex-data :type #{:reader-exception})
          nil
          (throw e))))))

(spec/def ::ns-form (spec/and sequential?
                              (comp #{'ns `ns} first)))

(speced/defn ^::resource-path ns-decl->resource-path [^::ns-form ns-decl, extension]
  (-> ns-decl
      second
      str
      munge
      (string/replace "." "/")
      (str extension)))

(spec/def ::resource-path (spec/and string?
                                    (complement string/blank?)
                                    (complement #{\. \! \? \-})
                                    (fn [x]
                                      (re-find #"\.clj([cs])?$" x))))

(speced/defn resource-path->filenames [^::resource-path resource-path]
  (->> (-> (Thread/currentThread)
           (.getContextClassLoader)
           (.getResources resource-path))
       (enumeration-seq)
       (distinct) ;; just in case
       (mapv str)))

(def jar-filename->ns-decls (memoize (fn [platform x]
                                       (when (classpath/jar-file? x)
                                         (-> x io/file JarFile. (find/find-ns-decls-in-jarfile platform))))))

(defmacro ignore-reader-exception
  {:style/indent 0}
  [& body]
  `(try
     ~@body
     (catch Exception e#
       (if (= :reader-exception (:type (ex-data e#)))
         nil
         (throw e#)))))

(def read-file-ns-decl (memoize (fn [^File f _timestamp platform]
                                  ;; _timestamp is simply there for memoize
                                  (ignore-reader-exception
                                   (clojure.tools.namespace.file/read-file-ns-decl f platform)))))

(defn find-ns-decls-in-dir
  "Like its `#'find/find-ns-decls-in-dir` counterpart but adds caching based on a given file's timestamp."
  ([dir]
   (find-ns-decls-in-dir dir nil))

  ([dir platform]
   (keep (fn [^File f]
           (read-file-ns-decl f
                              (-> f .lastModified)
                              (:read-opts platform)))
         (find/find-sources-in-dir dir platform))))

(defn find-ns-decls*
  "Like `#'find-ns-decls` but adds parallelism (for dirs) and caching (for jars)"
  ([files]
   (find-ns-decls* files nil))

  ([files platform]
   (concat
    (->> files
         (partitioning-pmap (fn [^File f]
                              (when (and (-> f .isDirectory)
                                         (not (->> f str (re-find #"target.*classes"))))
                                (find-ns-decls-in-dir f platform))))
         (filter identity)
         (apply concat))
    (->> files
         (keep (partial jar-filename->ns-decls platform))
         (apply concat)))))

(speced/defn ^{::speced/spec (spec/coll-of ::ns-form
                                           :min-count 1)}
  find-ns-decls []
  (->> (classpath/system-classpath)
       (find-ns-decls*)
       (filter identity)))

(spec/def ::classpath-ns-forms (spec/coll-of ::ns-form
                                             :kind set?
                                             :min-count 1))

;; NOTE: this used to be limited to the `refresh-dirs`, but that proved to be unsafe
;; e.g. many tests namespaces deemed as leaves would depend on e.g. matcher-combinators, which would then be
;; concurrently `require`d.
(speced/defn ^::classpath-ns-forms classpath-ns-forms
  "Returns all the ns-forms contained or required in the entire classpath."
  []
  (->> (find-ns-decls)
       (set)))

(defn files-and-deps [ns-forms read-opts]
  (->> ns-forms
       (reduce (speced/fn [m ^::ns-form decl]
                 (let [deps (parse/deps-from-ns-decl decl)
                       name (parse/name-from-ns-decl decl)]
                   (-> m
                       (assoc-in [:depmap name] (into (empty deps)
                                                      ;; remove self-requires (occasionally found in the wild)
                                                      (remove #{name}
                                                              deps))))))
               {})))

;; the original impl was: (#'clojure.tools.namespace.track/update-deps (clojure.tools.namespace.dependency/graph) depmap)
;; this one performs in 50m for a large project, whereas the old one was 1500ms
(defn classpath-graph [^::classpath-ns-forms base]
  (let [{:keys [depmap]} (files-and-deps base nil)
        dependents (atom {})]
    (->> depmap
         (run! (fn [[k v]]
                 (doseq [d v]
                   (swap! dependents update d (fn [v]
                                                (if-not v
                                                  #{k}
                                                  (conj v k))))))))
    {:dependencies (->> depmap
                        (keep (fn [[k v]]
                                (when (seq v)
                                  [k v])))
                        (into {}))
     :dependents @dependents}))

(def print-lock (Object.))

(defn safe-println
  "`println` can always fail, given that `*out*` can be bound to an arbitrary object.

  Failing during `println` cannot happen as it would cause a hang."
  [& xs]
  (locking print-lock
    (try
      (apply println xs)
      (catch Throwable _))))

(defn word-wrap [s]
  (->> (-> s
           (string/split #"\s"))
       (reduce (fn [lines word]
                 (let [line (last lines)
                       l (->> (conj line word) ^String (apply str) .length)]
                   (if (and (<= l 132)
                            (-> lines count pos?))
                     (update lines
                             (dec (count lines))
                             conj " " word)
                     (conj lines [word]))))
               [])
       (map (partial apply str))
       (string/join "\n")))

(defn safe-pst
  "`.printStackTrace` can always fail, given that `System/out` can be set to an arbitrary object.

  Failing during `.printStackTrace` cannot happen as it would cause a hang.

  Also uses a repl-friendly output format: it prints only the root cause (without a further stacktrace),
  removing some redundant wording and word-wrapping at 132 columns."
  [^Throwable e]
  (locking print-lock
    (try
      (let [sw (StringWriter.)
            v (binding [*out* sw
                        *err* sw]
                (clojure.repl/pst e 1)
                (->> (-> sw
                         str
                         (string/replace "Note: The following stack trace applies to the reader or compiler, your code was not executed.\n" "")
                         (string/replace #"^CompilerException.*at\s\(" "")
                         (string/split-lines))
                     (map (fn [s]
                            (if-not (and (re-find #"#:clojure.error" s)
                                         (re-find #":\d+:\d+" s))
                              s
                              (str "Error at "
                                   (-> s
                                       (string/replace #"\)\.\s#:clojure.error.*" ""))))))
                     (map word-wrap)
                     (string/join "\n")))]
        (binding [*out* *err*]
          (println v)))
      (catch Throwable _))))

(defn debug [& xs]
  (when (#{"true" "1" "yes" "y" "t" "debug"}
         (some-> "cisco.tools.namespace.parallel-refresh.debug"
                 System/getProperty
                 string/lower-case))
    (safe-println xs)))

;; XXX simplify
(defn positions [pred coll]
  (keep-indexed (fn [idx x]
                  (when (pred x)
                    idx))
                coll))

(defn elem-index [item xs]
  (->> xs (positions #{item}) first))

(defn sort-list-by-list [crit xs]
  (->> xs
       (sort (fn [a b]
               (< (or (elem-index a crit) Long/MAX_VALUE)
                  (or (elem-index b crit) Long/MAX_VALUE))))))

(defn executor []
  (let [c (-> (Runtime/getRuntime) .availableProcessors)]
    ;; important - given its a fixed size pool, the queue must be unbounded, so that no work gets rejected
    (ThreadPoolExecutor. c c 60 TimeUnit/SECONDS (LinkedBlockingQueue.))))

(def ^:dynamic ^ThreadPoolExecutor *executor* nil)

(spec/def ::error-result
  (spec/tuple simple-symbol? (partial instance? Throwable)))

(spec/def ::result
  (spec/or ::ok           #{::ok}
           ::error-result ::error-result))

(defn in-bg [promises-atom f]
  {:pre [*executor*]}
  (let [p (promise)]
    (swap! promises-atom conj p)
    (send-via *executor* (agent nil) (fn [_]
                                       (try
                                         (speced/let [^::result v (f)]
                                           (deliver p v))
                                         (catch Throwable e
                                           ;; XXX detect this
                                           ;; sometimes 'unknown-ns is attempted to be loaded. Shouldn't ever happen
                                           (deliver p ['unknown-ns e])))))))

(defmacro time
  "Modified to use `#'safe-println`."
  [expr]
  `(let [start# (. System (nanoTime))
         ret# ~expr]
     (safe-println (str "Elapsed time: "
                        (-> System
                            (. (nanoTime))
                            (- start#)
                            double
                            (/ 1000000.0))
                        " msecs"))
     ret#))

(spec/def ::workload (spec/coll-of symbol?))

(speced/defn process-leaf [idx, ^simple-symbol? leave, promises, leaves-promises]
  (let [leave-promise (get leaves-promises idx)]
    (assert leave-promise)
    (swap! promises conj leave-promise)
    (in-bg promises (fn []
                      (debug "Reloading leaf namespace in"
                             (-> (Thread/currentThread)
                                 .getName)
                             "-"
                             leave)
                      (deliver leave-promise
                               (try
                                 (require :reload leave)
                                 ::ok
                                 (catch Throwable e
                                   (safe-pst e)
                                   [leave e])))
                      ::ok))))

(spec/def ::project-namespaces (spec/coll-of simple-symbol? :kind set?))

(speced/defn ^::project-namespaces project-namespaces
  "Returns all the namespaces contained or required in the current project.

  Excludes third-party dependencies."
  []
  (->> (#'dir/find-files repl/refresh-dirs find/clj)
       (partitioning-pmap (speced/fn [^File file]
                            (let [decl (-> file str read-ns-decl)
                                  n (some-> decl parse/name-from-ns-decl)]
                              [n])))
       (apply concat)
       (distinct)
       (filter identity)
       (set)))

(speced/defn process-island [^deref promises
                             ^set? workload-set
                             ^::project-namespaces the-project-namespaces
                             ^set? leaves
                             ^::workload workload
                             island]
  (in-bg promises (fn []
                    (let [corpus (some->> island
                                          (filter workload-set)
                                          (seq) ;; shortcircuit a later costlier filtering
                                          (filter the-project-namespaces)
                                          (seq)
                                          (group-by (partial contains? leaves)))]
                      (if-not corpus
                        ::ok
                        (let [{island-nonleaves false, island-leaves true} corpus]
                          (when (seq island-nonleaves)
                            (debug "Reloading"
                                   (count island-nonleaves)
                                   "non-leaf namespaces in"
                                   (-> (Thread/currentThread) .getName (str ":\n  "))
                                   (str "["
                                        (->> island-nonleaves
                                             (string/join "\n    "))
                                        "]")))
                          (speced/let [will-req (some->> island-nonleaves
                                                         (sort-list-by-list workload)
                                                         (seq))
                                       ^::result succ (try
                                                        (some->> will-req
                                                                 (apply require :reload))
                                                        ::ok
                                                        (catch Throwable e
                                                          (safe-pst e)
                                                          [(first will-req) e]))]
                            (if-not (#{::ok} succ)
                              succ
                              (let [leaves-promises (mapv (fn [_]
                                                            (promise))
                                                          island-leaves)]
                                (doseq [p leaves-promises]
                                  (swap! promises conj p))
                                (->> island-leaves
                                     (sort-list-by-list workload)
                                     (map-indexed (fn [i l]
                                                    (process-leaf i
                                                                  l
                                                                  promises
                                                                  leaves-promises)))
                                     (doall))
                                succ)))))))))

(def dbg-executor
  "The last `java.util.concurrent.Executor` that is running / has run the latest workload."
  nil)

(def dbg-promises
  "An atom of `promises` representing the latest workfload. If some promise is still :pending,
  that means the work is still in progress."
  nil)

(speced/defn perform-refresh [^::workload workload]
  (let [the-classpath-ns-forms (classpath-ns-forms)
        the-classpath-ns-names (->> the-classpath-ns-forms
                                    (map parse/name-from-ns-decl)
                                    set)
        the-project-namespaces (project-namespaces)
        workload-set (set workload)
        {:keys [dependencies dependents]} (classpath-graph the-classpath-ns-forms)
        leaves (->> dependencies
                    (keys)
                    (remove (partial find dependents))
                    (set))
        leaves-3rd-party-dependencies (->> leaves
                                           (filter the-project-namespaces)
                                           (mapcat (speced/fn [^simple-symbol? n]
                                                     (speced/let [^::speced/nilable ^set? d (dependencies n)]
                                                       (if-not d
                                                         []
                                                         (remove the-project-namespaces d)))))
                                           (set)
                                           ;; filter out e.g. leiningen.* namespaces
                                           (filter the-classpath-ns-names)
                                           (seq))
        promises (atom [])]

    ;; These need to be `require`d before anything else, and sequentially, for safety.
    ;; There's no associated `println` because normally these will be actually loaded just once
    ;; during the entire JVM lifecycle:
    (some->> leaves-3rd-party-dependencies (apply require))

    (binding [*executor* (executor)]
      (alter-var-root #'dbg-executor (constantly *executor*))
      (->> dependencies
           (islands)
           (filterv (partial some workload-set))
           (run! (partial process-island
                          promises
                          workload-set
                          the-project-namespaces
                          leaves
                          workload)))
      (Thread/sleep 150) ;; ensure there are promises
      (alter-var-root #'dbg-promises (constantly promises))
      (while (or (-> *executor* .getActiveCount pos?)
                 (some (fn [p]
                         (= ::timeout
                            (deref p 50 ::timeout)))
                       @promises))
        (Thread/sleep 50))
      (-> *executor* .shutdown)
      (->> @promises
           (map deref)
           (remove #{::ok})
           first))))

;; XXX doesn't appear to affectively assoc an error. (refresh) after an error will say :ok
;; XXX move to different ns
(defn track-reload-one
  [tracker]
  (let [{unload ::track/unload, load ::track/load} tracker]
    (cond
      (seq unload)
      (let [n (first unload)]
        (reload/remove-lib n)
        (update-in tracker [::track/unload] rest))

      (seq load)
      (try
        (speced/let [^::speced/nilable ^::error-result v (perform-refresh load)
                     [maybe-failed-ns ex] v]
          (cond-> tracker
            true            (update-in [::track/load] (constantly ()))
            maybe-failed-ns (assoc ::reload/error-ns maybe-failed-ns)
            ex              (assoc ::reload/error ex)))
        (catch Throwable t
          (assoc tracker
                 ::reload/error    t
                 ::reload/error-ns (first load)
                 ::track/unload    load)))

      :else
      tracker)))

(defn track-reload
  [tracker]
  (loop [tracker (dissoc tracker ::reload/error ::reload/error-ns)]
    (let [{error ::reload/error, unload ::track/unload, load ::track/load} tracker]
      (if (and (not error)
               (or (seq load) (seq unload)))
        (recur (track-reload-one tracker))
        tracker))))

(defn referred
  "Plays better with vars coming from Potemkin.

  For an explanation see https://github.com/jonase/eastwood/issues/307#issuecomment-764304379"
  [ns]
  (reduce (fn [m [sym var-ref]]
            (let [ns-sym (-> var-ref symbol namespace symbol)
                  var-sym (-> var-ref symbol name symbol)]
              (assoc-in m [ns-sym var-sym] sym)))
          {}
          (ns-refers ns)))

(defn do-refresh [scan-opts after-sym]
  (when after-sym
    (assert (symbol? after-sym) ":after value must be a symbol")
    (assert (namespace after-sym)
            ":after value must be a namespace-qualified symbol"))
  (let [current-ns-name (ns-name *ns*)
        current-ns-refers (referred *ns*)
        current-ns-aliases (#'repl/aliased *ns*)]
    (alter-var-root #'repl/refresh-tracker dir/scan-dirs repl/refresh-dirs scan-opts)
    (alter-var-root #'repl/refresh-tracker #'repl/remove-disabled)
    ;; avoid a deadlock by using two steps:
    (let [v (track-reload repl/refresh-tracker)]
      (alter-var-root #'repl/refresh-tracker (constantly v)))
    (in-ns current-ns-name)
    (let [result (#'repl/print-and-return repl/refresh-tracker)]
      (if (= :ok result)
        (if after-sym
          (if-let [after (ns-resolve *ns* after-sym)]
            (after)
            (throw (Exception.
                    (str "Cannot resolve :after symbol " after-sym))))
          result)
        ;; There was an error, recover as much as we can:
        (do (when-not (or (false? (::repl/unload (meta *ns*)))
                          (false? (::repl/load (meta *ns*))))
              (#'repl/recover-ns current-ns-refers current-ns-aliases))
            (symbol ""))))))

(def ^ReentrantLock refresh-lock
  "Prevents concurrent invocations from e.g. an IDE + a terminal."
  (ReentrantLock.))

(defn refresh [& options]
  (try
    (if-not (-> refresh-lock (.tryLock 60 TimeUnit/SECONDS))
      (safe-println "Couldn't refresh because some other thread is already performing one.")
      (let [{:keys [after]} options]
        (do-refresh {:platform find/clj} after)))
    (finally
      (try
        (-> refresh-lock .unlock)
        (catch IllegalMonitorStateException _)))))

(defn gc-later
  "An optional helper, not directly used by this lib.

  You can set it as an `:after` option, which can improve performance in some cases."
  []
  (future
    (System/gc)))

;; Example: does clj-http.core depend on potemkin?
;; for that example we might have to observe that clj-http.core depends on clj-http.headers which depends on potemkin.
(defn transitively-depends-on? [{:keys [dependencies dependents] :as pg} x & {:keys [on
                                                                                     visited-atom]
                                                                              :or {visited-atom (atom {})}}]
  (let [cached (get @visited-atom [x on])
        r (if-some [v cached]
            v
            (or (boolean (re-find on (str x)))
                (boolean (some->> (some-> dependencies (get x))
                                  (some (fn [dependency]
                                          (or (re-find on (str dependency))
                                              (transitively-depends-on? pg
                                                                        dependency
                                                                        :on on
                                                                        :visited-atom visited-atom))))))))]
    (when-not cached
      (swap! visited-atom assoc [x on] r))
    r))

(comment
  (loop [l (clojure.lang.RT/baseLoader)
         i 1]
    (if-let [n (try
                 (-> l bean :parent)
                 (catch Throwable _))]
      (if (identical? n l)
        i
        (recur n (inc i)))
      i)))

(let [l (-> (clojure.lang.RT/baseLoader) bean :parent)]
  (= l (clojure.lang.RT/baseLoader)))

(defn compile-3rd-party-deps!
  "`require`s files that are detected as 3rd-party project dependencies:
  namely, namespaces that aren't contained in `refresh-dirs` but are referenced from them.

  This `require`ing is performed while `binding [*compile-files* true]`, i.e. it AOT-compiles those files.

  This way, subsequent JVM startups will be faster. (See: https://clojure.org/guides/dev_startup_time )

  Namespaces that appear to be AOT-unfriendly will be skipped fro AOT-compilation, on a best-effort basis."
  []
  {:pre [(seq repl/refresh-dirs)]}
  (-> *compile-path* File. .mkdirs)
  (let [the-classpath-ns-forms (classpath-ns-forms)
        the-classpath-ns-names (->> the-classpath-ns-forms
                                    (map parse/name-from-ns-decl)
                                    set)
        the-project-namespaces (project-namespaces)
        {:keys [dependencies dependents] :as pg} (classpath-graph the-classpath-ns-forms)
        corpus (->> the-project-namespaces
                    (mapcat (speced/fn [^simple-symbol? n]
                              (speced/let [^::speced/nilable ^set? d (dependencies n)]
                                (if-not d
                                  []
                                  (remove the-project-namespaces d)))))
                    (set)
                    ;; filter out e.g. leiningen.* namespaces
                    (filter the-classpath-ns-names)
                    (remove (fn [ns-sym]
                              (transitively-depends-on? pg ns-sym :on #"potemkin|^schema|trapperkeeper|clojure.tools.logging"))))]
    (binding [*compile-files* true]
      (some->> corpus
               seq
               (apply require)))))
