(ns cisco.tools.namespace.impl.dir
  "Fixes the usage of .lastModified to correctly handle Git branch changes
  (which often don't update .lastModified)"
  (:require
   [clojure.java.classpath :refer [classpath-directories]]
   [clojure.java.io :as io]
   [clojure.set :as set]
   [clojure.string :as string]
   [clojure.tools.namespace.dir :as dir]
   [clojure.tools.namespace.file :as file]
   [clojure.tools.namespace.find :as find]
   [clojure.tools.namespace.track :as track]
   [cisco.tools.namespace.impl.key-calculation :refer [sha256]])
  (:import
   (java.io File)
   (java.util.regex Pattern)))

(defn find-files [dirs platform]
  (->> dirs
       (map io/file)
       (map #(.getCanonicalFile ^File %))
       (filter #(.exists ^File %))
       (mapcat #(find/find-sources-in-dir % platform))
       (map #(.getCanonicalFile ^File %))))

(defn- modified-files [{:keys [::sha-mapping]
                        :as   tracker} files]
  (->> files
       (filter (fn [^File file]
                 (and (< (::dir/time tracker 0)
                         (-> file .lastModified))
                      (let [found (get sha-mapping file)]
                        (if-not found
                          true
                          (not= found
                                (-> file slurp sha256)))))))))

(defn- deleted-files [tracker files]
  (set/difference (::dir/files tracker #{}) (set files)))

(defn- update-files [tracker deleted modified {:keys [read-opts]}]
  (let [now (System/currentTimeMillis)
        {:keys [::dir/files]
         :as   new-tracker} (-> tracker
                                (update-in [::dir/files] #(if % (apply disj % deleted) #{}))
                                (file/remove-files deleted)
                                (update-in [::dir/files] into modified)
                                (file/add-files modified read-opts)
                                (assoc ::dir/time now))]
    (assoc new-tracker
           ::sha-mapping (into {}
                               (map (fn [^File f]
                                      [f (-> f slurp sha256)]))
                               files))))

(defn scan-files
  "Scans files to find those which have changed since the last time
  'scan-files' was run; updates the dependency tracker with
  new/changed/deleted files.

  files is the collection of files to scan.

  Optional third argument is map of options:

    :platform  Either clj (default) or cljs, both defined in
               clojure.tools.namespace.find, controls reader options for
               parsing files.

    :add-all?  If true, assumes all extant files are modified regardless
               of filesystem timestamps."
  {:added "0.3.0"}
  ([tracker files] (scan-files tracker files nil))
  ([tracker files {:keys [platform add-all?]}]
   (let [deleted (seq (deleted-files tracker files))
         modified (if add-all?
                    files
                    (seq (modified-files tracker files)))]
     (if (or deleted modified)
       (update-files tracker deleted modified platform)
       tracker))))

(defn scan-dirs
  "Scans directories for files which have changed since the last time
  'scan-dirs' or 'scan-files' was run; updates the dependency tracker
  with new/changed/deleted files.

  dirs is the collection of directories to scan, defaults to all
  directories on Clojure's classpath.

  Optional third argument is map of options:

    :platform  Either clj (default) or cljs, both defined in
               clojure.tools.namespace.find, controls file extensions
               and reader options.

    :add-all?  If true, assumes all extant files are modified regardless
               of filesystem timestamps."
  {:added "0.3.0"}
  ([tracker] (scan-dirs tracker nil nil))
  ([tracker dirs] (scan-dirs tracker dirs nil))
  ([tracker dirs {:keys [platform add-all?] :as options}]
   (let [ds (or (seq dirs) (classpath-directories))]
     (scan-files tracker (find-files ds platform) options))))
