(ns cisco.tools.namespace.impl.key-calculation
  (:refer-clojure :exclude [time])
  (:require
   [clojure.java.classpath :as classpath]
   [clojure.java.io :as io]
   [clojure.repl]
   [clojure.spec.alpha :as spec]
   [clojure.string :as string]
   [clojure.tools.namespace.dependency]
   [clojure.tools.namespace.dir :as dir]
   [clojure.tools.namespace.find :as find]
   [clojure.tools.namespace.parse :as parse]
   [clojure.tools.namespace.reload :as reload]
   [clojure.tools.namespace.repl :as repl]
   [clojure.tools.namespace.track :as track]
   [clojure.tools.reader.reader-types :refer [indexing-push-back-reader push-back-reader]]
   [loom.alg]
   [loom.graph]
   [nedap.speced.def :as speced]
   [nedap.utils.collections.eager :refer [partitioning-pmap]])
  (:import
   (java.io File StringWriter)
   (java.security MessageDigest)
   (java.util Base64)
   (java.util.concurrent LinkedBlockingQueue ThreadPoolExecutor TimeUnit)
   (java.util.concurrent.locks ReentrantLock)
   (java.util.jar JarFile)))

(speced/defn sha256 [^String s]
  (let [input (-> s (.getBytes "UTF-8"))
        output (-> (MessageDigest/getInstance "SHA-256")
                   (doto (.update input))
                   .digest)]
    (-> (Base64/getEncoder) (.encodeToString output))))

(def key-of--mapping (atom {}))

;; test should be:
;; 1st run - string, non SHA-based (this is the important optimization; else we'd simply run SHA for all cases)
;; 2nd run, no changes - different string, SHA-based
;; 3rd run, no changes - same string
;; 4th run, changes - different string, SHA-based
(speced/defn key-of
  "Calculates a cache key which, best-case scenario, favors timestamps over SHA calculation,
as a performance optimization."
  [^File f]
  (let [filename (-> f .getAbsolutePath)
        lM (-> f .lastModified)]

    (cond
      (not (contains? @key-of--mapping filename))
      (do
        (swap! key-of--mapping assoc filename #{lM})
        (str lM))

      (-> @key-of--mapping (get filename) (contains? lM))
      (-> f slurp sha256)

      true
      (do
        (swap! key-of--mapping update filename conj lM)
        (-> f slurp sha256)))))
