(ns tiesql.impl.sql-bind-impl
  (:require [schema.core :as s]
            [clj-common :as cc]
            [tiesql.common :refer :all]
            [tiesql.compiler.schema :as sch]))


(defn validate-input-not-empty!
  [m]
  (if (and (not-empty (rest (sql-key m)))
           (empty? (input-key m)))
    (cc/fail (format "Input is missing for %s " (name-key m)))
    m))


(defn validate-input-type!
  [m]
  (let [dml-type (dml-type-key m)
        input (input-key m)
        sql (sql-key m)]
    (if (and (not= dml-type dml-type-insert-key)
             (not-empty (rest sql))
             (not (map? input)))
      (cc/fail (format "Input Params for %s will be map format but %s is not map format " sql input))
      m)))


(defn- validate-required-params*!
  [p-set input]
  (let [p-set (into #{} p-set)
        i-set (into #{} (keys input))
        diff-keys (clojure.set/difference p-set i-set)]
    (if-not (empty? diff-keys)
      (cc/fail (format "Missing required parameter %s" (pr-str diff-keys)))
      input)))


(defn validate-required-params!
  [m]
  (let [input (cc/as-sequential (input-key m))
        r (-> (cc/comp-xf-until (map #(validate-required-params*! (rest (sql-key m)) %)))
              (transduce conj input))]
    (if (cc/failed? r)
      r
      m)))


(defn get-vali-type
  [coll id]
  (->> coll
       (filter #(and
                 (= id (first %))
                 (= vali-type (second %))))
       (map #(nth % 2))
       (first)))



(defn get-place-holder
  [type v]
  (if (contains? (supers type)
                 clojure.lang.Sequential)
    (clojure.string/join ", " (repeat (count v) "?"))
    "?"))


(defn update-sql-str
  [v sql-str k]
  (clojure.string/replace-first sql-str (re-pattern (str k)) v))


(defn default-proc
  [tm]
  (let [[sql-str & sql-params] (sql-key tm)
        input (input-key tm)
        validation (validation-key tm)
        rf (fn [sql-coll p-key]
             (let [p-value (cc/as-sequential (p-key input))
                   w (-> (get-vali-type validation p-key)
                         (get-place-holder p-value)
                         (update-sql-str (first sql-coll) p-key))
                   q-str (assoc sql-coll 0 w)]
               (reduce conj q-str p-value)))]
    (->> (reduce rf [sql-str] sql-params)
         (assoc tm sql-key))))


(defn do-default-proc
  [tm]
  (cc/try-> tm
            validate-input-not-empty!
            validate-input-type!
            validate-required-params!
            default-proc))


(defn insert-proc
  [tm]
  (let [sql (sql-key tm)
        sql-str (reduce (partial update-sql-str "?") sql)]
    (->> (input-key tm)
         (cc/as-sequential)
         (mapv #(cc/select-values %1 (rest sql)))
         (reduce conj [sql-str])
         (assoc tm sql-key))))


(defn do-insert-proc
  [tm]
  (cc/try-> tm
            validate-input-not-empty!
            validate-input-type!
            validate-required-params!
            insert-proc))



(defn child-dml-bind-type
  []
  (let [dd {k-name    dml-type-select-key
            k-process (map do-default-proc)}
        idd {k-name    dml-type-insert-key
             k-process (map do-insert-proc)}]
    {dml-type-select-key dd
     dml-type-call-key   dd
     dml-type-update-key dd
     dml-type-delete-key dd
     dml-type-insert-key idd}))


(defn dml-type
  [v]
  (-> v
      (first)
      (clojure.string/trim)
      (clojure.string/lower-case)
      (clojure.string/split #"\s+")
      (first)
      (keyword)))




(defn sql-str-emission
  [sql-str]
  (->> (re-seq #"\w*:\w+" sql-str)
       (transduce (comp (map read-string)) conj)
       (reduce (fn [acc v]
                 (let [w (cc/as-lower-case-keyword v)
                       sql-str-w (-> (first acc)
                                     (clojure.string/replace-first (re-pattern (cc/as-string v)) (cc/as-string w)))]
                   (-> (assoc-in acc [0] sql-str-w)
                       (conj w)))
                 ) [sql-str])))



(defn sql-emission
  [sql-str & w]
  (let [p (comp (filter not-empty)
                (map sql-str-emission)
                (map (fn [v] {sql-key      v
                              dml-type-key (dml-type v)})))
        sql (clojure.string/split (clojure.string/trim sql-str) #";")]
    (->> (transduce p conj [] sql)
         (mapv (fn [i m]
                 (assoc m index i)
                 ) (range)))))


(defn- not-blank? [^String v]
  (not (clojure.string/blank? v)))


(defn sql-schema
  [n acc]
  (sch/assoc-schema-key
    acc
    (s/required-key n)
    (s/both s/Str (s/pred not-blank? 'not-blank?))))


(defn new-sql-processor
  ([] (new-sql-processor sql-key))
  ([n]
   {k-emission sql-emission
    k-name     n
    k-validate (partial sql-schema n)}))



(defn new-dml-type
  [c order]
  {k-order        order
   k-name         dml-type-key
   k-process-type input-key
   childs         c}
  )


(defn apply-xf-fn [f v]
  (-> (into [] f [v])
      (first)))


(defn new-map-processor
  [m]
  (let [namek (k-name m)
        f1 (fn [tm]
             (let [kv (namek tm)
                   f (get-in m [childs kv k-process])]
               (if f
                 (apply-xf-fn f tm)
                 nil)))]
    (-> m
        (assoc k-process (map f1))

        (dissoc childs))))



(defn comp-child
  [p-context]
  (if-let [v (dml-type-key p-context)]
    (->> (new-map-processor v)
         (assoc p-context dml-type-key))
    p-context))
