(ns clj-common)


(def as-lower-case-keyword (comp keyword clojure.string/lower-case name))


(defn as-sequential
  [input]
  (if (sequential? input)
    input
    [input]))


(defn as-keyword
  [v-str]
  (if (and v-str
           (string? v-str)
           (not (keyword? v-str)))
    (keyword v-str)
    v-str))


(defn as-string
  [v]
  (if (string? v)
    v
    (str v)))


(defn stringify-keys2
  "Recursively transforms all map and first  vector keys from keywords to strings."
  {:added "1.1"}
  [m]
  (let [f (fn [[k v]] (if (keyword? k) [(name k) v] [k v]))
        fv (fn [v] (update-in v [0] (fn [w]
                                      (if-not (vector? w)
                                        w
                                        (mapv #(name %) w)))))]
    ;; only apply to maps and vector
    (clojure.walk/postwalk (fn [x]
                             (cond (map? x)
                                   (into {} (map f x))
                                   (vector? x) (fv x)
                                   :else x)) m)))



(defn select-values
  [m keyseq]
  (if-not (map? m) m (mapv #(%1 m) keyseq)))


(defn last-index
  [seq]
  (.indexOf seq (last seq)))


(defn replace-last-in-vector
  [seq v]
  (let [index (last-index seq)]
    (if (<= 0 index)
      (assoc seq index v)
      seq)))


;(assoc (list 2 3 4) 0 4)
#_(def ^{:private true} unique-id-atom (atom 0))

#_(defn unique-id []
    (swap! unique-id-atom inc)
    @unique-id-atom)


(defn get-meta [o]
  (->> *ns*
       ns-map
       (filter (fn [[_ v]] (and (var? v) (= o (var-get v)))))
       first
       second
       ))


(defrecord Failure [error])

(defn fail [error] (Failure. error))


(defprotocol ComputationFailed
  (failed? [self]))


(extend-protocol ComputationFailed
  Object (failed? [self] false)
  nil (failed? [self] false)
  Failure (failed? [self] true)
  Exception (failed? [self] true))


(defmacro try->
  [expr & forms]
  (let [g (gensym)
        pstep (fn [step] `(if (failed? ~g) ~g (-> ~g ~step)))]
    `(let [~g ~expr
           ~@(interleave (repeat g) (map pstep forms))]
       ~g)))


(defn try!
  [form & v]
  (try
    (apply form v)
    (catch Exception e
      (fail {:function (:name (:meta (get-meta form)))
             :value    v
             :detail   e}))))




(defn merge-with-key-type
  [f & maps]
  (when (some identity maps)
    (let [merge-entry (fn [m e]
                        (let [k (key e) v (val e)]
                          (if (contains? m k)
                            (assoc m k (f k (get m k) v))
                            (assoc m k v))))
          acc-fn (fn [acc v]
                   (reduce merge-entry (or acc {}) (seq v)))]
      (reduce acc-fn maps))))





(defn seql-contains?
  [range coll v]
  (let [v1 (take range v)]
    (some #(= v1
              (take range %)
              ) coll)))


(defn xf-distinct-with [pred?]
  (fn [rf]
    (let [seen (volatile! #{})]
      (fn
        ([] (rf))                                           ;; init arity
        ([result] (rf result))                              ;; completion arity
        ([result input]                                     ;; reduction arity
         (if (pred? @seen input)
           result
           (do (vswap! seen conj input)
               (rf result input))))))))


(defn distinct-with-range
  [range coll]
  (into [] (xf-distinct-with (partial seql-contains? range)) coll))


(defn acc-with-range
  [range acc v]
  (if (seql-contains? range acc v)
    acc
    (conj acc v)))


(defn concat-with-range
  [range old-coll new-coll]
  (->> old-coll
       (reduce #(acc-with-range range %1 %2) new-coll)
       (into [])))


(defn concat-distinct-with-range
  [range old-coll new-coll]
  (concat-with-range
    range
    (distinct-with-range range old-coll)
    (distinct-with-range range new-coll)))



(defn contain-all?
  [coll key]
  (reduce (fn [acc v]
            (if-not (= v key)
              (reduced nil)
              acc)
            ) key coll))



(defn group-by-value
  [k v]
  (let [fv (first v)]
    (cond
      (map? v) (if (get v k)
                 {(get v k) v}
                 nil)
      (map? fv) (group-by k v)
      (vector? fv) (let [index (.indexOf fv k)]
                     (->> (group-by #(get %1 index) (rest v))
                          (reduce (fn [acc [k w]]
                                    (assoc acc k (reduce conj [fv] w))
                                    ) {})))
      :else v)))







#_(defn xf-assoc-key-coll
    [k coll]
    (fn [rf]
      (let [coll2 (volatile! coll)]
        (fn
          ([] (rf))                                         ;; init arity
          ([result] (rf result))                            ;; completion arity
          ([result input]                                   ;; reduction arity
           (let [[f & r] @coll2]
             (vreset! coll2 r)
             (rf result (assoc input k f))
             ))))))


(defn xf-skip-type
  [pred]
  (fn [rf]
    (fn
      ([] (rf))
      ([result] (rf result))
      ([result input]
       (if (pred input)
         (conj result input)
         (rf result input))))))



(defn xf-until
  [pred]
  (fn [rf]
    (fn
      ([] (rf))
      ([result] (rf result))
      ([result input]
       (if (pred input)
         (reduced input)
         (rf result input))))))


(defn comp-xf-until
  [& steps-coll]
  (->> (interleave steps-coll (repeat (xf-until failed?)))
       (cons (xf-until failed?))
       (apply comp)))


#_(defn xf-take-until
    [pred]
    (fn [rf]
      (fn
        ([] (rf))
        ([result] (rf result))
        ([result input]
         (if (pred input)
           (reduced (rf result input))
           (rf result input))))))




;;;; Path finder

(defn empty-path
  []
  [[]])


(defn conj-index
  [data c-path]
  (let [path-value (get-in data c-path)]
    (if (sequential? path-value)
      (->> (count path-value)
           (range 0)
           (mapv #(conj c-path %)))
      [c-path])))


(defn get-path
  ([data name]
   (get-path data (empty-path) name))
  ([data p-path-coll name]
   (for [c-path p-path-coll
         i-path (conj-index data c-path)
         :let [n-path (conj i-path name)]
         w (conj-index data n-path)]
     w)))



(defn get-key-path
  [m & [p]]
  (if p
    (->> (get-in m p)
         (keys)
         (map (fn [v] (conj p v))))
    (->> (keys m)
         (map (fn [v] [v])))))


(defn get-key-path-with-child
  [m ck]
  (let [pp (get-key-path m)
        p (comp
            (map (fn [w] (conj w ck)))
            (filter #(get-in m %))
            (mapcat #(get-key-path m %)))]
    (->> (into [] p pp)
         (concat pp))))

