(ns tiesql.compiler.schema
  (:require [schema.core :as s]
            [schema.macros :as sm]
            [schema.utils :as su]
            [tiesql.common :refer :all]))


(defn assoc-schema-key
  [m k v]
  (assoc m k v))

(defn assoc-extend-schema-key
  [m k v]
  (assoc-in m [(s/optional-key extend-meta-key) s/Keyword k] v))


(defn comp-k-schema
  [default-s-m p-context]
  (->> p-context
       (map second)
       (filter (fn [n]
                 (contains? n k-validate)))
       (reduce (fn [acc proc]
                 ((k-validate proc) acc)
                 ) default-s-m)))


(defn get-error-key
  [m]
  (loop [a m
         r []]
    (let [k (first (keys a))]
      (if-not (map? (k a))
        (conj r k)
        (recur (k a) (conj r k))))))





(defonce CommitSchema (s/enum commit-all-key commit-any-key commit-none-key))
(defonce SkipSchema #{(s/enum validation-key column-key join-key)})


(defn default-schema
  []
  (let [m {(s/optional-key extend-meta-key) {s/Keyword {}}}]
    (-> m
        (assoc-schema-key (s/optional-key doc-key) s/Str)
        (assoc-extend-schema-key (s/optional-key doc-key) s/Str)
        (assoc-schema-key (s/optional-key timeout-key) s/Int)
        (assoc-extend-schema-key (s/optional-key timeout-key) s/Int))))



(defn tie-schema
  [pc]
  (-> (default-schema)
      (comp-k-schema pc)
      (assoc (s/optional-key group-key) s/Keyword)
      (assoc-schema-key (s/optional-key skip-key) SkipSchema)
      (assoc-extend-schema-key (s/optional-key skip-key) SkipSchema)
      (assoc-schema-key (s/optional-key commit-key) CommitSchema)))


(defn check-tx-proc?
  [v]
  (let [tt [{:isolation #{:none :read-committed :read-uncommitted :repeatable-read :serializable}}
            {:read-only? #{true false}}]
        v (partition 2 v)
        is-contains? (fn [k kv]
                       (->> (map #(contains? (k %) kv) tt)
                            (some true?)))]
    (reduce (fn [acc [k kv]]
              (if-not (is-contains? k kv)
                (reduced false)
                acc)
              ) true v)))


(defn tie-config-schema
  [pc]
  (-> (default-schema)
      (comp-k-schema (select-keys pc [column-key join-key validation-key]))
      (assoc (s/required-key name-key) s/Keyword)
      (assoc (s/optional-key tx-prop) (s/pred check-tx-proc? 'check-tx-proc))
      (assoc (s/optional-key file-name-key) s/Str)
      (assoc (s/optional-key file-reload-key) boolean)
      (assoc (s/optional-key reserve-name-key) #{s/Keyword})
      (assoc (s/optional-key ds-key) s/Any)))



(defn apply-schema
  [schema-m v]
  (try
    (s/validate schema-m v)
    (catch clojure.lang.ExceptionInfo e
      ;      (println (.toString e))
      (let [err (-> e ex-data :error)
            err-k (get-error-key err)
            r (get-in (-> e ex-data :value) err-k)
            r (update-in err err-k (fn [_] r))]
        (throw (ex-info "Schema Failed " {:path  r
                                          :error (get-in err err-k)}))))))



(defn validate-schema!
  [process-context v]
  (-> (tie-schema process-context)
      (apply-schema v)))


(defn validate-config-schema!
  [process-context v]
  (-> (tie-config-schema process-context)
      (apply-schema v)))

