(ns tiesql.impl.common-impl
  (:require [tiesql.common :refer :all]
            [clj-common :as cc]
            [tiesql.compiler.schema :as sch]
            [schema.core :as s]))


(defn resolve-model?
  [v]
  (if (keyword? v)
    true
    (every? keyword? v)))


(defn new-name-processor
  []
  {k-validate (fn [acc]
                (sch/assoc-schema-key acc (s/required-key name-key) (s/pred resolve-model? 'resolve-model?)))
   k-emission (fn [n & r] (cc/as-sequential n))
   })


(defn model-compile
  [model-name & r]
  (cond
    (sequential? model-name)
    model-name
    (or
      (keyword? model-name)
      (string? model-name))
    (repeat model-name)
    :else
    (repeat nil)))


(defn model-schema
  [acc]
  (-> acc
      (sch/assoc-schema-key (s/optional-key model-key) (s/pred resolve-model? 'resolve-model?))
      (sch/assoc-extend-schema-key (s/optional-key model-key) s/Keyword)))


(defn new-model-processor
  []
  {k-emission model-compile
   k-validate model-schema
   k-name     model-key})



(defn result-process
  [tm]
  (if-not (or (= (dml-type-key tm) dml-type-select-key)
              (= (dml-type-key tm) dml-type-call-key))
    tm
    (let [result (result-key tm)
          output (output-key tm)]
      (cond
        (nil? result)
        tm
        (cc/failed? output)
        tm
        (and (empty? output)
             (contains? result result-single-key))
        (assoc tm output-key {})
        (and (contains? result result-array-key)
             (contains? result result-single-key))
        (assoc tm output-key [(first output) (second output)])
        (and (contains? result result-single-key))
        (assoc tm output-key (first output))
        :else tm))))


(defn result-schema
  [kname acc]
  (let [v #{(s/enum result-array-key result-single-key)}]
    (-> acc
        (sch/assoc-schema-key (s/optional-key kname) v)
        (sch/assoc-extend-schema-key (s/optional-key kname) v))))


(defn new-result-processor
  ([type order] (new-result-processor result-key type order))
  ([kname type order]
   {k-validate     (partial result-schema kname)
    k-name         kname
    k-order        order
    k-process-type type
    k-process      (map result-process)}))


(defn column-process
  [tm]
  (if-not (or (= (dml-type-key tm) dml-type-call-key)
              (= (dml-type-key tm) dml-type-select-key))
    tm
    (let [column (column-key tm)
          output (output-key tm)]
      (cond
        (or (nil? column)
            (cc/failed? tm)
            (cc/failed? output)
            (not (map? (first output))))
        tm
        :else
        (->> (repeat column)
             (map clojure.set/rename-keys output)
             (into [])
             (assoc tm output-key))))))

(defn column-schema
  [kname acc]
  (let [w {s/Keyword s/Keyword}]
    (-> acc
        (sch/assoc-schema-key (s/optional-key kname) w)
        (sch/assoc-extend-schema-key (s/optional-key kname) w))))


(defn new-column-processor
  ([type order] (new-column-processor column-key type order))
  ([kname type order]
   {k-validate     (partial column-schema kname)
    k-name         kname
    k-order        order
    k-process-type type
    k-process      (map column-process)}))




