(ns self-kafka.producer
  (:require [self-kafka.core :refer [as-properties]]
            [manifold.stream :as s])
  (:import [kafka.javaapi.producer Producer]
           [kafka.producer ProducerConfig KeyedMessage]
           [java.util List]))

(defn producer
  "Creates a Producer. m is the configuration
   metadata.broker.list : \"server:port,server:port\""
  [m]
  (Producer. (ProducerConfig. (as-properties m))))

(defn message
  ([topic value] (message topic nil value))
  ([topic key value] (KeyedMessage. topic key value)))

(defn send-message
  [^Producer producer ^KeyedMessage message]
  (.send producer message))

(defn send-messages
  [^Producer producer ^List messages]
  (.send producer messages))

(defn- m->msg
  [topic {:keys [key val] :as msg}]
  (if (and key val)
    (message topic key val)
    (message topic msg)))

(defn producer-stream
  "takes a kafka producer config map and a topic and returns a manifold stream representing the kafka topic
  producer-config properties are those of kafka.producer.ProducerConfig
  Put a message on the stream to put it on the Kafka topic
  If a message has both :key and :val then the message will be put onto kafka using that key and val,
  otherwise the whole message will be used.
  Closing the stream will close the Kafka Producer.
  See tests for more examples."
  [producer-config topic]
  (let [kafka-producer (producer producer-config)
        producer-stream (s/stream)]
    (s/on-closed producer-stream #(.close kafka-producer))
    (s/consume #(send-message kafka-producer (m->msg topic %)) producer-stream)
    producer-stream))
