(ns tiesql.client
  (:import [goog.net EventType ErrorCode XhrIo]
           [goog.net Jsonp])
  (:require [cljs-http.client :as http]
            [cljs.core.async :as async :refer [chan close!]]
            [devcards.util.edn-renderer :as ednr])
  (:require-macros [cljs.core.async.macros :refer [go alt!]]))


(defn log [& s]
  (.log js/console (apply str s)))



(defn validate-batch-request?
  [{:keys [name params options]
    :as   r}]
  (cond
    (not (sequential? name))
    [nil "Name will be sequential"]
    (not (map? params))
    [nil "Param will be map "]
    (not (map? options))
    [nil "Option will be map "]
    :else
    [r nil]))


(defn validate-one-request?
  [{:keys [name params options]
    :as   r}]
  (cond
    (not (keyword? name))
    [nil "Name will be keyword"]
    (not (map? params))
    [nil "Param will be map "]
    (not (map? options))
    [nil "Option will be map "]
    :else
    [r nil]))



(defn call-http-service
  [req url-str callback]
  (let [[v e] req]
    (if e
      (callback e)
      (do
        (go
          (let [m-req {:with-credentials? false
                       :accept            "application/transit+json"
                       ;:accept            "application/edn"
                       :transit-params    (select-keys v [:name :params :options])}
                res (->> m-req
                         (http/post url-str)
                         (async/<!))]
            (if (and (:success res)
                     (= (:status res) 200))
              (callback (:body res))
              (callback [nil res]))))
        nil)))
  nil)


(defn pull-one-request
  [{:keys [url callback]
    :as   r}]
  (-> r
      (update-in [:params] (fnil identity {}))
      (update-in [:options]
                 (fnil identity {:input  :keyword
                                 :output :keyword}))
      (validate-one-request?)
      (call-http-service (str url "/pullbatch") callback)))


(defn pull-batch-request
  [{:keys [url callback]
    :as   r}]
  (-> r
      (update-in [:params] (fnil identity {}))
      (update-in [:options]
                 (fnil identity {:input  :keyword
                                 :output :keyword}))
      (validate-batch-request?)
      (call-http-service (str url "/pullbatch") callback)))

;;callback

(defn push-one-request
  [{:keys [url callback]
    :as   r}]
  (-> r
      (update-in [:params] (fnil identity {}))
      (update-in [:options]
                 (fnil identity {:input  :keyword
                                 :output :keyword}))
      (validate-one-request?)
      (call-http-service (str url "/pushone") callback)))


(defn push-batch-request
  [{:keys [url callback]
    :as   r}]
  (-> r
      (update-in [:params] (fnil identity {}))
      (update-in [:options]
                 (fnil identity {:input  :keyword
                                 :output :keyword}))
      (validate-batch-request?)
      (call-http-service (str url "/pushbatch") callback)))


(defn pull-one!
  [& {:as r}]
  (pull-one-request r))


(defn pull-batch!
  [& {:as r}]
  (pull-batch-request r))


(defn push-one!
  [& {:as r}]
  (push-one-request r))


(defn push-batch!
  [& {:as r}]
  (push-batch-request r))



(defn make-request-for-html
  [url-str name params callback]
  (let [namev (keyword (js->clj name))
        params (js->clj params)
        cb (fn [res] (callback (ednr/html-edn res)))]
    {:url      url-str
     :name     namev
     :params   params
     :options  {:input  :string
                :output :string}
     :callback cb}))


(defn make-request-for-js
  [url-str name params callback]
  (let [namev (keyword (js->clj name))
        params (js->clj params)
        cb (fn [res] (do (print res)
                         (callback (clj->js res))))]
    {:url      url-str
     :name     namev
     :params   params
     :options  {:input  :string
                :output :string}
     :callback cb}))


(defn ^:export pullBatch
  [url-str name params callback]
  (-> (make-request-for-js url-str name params callback)
      (pull-batch-request)))


(defn ^:export pullOne
  [url-str name params callback]
  (->> (make-request-for-js url-str name params callback)
       (pull-one-request)))


(defn ^:export pushBatch
  [url-str name params callback]
  (-> (make-request-for-js url-str name params callback)
      (push-batch-request)))


(defn ^:export pushOne
  [url-str name params callback]
  (-> (make-request-for-js url-str name params callback)
      (push-one-request)))



(defn ^:export pullBatchAsHtml
  [url-str name params callback]
  (-> (make-request-for-html url-str name params callback)
      (pull-batch-request)))


(defn ^:export pullOneAsHtml
  [url-str name params callback]
  (-> (make-request-for-html url-str name params callback)
      (pull-one-request)))

