;; Expose as API
;;
(ns tiesql.core
  (:require
    [clj-common :as cc]
    [tiesql.common :refer :all]
    [tiesql.core-tracking :as e]
    [tiesql.core-util :as cu]
    [tiesql.impl.join-impl :as j]
    [tiesql.impl.param-impl :as p]
    [tiesql.impl.common-impl :as ci]
    [tiesql.impl.sql-bind-impl :as sb]
    [tiesql.impl.validation-impl :as ki]))


(defn dissoc-k-schema
  [process-context]
  (->> (cc/get-key-path-with-child process-context childs)
       (reduce (fn [acc v]
                 (->> [k-validate k-emission]
                      (reduce #(update-in %1 v dissoc %2) acc))
                 ) process-context)))


(defn new-process-context-impl
  []
  (-> {}
      (assoc params-key (p/new-param-key (p/new-param-child-key) 1))
      (assoc validation-key (ki/new-validation-key (ki/validation-key-child) 3))
      (assoc dml-type-key (sb/new-dml-type (sb/child-dml-bind-type) 5))
      (assoc name-key (ci/new-name-processor))
      (assoc sql-key (sb/new-sql-processor))
      (assoc column-key (ci/new-column-processor output-key 1))
      (assoc result-key (ci/new-result-processor output-key 3))
      (assoc model-key (ci/new-model-processor))
      (assoc join-key (j/new-join-processor))))


(defn comp-child-key
  ([p-context] (comp-child-key p-context false))
  ([p-context atype]
   (-> p-context
       (sb/comp-child)
       (ki/comp-validation-child atype)
       (p/comp-param-child atype))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;; Selecting impl ;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn select-config
  [tms]
  (global-key tms))


(defn select-name
  "Return list module "
  [tms name-coll]
  (let [name-key-coll (cc/as-sequential name-coll)
        tm-map (select-keys tms name-key-coll)]
    (cond
      (or (nil? tms)
          (empty? tms))
      (cc/fail " Source is empty ")
      (or (nil? tm-map)
          (empty? tm-map))
      (cc/fail (format " %s Name not found" (str name-key-coll)))
      (cu/is-reserve? tms name-key-coll)
      tm-map
      :else
      (cc/try-> tm-map
                (cu/validate-name! name-key-coll)
                (cc/select-values name-key-coll)
                (cu/validate-model!)
                (cu/filter-join-key)))))


(defn select-name-for-groups
  [tms gname name-coll]
  (let [name-set (into #{} name-coll)
        p (if name-coll
            (comp (filter #(= (group-key %) gname))
                  (filter #(contains? name-set (name-key %))))
            (comp (filter #(= (group-key %) gname))))
        t (into [] p (vals tms))
        w (sort-by index t)]
    (into [] (map name-key) w)))


;;;;;;;;;;;;;;;;;;;;;;;;;;; Processing impl  ;;;;;;;;;;;;;;;;;;;;;;;;
(defn select-processor
  [process-type p-coll]
  (->> p-coll
       (filter #(= process-type (k-process-type %)))
       (sort-by (fn [w] (k-order w)))
       (remove nil?)
       (mapv k-process)))



(defn xf-bind-input-step
  [input in-type m]
  (if (= in-type model-type)
    (assoc m input-key ((model-key m) input))
    (assoc m input-key input)))


(defn do-input-bind
  [tm-coll step-m in-type input]
  (let [input (cc/try-> input
                        (p/do-params in-type tm-coll step-m)
                        (j/do-join-impl in-type tm-coll step-m))]
    (if (cc/failed? input)
      input
      (let [t-context (vals (dissoc step-m params-key join-key :sql-executor :generator))
            xf-input-step (map (fn [m] (xf-bind-input-step input in-type m)))
            steps (->> (select-processor input-key t-context)
                       (into [xf-input-step])
                       (apply cc/comp-xf-until))]
        (transduce steps conj tm-coll)))))


(defn into-model-map
  [v]
  (if (cc/failed? v)
    (hash-map (model-key v) v)
    (hash-map (model-key v)
              (output-key v))))


(defn do-output-bind
  [coll t-context]
  (let [t-context (vals (dissoc t-context params-key join-key :sql-executor :generator))
        p (->> (map into-model-map)
               (conj (select-processor output-key t-context))
               (apply comp))]
    (->> (transduce p conj coll)
         (into {}))))


(defn do-execute
  [coll step-m]
  (let [exec (:sql-executor step-m)]
    (if (nil? exec)
      (cc/fail (pr-str "Executor is not set " coll))
      (exec coll))))


(defn run-process
  ([tm-coll step-m input]
   (run-process tm-coll step-m input map-type))
  ([tm-coll step-m input in-type]
   (cc/try-> tm-coll
             (do-input-bind step-m in-type input)
             (do-execute step-m)
             (e/warp-async-tracking (vals @e/tracking-fns) e/tracking-keys)
             (do-output-bind step-m))))


;(def run-map-process   (partial run-process params-input-map))
(def run-model-process
  (fn [tm-coll step-m input]
    (run-process tm-coll step-m input model-type)))

;;;;;;;;;;;;;;;;;;;;;;



(defn fetch-join-data
  [tm-coll step-m input-m]
  (let [[root & more-tm] tm-coll
        model-name (model-key root)
        root-result (run-process [root] step-m input-m)]
    (if (or (cc/failed? root-result)
            (nil? (get-in root-result [model-name]))
            (empty? (get-in root-result [model-name])))
      root-result
      (let [more-input (-> root-result
                           (j/get-source-relational-key-value (join-key root))
                           (merge input-m))
            more-result (run-process more-tm step-m more-input)
            do-join (get-in step-m [join-key k-join])]
        (->> (into root-result more-result)
             (do-join (join-key root)))))))



(defn fetch-data
  [step-m tms name-coll input-m]
  (let [tm-coll (select-name tms name-coll)]
    (if (cc/failed? tm-coll)
      tm-coll
      (if (not-empty (join-key (first tm-coll)))
        (fetch-join-data tm-coll step-m input-m)
        (run-process tm-coll step-m input-m)))))



(defn assoc-executor
  [step-m executor]
  (let [executor (or (get step-m :sql-executor)
                     executor)]
    (assoc step-m :sql-executor executor)))






