(ns tiesql.impl.param-impl
  (:require [schema.core :as s]
            [schema.macros :as sm]
            [schema.utils :as su]
            [clj-common :as cc]
            [tiesql.common :refer :all]
            [tiesql.impl.util :as pu]
            [tiesql.compiler.schema :as sch]
            ))


(defn assoc-param-path
  [data root-path param-coll]
  (for [p param-coll
        mp (cc/get-path data root-path (first p))
        :when (not (get-in data mp))]
    (assoc p 0 mp)))



(defmulti param-paths (fn [type _ _] type))


(defmethod param-paths
  model-type
  [_ [root-m & child-m] data]
  (let [model-name (get root-m model-key)
        rp (cc/get-path data model-name)
        rpp (assoc-param-path data rp (params-key root-m))
        cpp (for [c child-m
                  :let [crp (cc/get-path data rp (model-key c))]
                  p (assoc-param-path data crp (params-key c))]
              p)]
    (-> []
        (into rpp)
        (into cpp))))


(defmethod param-paths
  :default
  [_ tm-coll data]
  (->> (map params-key tm-coll)
       (reduce concat)
       (cc/distinct-with-range 1)
       (assoc-param-path data (cc/empty-path))))



(defn do-params
  [rinput p-type tm-coll p-context]
  (if-let [fn-params (get-in p-context [params-key k-process])]
    (->> (param-paths p-type tm-coll rinput)
         (fn-params rinput))
    rinput))



;;;;;;;;;;;;;;;;; Implementation



(defn new-param-ref-con
  [name index]
  (let [sv [(s/one s/Keyword "Source Data Model")
            (s/one s/Keyword "Type of Param ")
            (s/one s/Any "Any value")]]
    {k-name     name
     k-order    index
     k-validate (fn [w] (s/validate sv w))
     k-emission (fn [w] (update-in w [0] cc/as-lower-case-keyword))
     k-process  (fn [[_ _ v] _] v)
     }))

;(name :hello)

(defn new-param-ref-key
  [name index]
  (let [sv [(s/one s/Keyword "Source Data Model")
            (s/one s/Keyword "Type of Params ")
            (s/one s/Keyword "Refer keyword")]]
    {k-name     name
     k-order    index
     k-validate (fn [w] (s/validate sv w))
     k-emission (fn [w]
                  (-> w
                      (update-in [0] cc/as-lower-case-keyword)
                      (update-in [2] cc/as-lower-case-keyword)))
     k-process  (fn [[s _ k] m]
                  (->> (cc/replace-last-in-vector s k)
                       (get-in m)))
     }))


(def resolve-clj? (s/pred resolve 'resolve-clj))


(defn new-param-ref-fn
  [name index]
  (let [sv [(s/one s/Keyword "Source Data Model")
            (s/one s/Keyword "Type of params ")
            (s/one resolve-clj? "Any value")
            (s/one s/Keyword "Refer Keyword ")]]
    {k-name     name
     k-order    index
     k-validate (fn [w] (s/validate sv w))
     k-emission (fn [w]
                  (-> w
                      (assoc 2 (resolve (nth w 2)))
                      (update-in [0] cc/as-lower-case-keyword)
                      (update-in [3] cc/as-lower-case-keyword)))
     k-process  (fn [[s _ f k] m]
                  (->> (cc/replace-last-in-vector s k)
                       (get-in m)
                       (f)))
     }))




(defn new-param-ref-gen
  ([name-key index] (new-param-ref-gen name-key index nil))
  ([name order generator]
   (let [sv [(s/one s/Keyword "Source Data Model")
             (s/one s/Keyword "Type of params ")
             (s/one s/Keyword "Refer type ")]]
     {k-name     name
      k-order    order
      k-validate (fn [w] (s/validate sv w))
      k-emission (fn [w] (update-in w [0] cc/as-lower-case-keyword))
      k-process  (fn [[_ _ n] m]
                   (generator n {})
                   #_(let [r (generator n {})]
                       (if (cc/failed? r)
                         r
                         (get r n))))
      })))


(defn identiy-gen [v]
  (fn [k m]
    v))


(defn get-name
  [[_ t]]
  t)


(defn process-batch
  [pm input ks-coll]
  (->> ks-coll
       (sort-by (fn [[_ n]]
                  (k-order (n pm))))
       (reduce (fn [acc-input ks]
                 (let [[src n] ks
                       p (get-in pm [n k-process])
                       rv (cc/try! p ks acc-input)]
                   (if (cc/failed? rv)
                     (reduced rv)
                     (assoc-in acc-input src rv)))
                 ) input)))


(defn emission-batch
  [child-proc-m param-seq sql-key-seq]
  (->> param-seq
       (cc/distinct-with-range 2)
       (filter (fn [[f]] (contains? (into #{} sql-key-seq) f)))
       (into [])
       (pu/proc-compile get-name child-proc-m)))


(defn param-schema-value
  [child-proc-m]
  (s/pred (partial pu/proc-schema-validate get-name child-proc-m) 'k-schema-validate))


(defn assoc-param-schema
  [kname v acc]
  (-> acc
      (sch/assoc-schema-key (s/optional-key kname) v)
      (sch/assoc-extend-schema-key (s/optional-key kname) v)))


(defn comp-param-child
  [p-context atype]
  (if-not (params-key p-context)
    p-context
    (let [child-proc-m (get-in p-context [params-key childs])]
      (if atype
        (-> p-context
            (update-in [params-key] dissoc childs)
            (assoc-in [params-key k-process] (partial process-batch child-proc-m)))
        (let [kname (get-in p-context [params-key k-name])
              v (param-schema-value child-proc-m)]
          (-> p-context
              (update-in [params-key] dissoc childs)
              (assoc-in [params-key k-validate] (partial assoc-param-schema kname v))
              (assoc-in [params-key k-emission] (partial emission-batch child-proc-m))))))))



(defn new-param-child-key
  []
  {params-ref-con-key (new-param-ref-con params-ref-con-key 0)
   params-ref-key     (new-param-ref-key params-ref-key 1)
   params-ref-fn-key  (new-param-ref-fn params-ref-fn-key 2)
   params-ref-gen-key (new-param-ref-gen params-ref-gen-key 3)})


(defn new-param-key
  [c order]
  {k-order        order
   k-name         params-key
   k-process-type input-key
   childs         c})


(defn assoc-params-ref-gen-key
  [p-context generator]
  (->> (new-param-ref-gen params-ref-gen-key 3 generator)
       (assoc-in p-context [params-key childs params-ref-gen-key])))

