(ns tiesql.tools.http-service
  (:require [clojure.java.io :as io]
            [clojure.tools.reader.edn :as edn]
            [clojure.tools.logging :as log]
            [clj-common :as cc]
            [tiesql.common :refer :all]
            [tiesql.jdbc :as tj]
            [tiesql.tools.ds :as ds]))


(defn default-config-map [tie-app-file]
  (-> tie-app-file
      (io/resource)
      (slurp)
      (edn/read-string)))


(defn http-result
  [m]
  (if (cc/failed? m)
    [nil (into {} m)]
    [m nil]))


(defn seqable?
  "Returns true if (seq x) will succeed, false otherwise."
  [x]
  (or (seq? x)
      (instance? clojure.lang.Seqable x)
      (nil? x)
      (instance? Iterable x)
      (-> x .getClass .isArray)
      (string? x)
      (instance? java.util.Map x)))


(defn http-error-result
  [& [msg]]
  (->> (cc/fail (or msg "Input format is wrong, should be name=name-str&param=param-str"))
       (http-result)))


(defn read-params-string
  [params]
  (->> params
       (reduce (fn [acc [k v]]
                 (let [v1 (edn/read-string v)]
                   (if (symbol? v1)
                     (assoc acc k v)
                     (assoc acc k v1)))
                 ) {})))



(defn warp-param-name-with-group
  [tie gname {:keys [name] :as params}]

  #_(->> (r/select-name-for-groups tie (cjc/as-keyword gname) name)

         (assoc params :name)))

(defn read-file
  ([file-name ds] (read-file file-name ds))
  ([file-name ds init-name]
   (let [v (tj/read-file file-name)
         ds (ds/connection ds)]
     (when init-name
       (tj/db-do v ds init-name))
     (tj/validate-dml! ds (tj/get-dml v))
     (atom v))))


(defn reload-file
  ([tie ds]
   (when (get-in @tie [global-key file-reload-key])
     (let [f-name (get-in @tie [global-key file-name-key])
           new-tms (tj/read-file f-name)]
       (log/info "file is reloading ---" f-name)
       (tj/validate-dml! ds (tj/get-dml new-tms))
       (reset! tie new-tms)))))



(defn read-handler
  [tie ds {:keys [name params]}]
  (->> (tj/read-batch! @tie ds :name name :params params)
       (http-result)))


(defn write-handler
  [tie ds {:keys [name params]}]
  (->> (tj/write-batch! @tie ds :name name :params params)
       (http-result)))


(defn warp-reload-file
  [handler]
  (fn [tie ds r-params]
    (do
      (reload-file tie ds)
      (handler tie ds r-params))))


(defn warp-datasource-component
  [handler]
  (fn [tie data-sources r-params]
    (let [ds (ds/connection data-sources)]
      (handler tie ds r-params))))


(defn warp-name-validation
  [handler]
  (fn [TieComponent data-sources {:keys [name] :as r-params}]
    (if (nil? name)
      (http-error-result "Need name params as an example \"/query?name=?\"")
      (->> name
           (cc/as-sequential)
           (mapv cc/as-keyword)
           (assoc r-params :name)
           (handler TieComponent data-sources)))))


(defn warp-split-and-read-params
  [handler]
  (fn [TieComponent i-data-sources r-params]
    (let [r (read-params-string r-params)
          r (hash-map :name (get r :name)
                      :params (dissoc r :name))]
      (handler TieComponent i-data-sources r))))


(defn warp-exception
  [handler]
  (fn [TieCompnent i-data-sources r-params]
    (try
      (handler TieCompnent i-data-sources r-params)
      (catch Exception e
        (log/error e "Query handler error ")
        (http-error-result (str (.getMessage e) " for " r-params))))))


(def query-with-param-read (-> read-handler
                               (warp-reload-file)
                               (warp-datasource-component)
                               (warp-name-validation)
                               (warp-split-and-read-params)
                               (warp-exception)))


(def read! (-> read-handler
               (warp-reload-file)
               (warp-datasource-component)
               (warp-name-validation)
               (warp-exception)))


(def write! (-> write-handler
                (warp-reload-file)
                (warp-datasource-component)
                (warp-name-validation)
                (warp-exception)))


(defn get-sql-file-value [tie]
  (->> @tie
       (vals)
       (mapv (fn [w] (select-keys w [name-key model-key sql-key])))))
