(ns currency.core
  (:require [currency.currencies.core :refer [monetary-unit->string]]
            [currency.currencies.cad]
            [currency.currencies.usd]
            [clojure.spec.alpha :as s]))

;;; Records

(defrecord MonetaryUnit [currency amount]
  Object
  (toString [this]
    (monetary-unit->string this)))

;;; API

(defn monetary-unit
  [currency amount]
  {:pre [(s/valid?
          (s/cat :currency ::currency
                 :amount ::amount)
          [currency amount])]
   :post [(s/valid? ::monetary-unit %)]}
  (->MonetaryUnit currency amount))

(defn add
  [& units]
  {:pre [(s/valid?
          (s/and
           (s/every ::monetary-unit)
           ::same-currency)
          units)]
   :post [(s/valid? ::monetary-unit %)]}
  (monetary-unit
   (:currency (first units))
   (apply + (map :amount units))))

(defn multiply
  [& units-or-constants]
  {:pre [(s/valid?
          (s/every
           ::unit-or-constant)
          units-or-constants)
         (s/valid?
          ::same-currency
          (filter :currency units-or-constants))]
   :post [(s/valid? ::monetary-unit %)]}
  (let [units (filter :currency units-or-constants)
        constants (remove :currency units-or-constants)]
    (monetary-unit
     (:currency (first units))
     (->> units
          (map :amount)
          (concat constants)
          (apply *)
          long))))

;;; Specs

(s/def ::currency #{:cad :usd})
(s/def ::amount integer?)

(s/def ::unit-or-constant
  #(or (number? %)
       (s/valid? ::monetary-unit %)))

(s/def :unq/monetary-unit
  (s/keys :req-un [::currency ::amount]))

(s/def ::same-currency #(->> % (map :currency) distinct count (= 1)))

(s/def ::monetary-unit
  (s/and
   (partial instance? MonetaryUnit)
   (partial s/valid? :unq/monetary-unit)))
