(ns currency.core
  (:require [currency.currencies.core :refer [monetary-unit->string]]
            [currency.currencies.cad]
            [currency.currencies.usd]
            [clojure.spec.alpha :as s]))

;;; Records

(defrecord MonetaryUnit [currency amount]
  Object
  (toString [this]
    (monetary-unit->string this)))

;;; API

(defn monetary-unit
  [currency amount]
  {:pre [(s/valid?
          (s/cat :currency ::currency
                 :amount ::amount)
          [currency amount])]
   :post [(s/valid? ::monetary-unit %)]}
  (->MonetaryUnit currency amount))

(defn add
  [& units]
  {:pre [(s/valid?
          (s/and
           (s/every ::monetary-unit)
           ::same-currency)
          (remove nil? units))]}
  (when-let [units (not-empty (remove nil? units))]
    (monetary-unit
     (:currency (first units))
     (apply + (map :amount units)))))

(defn multiply
  [& units-or-constants]
  {:pre [(s/valid?
          (s/every
           ::unit-or-constant)
          (remove nil? units-or-constants))
         (s/valid?
          ::same-currency
          (filter :currency units-or-constants))]}
  (let [units (filter :currency units-or-constants)
        constants (remove
                   #(or (nil? %) (:currency %))
                   units-or-constants)]
    (when-let [combined (->> units
                             (map :amount)
                             (concat constants)
                             not-empty)]
      (monetary-unit
       (:currency (first units))
       (->> combined (apply *) long)))))

;;; Specs

(s/def ::currency #{:cad :usd})
(s/def ::amount integer?)

(s/def ::unit-or-constant
  #(or (number? %)
       (s/valid? ::monetary-unit %)))

(s/def :unq/monetary-unit
  (s/keys :req-un [::currency ::amount]))

(s/def ::same-currency #(->> % (map :currency) distinct count (>= 1)))

(s/def ::monetary-unit :unq/monetary-unit)
