(ns hub.http.client
  "The emphasis on this http client is to provide fast responses, with basic
  caching enabled, and a politeness toggle to make sure we don't spam any one
  domain too much."
  (:refer-clojure :exclude [get])
  (:require [cemerick.url :as cemurl]

            [clojure.core.async :as async]
            [clojure.string :as st]

            [com.stuartsierra.component :as component :refer [Lifecycle]]

            [hub.cache :as hc]
            [hub.security :as hsec]

            [org.bovinegenius.exploding-fish :as uri]
            [org.httpkit.client :as http]

            [taoensso.timbre :as log]

            [tidy.core :as tidy]))

;;; Declarations

(declare resolve-url url->key get-cache)

(def ^:private default-max-body-size-bytes (* 10 1024 1024))

;;; Records

(defrecord Client [cache politeness-ms max-body-size-bytes]
  Lifecycle
  (start [component]
    (if (:started? component)
      component
      (let [_ (log/debug "starting http client")
            max-body-size-bytes (or max-body-size-bytes)
            pub-ch (when (and (number? politeness-ms)
                              (pos? politeness-ms)) (async/chan))
            c (assoc component
                     :worker-pub-ch pub-ch
                     :worker-pub (when pub-ch (tidy/interval-pub pub-ch first politeness-ms))
                     :started? true)]
        (log/debug "started http client")
        c)))
  (stop [component]
    (if-not (:started? component)
      component
      (do (log/debug "stopping http client")
          (when-let [wp (:worker-pub component)] (async/unsub-all wp))
          (when-let [wpc (:worker-pub-ch component)] (async/close! wpc))
          (log/debug "stopped http client")
          (dissoc component :started? :worker-pub :worker-pub-ch)))))

;;; Public

(defn client
  ([] (client nil 0))
  ([cache] (client cache 0))
  ([cache politeness-ms] (->Client cache politeness-ms default-max-body-size-bytes)))

(defn get
  ([this url] (hub.http.client/get this url nil))
  ([this url opts]
   (assert (:started? this))
   (if-let [tx-fn (:tx-fn this)]
     (tx-fn (dissoc this :tx-fn) url opts)
     (async/go
       (or (get-cache this url)
           (async/<! (resolve-url this url)))))))

(defn url->key
  [url]
  (when-let [url (try (let [url (cemurl/url (st/lower-case url))]
                        (str (st/replace (:host url) #"^www\." "")
                             (when-let [port (:port url)]
                               (when (pos? port)
                                 (str ":" port)))
                             (:path url)
                             (when-let [query (:query url)]
                               (str "?" (cemurl/map->query query)))))
                      (catch Exception e))]
    (hsec/sha1 url)))

;;; Private

(defn- get-cache
  [this url]
  (when-let [c (:cache this)]
    (if (:started? c)
      (hc/get c (url->key (str url)))
      (do (log/warn "Client has been given a cache that has not been started.") nil))))

(defn- fetch-and-cache!
  [this url response-ch]
  (http/get url {:insecure? true
                 :filter (http/max-body-filter (:max-body-size-bytes this))}
            (fn [response]
              (when-let [url-k (and (= (:status response) 200) (url->key url))]
                (when-let [cache (:cache this)]
                  (when (:started? cache)
                    (hc/put! cache url-k response))))
              (async/go
                (async/>! response-ch (assoc response :url url))
                (async/close! response-ch)))))

(defn resolve-url
  [this url]
  (let [ch (async/chan)
        p (:worker-pub this)
        host (uri/host url)
        id (str (gensym))]
    (if-let [sub-ch (when p
                      (let [sub-ch (async/chan)]
                        (async/sub p host sub-ch)
                        sub-ch))]
      (do (async/go
            (loop []
              (when-let [[_ id*] (async/<! sub-ch)]
                (if (= id* id)
                  (do
                    (async/unsub p host sub-ch)
                    (fetch-and-cache! this url ch))
                  (recur)))))
          (async/put! (:worker-pub-ch this) [host id]))
      (fetch-and-cache! this url ch))
    ch))

(defn- web-worker-log-string
  [n s]
  (str "[" n "] " (st/trim s)))
