(ns re-frame-auth.flows.token
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.flows.core :as flow]
            [re-frame-auth.stores.core :as store]
            [re-frame-auth.util.crypto :as crypto]
            [re-frame-auth.util.spec :refer [validate]]
            [utilis.fn :refer [fsafe]]
            [utilis.types.keyword :refer [->keyword]]
            [utilis.map :refer [compact map-keys map-vals]]
            [clojure.data.json :as json]
            [clojure.spec.alpha :as s]))

;;; Declarations

(declare authenticate* refresh*
         generate-access-token
         generate-id-token
         token-use-matches-key? throwable? ->boolean)

;;; Records

(defrecord AccessTokenFlow []
  proto/AuthenticationFlow
  (authenticate [_ request] (authenticate* request)))

(defrecord RefreshTokenFlow []
  proto/RefreshFlow
  (refresh [_ request] (refresh* request)))

;;; API

(defn access-token-flow
  []
  (flow/authenticator
   (AccessTokenFlow.)))

(defn refresh-token-flow
  []
  (flow/refresher
   (RefreshTokenFlow.)))

(defn tokens
  "Return the tokens found in either the :params or :cookie of 'request'"
  [request]
  (let [token-generators (:auth/token-generators request)]
    (when-let [token-keys (not-empty (keys token-generators))]
      (compact
       (select-keys
        (or (when-let [tokens (:tokens (:params request))]
              (if (string? tokens)
                (json/read-json tokens true)
                tokens))
            (when (-> request :params :cookie ->boolean)
              (->> request
                   :cookies
                   (map-keys ->keyword)
                   (map-vals :value))))
        token-keys)))))

(defn parse-tokens
  [request]
  (let [tokens (->> (:auth/tokens request)
                    (map (fn [[token-key token-value]]
                           (when-let [generator (get (:auth/token-generators request) token-key)]
                             (let [parsed-token (try (proto/parse generator token-value)
                                                     (catch Exception e e))]
                               [token-key
                                (cond

                                  (throwable? parsed-token) parsed-token

                                  (not (token-use-matches-key?
                                        token-key ((fsafe ->keyword) (:token-use parsed-token))))
                                  (ex-info "Token type does not match key." {token-key parsed-token})

                                  :else parsed-token)]))))
                    (filter seq)
                    (into {}))]

    (cond

      (not (seq tokens))
      (ex-info "No tokens found." {:tokens tokens})

      (pos? (count (filter throwable? (vals tokens))))
      (ex-info "At least one token could not be extracted" {:tokens tokens})

      (not (apply = (map :secret (vals tokens))))
      (ex-info "Token secrets do not match." {:tokens tokens})

      (not (apply = (map :sub (vals tokens))))
      (ex-info "Token subs to not match." {:tokens tokens})

      :else tokens)))

(defn generate-tokens
  [{:keys [user-id secret generators] :or {secret (crypto/generate-secret)}}]
  {:pre [(seq user-id) (seq secret) (seq generators)]}
  (let [refresh-claims {:token-use :refresh
                        :sub user-id
                        :secret secret}
        tokens (->> generators
                    (map (fn [[k generator]]
                           (condp = k
                             :id-token
                             [k (generate-id-token
                                 generator
                                 {:user-id user-id
                                  :secret secret})]

                             :access-token
                             [k (generate-access-token
                                 generator
                                 {:user-id user-id
                                  :secret secret})]

                             :refresh-token
                             [k (proto/generate generator refresh-claims)]

                             nil)))
                    (remove nil?)
                    (into {}))]
    (if (:refresh-token tokens)
      (with-meta tokens {:refresh-claims refresh-claims})
      tokens)))

;;; Private

(defn- ->boolean
  [x]
  (boolean
   (cond

     (string? x)
     (= x "true")

     (integer? x)
     (= x 1)

     :else x)))

(defn- token-use-matches-key?
  [token-key token-use]
  (or (and (= token-key :id-token)
           (= token-use :id))
      (and (= token-key :access-token)
           (= token-use :access))
      (and (= token-key :refresh-token)
           (= token-use :refresh))))

(defn- throwable?
  [x]
  (instance? Throwable x))

(defn- generate-access-token
  [generator {:keys [user-id secret]}]
  (proto/generate
   generator {:token-use :access
              :sub user-id
              :secret secret}))

(defn- generate-id-token
  [generator {:keys [user-id secret]}]
  (proto/generate
   generator {:sub user-id
              :token-use :id
              :secret secret}))

(defn- authenticate*
  [request]
  (let [tokens (parse-tokens request)]

    (if (or (throwable? tokens)
            (and (map? tokens)
                 (some throwable? (vals tokens))))
      (flow/not-authenticated
       {:type :flow/validation-error
        :cause :auth/bad-token}
       {:tokens tokens})
      (let [access-token (-> request :auth/tokens :access-token)]
        (cond

          (instance? Throwable access-token)
          (flow/not-authenticated
           {:type :flow/validation-error
            :cause :auth/bad-access-token}
           {:events [{:request request
                      :type :auth/bad-access-token
                      :error access-token}]})

          (seq (:auth/identity request))
          (flow/authenticated
           {:user (:auth/identity request)
            :events [{:request request
                      :type :auth/valid-access-token}]
            :tokens (:auth/tokens request)
            :new-tokens? false})

          (or (not (seq (:auth/identity request)))
              (not (:sub (:auth-identity request))))
          (flow/not-authenticated
           {:type :flow/validation-error
            :cause :auth/nil-identity})

          :else (flow/not-authenticated
                 {:type :flow/validation-error
                  :cause :auth/access-denied}))))))

(defn- refresh*
  [request]
  (let [{:keys [access-token refresh-token]} (:auth/token-generators request)
        access-token-generator access-token
        refresh-token-generator refresh-token
        {:keys [access-token refresh-token]} (tokens request)
        access-token (proto/parse access-token-generator access-token {:skip-validation true})
        refresh-token (proto/parse refresh-token-generator refresh-token)]

    (cond

      (and access-token refresh-token (not (throwable? refresh-token)))
      (if (and

           ;; The refresh token must have a user-id present
           (seq (:sub refresh-token))

           ;; Both must have a secret present
           (seq (:secret refresh-token))
           (seq (:secret access-token))

           ;; Both must share the same secret
           (= (:secret refresh-token)
              (:secret access-token))

           ;; Makes round trip to db (potentially). Checks that refresh
           ;; hasn't been revoked.
           (not (proto/refresh-token-revoked? (:auth/auth-store request) refresh-token))

           ;; Checks that the user referred to actually exists
           (boolean
            (proto/find-user
             (:auth/user-store request)
             (:sub refresh-token))))
        (flow/authenticated
         {:events [{:request request
                    :type :auth/token-refreshed}]
          :tokens (generate-tokens
                   {:user-id (:sub refresh-token)
                    :secret (:secret refresh-token)
                    :generators (:auth/token-generators request)})
          :new-tokens? true})
        (flow/not-authenticated
         {:type :flow/validation-error
          :cause :auth/bad-refresh-token}
         {:events
          [{:request request
            :type :auth/bad-refresh-token}]}))

      (throwable? (:auth/tokens request))
      (flow/not-authenticated
       {:type :flow/validation-error
        :cause :auth/bad-token-extraction}
       {:events [{:request request
                  :type :auth/bad-token-extraction
                  :error (:auth/tokens request)}]})

      :else
      (flow/not-authenticated
       {:type :flow/validation-error
        :cause :auth/refresh-unauthorized}
       {:events [{:request request
                  :type :auth/refresh-unauthorized}]}))))

;;; Specs

(s/def :auth-token-flow/access-token-flow
  (partial instance? AccessTokenFlow))

(s/def :auth-token-flow/refresh-token-flow
  (partial instance? RefreshTokenFlow))
