(ns re-frame-auth.flows.email
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.flows.core :as flow]
            [re-frame-auth.stores.core :as store]
            [re-frame-auth.util.email :refer [normalize]]
            [buddy.hashers :as hashers]
            [clojure.string :as st]
            [clojure.spec.alpha :as s]
            [utilis.fn :refer [fsafe]]))

;;; Declarations

(def ^:private min-password-length 6)

(declare authenticate*)

;;; Records

(defrecord EmailPasswordFlow []
  proto/AuthenticationFlow
  (authenticate [_ request] (authenticate* request)))

(defn email-password-flow
  []
  (flow/authenticator (EmailPasswordFlow.)))

(defn auth-record
  [email password]
  (store/auth-record
   :email {:original-email email
           :email (:normalized (normalize email))
           :password (hashers/derive password)}))

(defn check-password?
  [password-attempt hashed-password]
  (boolean (hashers/check password-attempt hashed-password)))

;;; Private

(defn- bad-password?
  [password]
  (or (not (string? password))
      (< (count password) min-password-length)))

(defn- authenticate*
  [{:keys [params] :as request}]
  (try (let [params (-> params
                        (update :email (fsafe normalize))
                        (update :password (fsafe st/trim)))]
         (cond

           (not (s/valid? ::email-flow params))
           (flow/not-authenticated
            {:type :flow/validation-error
             :cause (cond

                      (-> params :email :valid? not)
                      :email-flow/bad-email

                      (-> params :email :dangerous?)
                      :email-flow/dangerous-email

                      (bad-password? (:password params))
                      :email-flow/bad-password

                      :else :email-flow/validation-error)
             :message (s/explain-str ::email-flow params)})

           :else
           (let [normalized-email (-> params :email :normalized)
                 original-email (-> params :email :original)
                 auth-store (:auth/auth-store request)
                 auth-record (store/auth-record
                              :email {:email normalized-email
                                      :original-email original-email})
                 match (proto/find-auth auth-store auth-record)]
             (if-let [{:keys [password user-id]} match]
               (let [hashed-password password
                     password-attempt (st/trim (str (:password params)))]
                 (if (check-password? password-attempt hashed-password)
                   (flow/authenticated {:user {:id user-id}})
                   (flow/not-authenticated
                    {:type :flow/validation-error
                     :cause :email-flow/password-mismatch})))
               (flow/new-user
                (store/auth-record
                 :email (->> params :password
                             hashers/derive
                             (assoc auth-record :password))))))))
       (catch Exception e
         (flow/authentication-result
          {:error {:throwable e
                   :type :email-flow/error
                   :cause :email-flow/exception}}))))

(s/def :email-auth-flow/password
  (s/and string? (complement bad-password?)))
(s/def :email-auth-flow/email
  (s/and #(:valid? %)
         #(not (:dangerous? %))))
(s/def ::email-flow
  (s/keys :req-un [:email-auth-flow/email
                   :email-auth-flow/password]))
