(ns re-frame-auth.stores.rethinkdb
  (:require [re-frame-auth.stores.core :as store]
            [re-frame-auth.flows.core :as flow]
            [re-frame-auth.protocols :as proto]
            [re-frame-auth.util.id :refer [generate-auth-key]]

            [rethink.core :as rdb]

            [utilis.types.keyword :refer [->keyword]]
            [utilis.types.string :refer [->string]]
            [utilis.map :refer [compact map-keys map-vals]]

            [integrant.core :as ig]
            [taoensso.timbre :as log]
            [clj-time.coerce :as tc]
            [clj-time.core :as t]

            [clojure.string :as st]))

;;; Declarations

(declare find-user* create-user!* delete-user!*

         find-auth*
         create-auth!*
         delete-auth!*
         create-refresh-token!*
         revoke-refresh-token!*
         refresh-token-revoked?*)

;;; Records

(defrecord RethinkDBUserStore [connection]
  proto/UserStore
  (find-user [this user-id] (find-user* this user-id))
  (create-user! [this user] (create-user!* this user))
  (delete-user! [this user] (delete-user!* this user)))

(defrecord RethinkDBAuthStore [connection]
  proto/AuthStore
  (find-auth [this auth] (find-auth* this auth))
  (create-auth! [this auth] (create-auth!* this auth))
  (delete-auth! [this auth] (delete-auth!* this auth))
  (create-refresh-token! [this claims] (create-refresh-token!* this claims))
  (revoke-refresh-token! [this claims] (revoke-refresh-token!* this claims))
  (refresh-token-revoked? [this claims] (refresh-token-revoked?* this claims)))

;;; Integrant

(defmethod ig/init-key :user-store/rethinkdb
  [_ {:keys [connection]}]
  (try (rdb/init-table! :users :id [:created] connection)
       (catch Exception e
         (log/error e "Exception occurring running startup operation.")))
  (map->RethinkDBUserStore {:connection connection}))

(defmethod ig/init-key :auth-store/rethinkdb
  [_ {:keys [connection]}]
  (try (rdb/init-table! :auth :id [:user-id :created] connection)
       (rdb/init-table! :refresh-tokens :secret connection)
       (catch Exception e
         (log/error e "Exception occurring running startup operation.")))
  (map->RethinkDBAuthStore {:connection connection}))

;;; Private

(defn- find-user*
  [{:keys [connection]} user-id]
  (rdb/run (rdb/get (rdb/table :users) user-id) connection))

(defn- create-user!*
  [{:keys [connection]} user]
  (rdb/run (rdb/insert (rdb/table :users) user {:durability :hard}) connection))

(defn- delete-user!*
  [{:keys [connection]} user-id]
  (rdb/run (rdb/delete (rdb/table :users) user-id {:durability :hard}) connection))

(defn- find-auth*
  [{:keys [connection]} auth]
  (rdb/run (rdb/get (rdb/table :auth) (generate-auth-key auth)) connection))

(defn- delete-auth!*
  [{:keys [connection]} auth]
  (rdb/run
    (rdb/delete
     (rdb/get (rdb/table :auth) (generate-auth-key auth)))
    connection))

(defn- create-auth!*
  [{:keys [connection]} auth]
  (let [{:keys [primary-key type]} auth]
    (when (or (not primary-key)
              (not type)
              (not (get auth primary-key)))
      (throw
       (ex-info
        "Auth record must have a primary key, type, and have a value present for the primary key."
        {:auth auth})))
    (rdb/run
      (rdb/upsert
       (rdb/table :auth)
       (assoc auth
              :id (generate-auth-key auth)
              :created (tc/to-long (t/now)))
       :id
       {:durability :hard})
      connection)))

(defn- create-refresh-token!*
  [{:keys [connection]} claims]
  (-> (rdb/table :refresh-tokens)
      (rdb/insert claims {:durability :hard})
      (rdb/run connection)))

(defn- revoke-refresh-token!*
  [{:keys [connection]} claims]
  (when-let [secret (:secret claims)]
    (-> (rdb/table :refresh-tokens)
        (rdb/update
         {:revoked true
          :secret (:secret claims)})
        (rdb/run connection))))

(defn- refresh-token-revoked?*
  [{:keys [connection]} claims]
  (let [result (-> (rdb/table :refresh-tokens)
                   (rdb/get (:secret claims))
                   (rdb/run connection))]
    (or (nil? result)
        (-> result :revoked boolean))))
