(ns re-frame-auth.token-generators.jwe
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.util.crypto :as crypto]

            [buddy.sign.jwt :as jwt]
            [buddy.core.codecs :as buddy-codecs]

            [clj-time.core :as t]
            [clj-time.coerce :as tc]

            [utilis.map :refer [map-keys compact]]
            [utilis.fn :refer [fsafe]]
            [utilis.types.keyword :refer [->keyword]]))

;;; Declarations

(declare generate-secret parse*)

;;; Records

(defrecord JWE [claims-fn secret token-expiry-time-ms]
  proto/TokenGenerator
  (generate [this data]
    (jwt/encrypt
     (claims-fn
      (assoc data
             :exp (tc/from-long
                   (+ (tc/to-long (t/now))
                      token-expiry-time-ms))))
     secret))
  (parse [this token opts] (parse* this token opts))
  (parse [this token] (parse* this token nil))
  (expiry-time-ms [this] token-expiry-time-ms))

;;; API

(defn jwe
  [& {:keys [token-expiry-time-ms claims-fn secret]
      :or {token-expiry-time-ms (* 1000 60 15)
           claims-fn identity
           secret (crypto/generate-secret)}}]
  (let [secret (if (string? secret)
                 (buddy-codecs/hex->bytes secret)
                 secret)]
    (JWE. claims-fn secret (long token-expiry-time-ms))))

;;; Private

(defn- parse*
  [this token opts]
  (try
    (compact
     (-> (jwt/decrypt token (:secret this) opts)
         (update :token-use (fsafe ->keyword))
         (update :exp (fsafe (partial * 1000)))))
    (catch Exception e e)))
