(ns rethink.core
  "RethinkDB driver https://github.com/mcjp78/jrethinkdb"
  (:refer-clojure :exclude [filter remove get update not count sort-by reverse
                            or and nth fn map])
  (:require [utilis.fn :refer [fsafe]]
            [utilis.map :refer [map-keys]]
            [inflections.core :refer [hyphenate underscore camel-case]]
            [clojure.string :as st])
  (:import [java.util.concurrent LinkedBlockingQueue]
           [java.util Optional]
           [com.rethinkdb RethinkDB]
           [com.rethinkdb.net Connection Cursor]
           [com.rethinkdb.model Arguments OptArgs]
           [com.rethinkdb.ast Util Query]
           [com.rethinkdb.gen.ast
            Eq GetField Filter Table Insert GetAll Branch Get Update Without
            Delete Not SetIntersection IsEmpty Count OrderBy Desc Asc
            TableCreate IndexCreate IndexWait DbList TableList DbCreate
            IndexList DbCreate Or And Nth Changes DeleteAt Map]))

;;; Declarations

(defonce r (RethinkDB/r))

(declare rt-> ->rt ->rt-name prep-optargs ensure-seq arguments ->changefeed eq
         drain-and-shutdown-connection prep-keyword-val close! changes? run)

(deftype Changefeed [conn]
  java.io.Closeable
  (close [this]
    (when-let [{:keys [cursor ft]} (clojure.core/get @(:changefeeds conn) this)]
      (try (.close cursor)
           (catch com.rethinkdb.gen.exc.ReqlDriverError e
             (when-not (#{"java.lang.InterruptedException"} (-> e Throwable->map :via first :message))
               (throw e)))
           (finally (future-cancel ft))))))

;;; API

(defn run
  ([query conn] (run query conn nil))
  ([query conn opts]
   (when (= :rethinkdb/running @(:status conn))
     (swap! (:queries conn) conj query)
     (try
       (let [result (.run query (:conn conn))]
         (if (changes? query)
           (if-let [on-value (:on-value opts)]
             (->changefeed conn result on-value (:on-close opts))
             (do
               (close! result)
               (throw
                (ex-info
                 "Must provide an 'on-value' function in 'opts' to consume changefeed."
                 {:opts opts}))))
           (rt-> result)))
       (catch Exception e
         (throw e))
       (finally (swap! (:queries conn) disj query))))))

(defn connect
  [& [{:keys [hostname port db]
       :or {hostname "localhost"
            port 28015
            db "test"}}]]
  {:conn (-> r
             (.connection)
             (.hostname hostname)
             (.port port)
             (.db db)
             (.connect))
   :changefeeds (atom {})
   :queries (atom #{})
   :status (ref :rethinkdb/running)})

(defn quick-run
  [query & {:keys [hostname port db]
            :or {hostname "localhost"
                 port 28015
                 db "test"}}]
  (with-open [conn (connect :hostname hostname
                            :port port
                            :db db)]
    (run query conn)))

(defn changefeed?
  [x]
  (instance? Changefeed x))

(defn changes?
  [x]
  (instance? Changes x))

(defn close!
  [conn-or-changefeed]
  (if-let [conn (:conn conn-or-changefeed)]
    (drain-and-shutdown-connection conn-or-changefeed)
    (if (instance? java.io.Closeable conn-or-changefeed)
      (.close conn-or-changefeed)
      (throw (ex-info "Unable to close" {:conn-or-changefeed conn-or-changefeed})))))

;;; Query Building

(defn table
  ([table-name] (table table-name nil))
  ([table-name optargs]
   (Table.
    (arguments (->rt-name table-name))
    (prep-optargs optargs))))

(defn branch
  [expr expr-a expr-b & exprs]
  (Branch. (apply arguments expr expr-a expr-b exprs)))

(defn get
  [table id]
  (Get. (arguments table id)))

(defn get-all
  ([table ids] (get-all table ids nil))
  ([table ids optargs]
   (GetAll.
    (apply arguments
           table
           (->> ids ensure-seq
                (clojure.core/map prep-keyword-val)))
    (prep-optargs optargs))))

(defn not
  [expr]
  (Not. (arguments expr)))

(defn or
  [expr & exprs]
  (Or. (apply arguments expr exprs)))

(defn and
  [expr & exprs]
  (And. (apply arguments expr exprs)))

(defn nth
  [sel index]
  (Nth. (arguments sel index)))

(defn without
  [sel paths]
  (Without. (apply arguments sel paths)))

(defn update
  ([obj expr] (update obj expr nil))
  ([obj expr optargs]
   (Update.
    (arguments obj (->rt expr))
    (prep-optargs optargs))))

(defn insert
  ([table doc] (insert table doc nil))
  ([table doc optargs]
   (Insert.
    (arguments table (->rt doc))
    (prep-optargs optargs))))

(defn delete-at
  ([expr-a expr-b] (delete-at expr-a expr-b nil))
  ([expr-a expr-b optargs]
   (DeleteAt.
    (arguments expr-a expr-b)
    (prep-optargs optargs))))

(defn delete
  ([expr] (delete expr nil))
  ([expr optargs]
   (Delete.
    (arguments expr)
    (prep-optargs optargs))))

(defn upsert
  ([table doc] (upsert table doc :id))
  ([table doc id-field] (upsert table doc id-field nil))
  ([table doc id-field optargs]
   (let [rt-doc (->rt doc)
         rt-id-field (->rt-name id-field)
         got (get table (clojure.core/get doc id-field))]
     (branch
      (eq got nil)
      (insert table doc optargs)
      (update got (without rt-doc [rt-id-field]) optargs)))))

(defn changes
  ([sel] (changes sel nil))
  ([sel optargs]
   (Changes.
    (arguments sel)
    (prep-optargs optargs))))

(defn desc
  [sort-key]
  (Desc. (arguments (->rt-name sort-key))))

(defn asc
  [sort-key]
  (Asc. (arguments (->rt-name sort-key))))

(defn order-by
  ([sel sort-key] (order-by sel sort-key :asc))
  ([sel sort-key direction]
   (OrderBy.
    (arguments
     sel
     ({:asc (if (instance? Asc asc) asc (asc sort-key))
       :desc (if (instance? Desc desc) desc (desc sort-key))}
      direction)))))

(def sort-by order-by)

;;; Aggregate Functions

(defn count
  [sel]
  (Count. (arguments sel)))

;;; Predicate Building

(defn filter
  ([expr pred] (filter expr pred nil))
  ([expr pred optargs]
   (Filter.
    (arguments expr ^com.rethinkdb.gen.ast.ReqlFunction1 pred)
    (prep-optargs optargs))))

(defn map
  [expr f]
  (Map. (arguments expr ^com.rethinkdb.gen.ast.ReqlFunction1 f)))

(defn remove
  ([expr pred] (remove expr pred nil))
  ([expr pred optargs]
   (filter expr (not pred) optargs)))

(defn func
  [f]
  (reify com.rethinkdb.gen.ast.ReqlFunction1
    (apply [this row]
      (f row))))

(def pred func)

(defmacro fn
  [args & body]
  `(func (clojure.core/fn ~args ~@body)))

(defn get-field
  [o k]
  (GetField. (arguments o (->rt-name k))))

(defn eq
  [obj1 obj2 & objs]
  (Eq. (apply arguments obj1 (cons obj2 objs))))

(defn set-intersection
  [sel values]
  (SetIntersection. (arguments sel values)))

(defn is-empty
  [sel]
  (IsEmpty. (arguments sel)))

;;; Useful Predicates

(defn matches-one-of?
  "Matches on items in 'sel' that have a 'field-name' that matches any of the
  options in 'values'."
  [sel field-name values]
  (pred
   (clojure.core/fn
     [row]
     (-> row
         (get-field (->rt-name field-name))
         (set-intersection values)
         (is-empty)))))

(defn matches-all-of?
  "Matches on items in 'sel' that have a 'field-name' that matches all of the
  options in 'values'."
  [sel field-name values]
  (pred
   (clojure.core/fn
     [row]
     (-> row
         (get-field (->rt-name field-name))
         (set-intersection values)
         (count)
         (eq (count values))))))

;;; Table Manipulation

(defn table-create
  ([table] (table-create table nil))
  ([table optargs]
   (TableCreate.
    (arguments table)
    (prep-optargs optargs))))

(defn index-create
  ([table index-name] (index-create table index-name nil))
  ([table index-name index-fn]
   (IndexCreate.
    (apply arguments
           (concat
            [table (->rt-name index-name)]
            (when index-fn
              [index-fn]))))))

(defn db-create
  [db-name]
  (DbCreate. (arguments (->rt-name db-name))))

(defn index-wait
  [table index-name]
  (IndexWait.
   (arguments
    table
    (->rt-name index-name))))

(defn db-list
  []
  (DbList. (arguments)))

(defn table-list
  []
  (TableList. (arguments)))

(defn index-list
  [table]
  (IndexList. (arguments table)))

;;; Administration Syntactic Sugar

(defn create-table!
  ([table-name conn] (create-table! table-name conn nil))
  ([table-name conn opts]
   (-> (->rt-name table-name)
       (table-create opts)
       (run conn {:durability :hard}))))

(defn create-index!
  [table-name index-name conn]
  (-> table-name table
      (index-create
       index-name
       (func
        (clojure.core/fn
          [row]
          (get-field row (->rt-name index-name)))))
      (run conn)))

(defn create-db!
  [db-name conn]
  (run (db-create db-name) conn))

(defn wait-for-index!
  [table-name index-name conn]
  (-> table-name table
      (index-wait index-name)
      (run conn)))

(defn db-exists?
  [db-name conn]
  (-> (db-list)
      (run conn)
      (->> (some (partial = (->rt-name db-name))))
      boolean))

(defn table-exists?
  [table-name conn]
  (-> (table-list)
      (run conn)
      (->> (some (partial = (->rt-name table-name))))
      boolean))

(defn index-exists?
  [table-name index-name conn]
  (-> table-name table
      (index-list)
      (run conn)
      (->> (some (partial = (->rt-name index-name))))
      boolean))

(defn init-table!
  ([table-name conn] (init-table! table-name :id conn))
  ([table-name primary-key conn] (init-table! table-name primary-key nil conn))
  ([table-name primary-key indexes conn]
   (when-not (table-exists? table-name conn)
     (create-table! table-name conn {:primary-key (->rt-name primary-key)}))
   (doseq [index-name indexes]
     (when-not (= (->rt-name index-name) (->rt-name primary-key))
       (when-not (index-exists? table-name index-name conn)
         (create-index! table-name index-name conn)
         (wait-for-index! table-name index-name conn))))))



;;; Utilities

(defn any-map?
  [m]
  (clojure.core/or
   (instance? java.util.HashMap m)
   (map? m)))

(defn any-coll?
  [v]
  (clojure.core/or
   (instance? java.util.ArrayList v)
   (sequential? v)))

(defn keyword->string
  [kw]
  (if (keyword? kw)
    (str
     (when-let [ns (namespace kw)]
       (str ns "/"))
     (name kw))
    kw))

(defn ->rt-key
  [k]
  (underscore (name k)))

(defn ->rt-keyword
  [k]
  ["rdb/kw" (keyword->string k)])

(defn ->rt-val
  [v]
  (cond
    (keyword? v)
    (->rt-keyword v)
    :else v))

(defn rt-keyword->
  [k]
  (keyword (second k)))

(defn prep-keyword-val
  [k]
  (if (keyword? k) (->rt-keyword k) k))

(defn xform-map
  [m kf vf]
  (->> (into {} m)
       (clojure.core/map
        (clojure.core/fn
          [[k v]]
          [(kf k)
           (cond
             (any-map? v)
             (xform-map v kf vf)

             (clojure.core/and
              (any-coll? v)

              (clojure.core/not (keyword? (first v))))
             (mapv #(if (any-map? %) (xform-map % kf vf) %) v)

             :else (vf v))]))
       (into {})))

(defn ->rt
  [m]
  (cond

    (any-map? m)
    (->> m
         (clojure.core/map
          (clojure.core/fn [[k v]]
            [(->rt-key k)
             (->rt v)]))
         (clojure.core/into {}))

    (any-coll? m)
    (clojure.core/map ->rt m)

    :else (->rt-val m)))

(defn rt-key->
  [k]
  (keyword (hyphenate k)))

(defn rt-keyword-val?
  [v]
  (clojure.core/and
   (any-coll? v)
   (string? (first v))
   (re-find #"^rdb/.*$" (first v))))

(defn rt-val->
  [v]
  (if (rt-keyword-val? v)
    (rt-keyword-> v)
    v))

(defn rt->
  [m]
  (cond

    (any-map? m)
    (->> m
         (clojure.core/mapv
          (clojure.core/fn [[k v]]
            [(rt-key-> k)
             (rt-> v)]))
         (clojure.core/into {}))

    (clojure.core/or
     (instance? Cursor m)
     (clojure.core/and
      (any-coll? m)
      (clojure.core/not
       (rt-keyword-val? m))))
    (clojure.core/mapv rt-> m)

    :else (rt-val-> m)))

(defn ->rt-name
  [x]
  (cond

    (string? x)
    (underscore x)

    (keyword? x)
    (->rt-name (keyword->string x))

    :else (throw (ex-info "Unable to convert to name" {:x x}))))

(defn- lower-case-first
  [s]
  (apply str (st/lower-case (first s)) (rest s)))

(defn- prep-optarg-key
  [k]
  (->rt-name k))

(defn- prep-optargs
  [optargs]
  (OptArgs/fromMap
   (->> optargs
        (mapv (clojure.core/fn [[k v]]
                [(prep-optarg-key k)
                 (Util/toReqlAst
                  (cond
                    (= k :durability)
                    (keyword->string v)

                    (= k :index)
                    (->rt-name v)

                    :else v))]))
        (into {}))))

(defn- ensure-seq
  [x]
  (if (sequential? x) x [x]))

(defn- arguments
  ([] (Arguments.))
  ([this & args]
   (let [this (prep-keyword-val this)
         arguments (Arguments. this)]
     (when (seq args)
       (->> args
            (clojure.core/map prep-keyword-val)
            (.coerceAndAddAll arguments)))
     arguments)))

(defn- ->changefeed
  [conn cursor on-value on-close]
  {:pre [(instance? Cursor cursor)]}
  (let [changefeed (Changefeed. conn)
        maybe-print-error
        (clojure.core/fn [msg e & args]
          (when-not (#{"java.lang.InterruptedException"
                       "java.util.concurrent.CancellationException"
                       "java.util.concurrent.ExecutionException: com.rethinkdb.gen.exc.ReqlDriverError: Reached the end of the read stream."}
                     (-> e Throwable->map :via first :message))
            (apply println msg (pr-str (Throwable->map e)) args)))
        ft (future
             (try
               (loop [changes (.iterator cursor)]
                 (when-let [next (.next changes)]
                   (when-let [change (-> next
                                         rt->
                                         (clojure.core/update :type keyword)
                                         not-empty)]
                     (try (on-value {:change change :type :rethinkdb/change})
                          (catch Exception e
                            (maybe-print-error
                             "Exception occurred notifying change" e
                             {:change change
                              :type :rethinkdb/change})))
                     (recur changes))))
               (catch com.rethinkdb.gen.exc.ReqlDriverError e
                 (maybe-print-error "A query error occurred in changefeed:" e))
               (catch Exception e
                 (->> e
                      Throwable->map pr-str
                      (println "Error shutting down changefeed:"))
                 (println (.printStackTrace e)))
               (finally
                 (swap! (:changefeeds conn) dissoc changefeed)
                 (.close cursor)
                 (when (fn? on-close) (on-close)))))]
    (swap! (:changefeeds conn) assoc changefeed {:cursor cursor :ft ft})
    changefeed))

(defn- drain-and-shutdown-connection
  [conn]
  (do (dosync (ref-set (:status conn) :rethinkdb/draining))
      (doseq [[queue cursor] @(:changefeeds conn)]
        (try
          (.close cursor)
          (catch Exception e)))
      @(future
         (try (let [elapsed (atom 0)
                    force-close (atom false)
                    increment 100]
                (loop []
                  (when-not (true? @force-close)
                    (when (clojure.core/or
                           (seq @(:queries conn))
                           (seq @(:changefeeds conn)))
                      (Thread/sleep increment)
                      (let [elapsed (swap! elapsed (partial + increment))]
                        (cond
                          (clojure.core/and
                           (> elapsed 5000)
                           (< elapsed (+ 5000 (* increment 1.5))))
                          (println "More than 5 seconds elapsed shutting down rethinkdb connection.")
                          (> elapsed 10000)
                          (do (println "More than 10 seconds elapsed shutting down rethinkdb connection. Forcing close.")
                              (reset! force-close true))
                          :else nil))
                      (recur)))))
              (catch Exception e
                (println e "Exception occurred shutting down rethinkdb connection"))))
      (.close (:conn conn))))
